#include "yang_tasks_creator.h"

#include <drive/backend/chat_robots/abstract.h>

using TTagsOnlyCreator = TDocumentsVerificationYangTasksCreatorTagsOnly;
using TCreator = TDocumentsVerificationYangTasksCreator; // Tags + Assignments
using TCreatorState = TRTDocumentsVerificationTasksCreatorState;

TTagsOnlyCreator::TFactory::TRegistrator<TTagsOnlyCreator> TTagsOnlyCreator::Registrator("user_documents_yang_tasks_creator_tags_only");
TCreator::TFactory::TRegistrator<TCreator> TCreator::Registrator("user_documents_yang_tasks_creator");
TCreatorState::TFactory::TRegistrator<TCreatorState> TCreatorState::Registrator(TCreator::GetTypeName());

TString TCreatorState::GetType() const {
    return TCreator::GetTypeName();
}

// TDocumentsVerificationYangTasksCreatorTagsOnly

void TTagsOnlyCreator::ResendTagsToYang(const NDrive::IServer* server, const TDocumentsVerificationManager& manager) const {
    TVector<TDBTag> dbTags;
    auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
    if (server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({}, {ResendToYangTagName}, dbTags, session) && dbTags.size() > 0) {
        TSet<TString> usersForYangRecheck;
        for (auto&& tag : dbTags) {
            usersForYangRecheck.insert(tag.GetObjectId());
        }
        manager.ResendToYang(usersForYangRecheck, ResendToYangTagName, GetRobotUserId(), server);
    }
}


TExpectedState TTagsOnlyCreator::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /* stateExt */, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();

    if (!server || !server->GetDriveAPI() || !server->GetDriveAPI()->HasDocumentPhotosManager()) {
        return MakeUnexpected<TString>("user photo manager undefined");
    }
    TDocumentsVerificationManager manager(VerificationManagerConfig, server->GetDriveAPI()->GetDocumentPhotosManager());

    ResendTagsToYang(server, manager);

    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TTagsOnlyCreator::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSString>("requests_path", "Путь к директории заданий").SetRequired(true);
    scheme.Add<TFSString>("temp_requests_path", "Путь к временной директории для хранения заданий до записи в базу").SetRequired(false);
    scheme.Add<TFSString>("yt_clusters", "Кластера YT (через запятую)").SetRequired(true);
    scheme.Add<TFSString>("geo", "Гео регистраций").SetRequired(true);
    scheme.Add<TFSNumeric>("max_pool_size", "Максимальный возможный размер пула").SetRequired(true);
    scheme.Add<TFSString>("recheck_tag_name", "Имя тега для повторной отправки на проверку").SetRequired(false);
    scheme.Add<TFSVariants>("document_types", "Отправляемые документы").InitVariants<NUserDocument::EType>().SetMultiSelect(true).SetRequired(false);

    return scheme;
}

NJson::TJsonValue TTagsOnlyCreator::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    NJson::InsertField(result, "requests_path", VerificationManagerConfig.GetRequestsPath());
    NJson::InsertField(result, "temp_requests_path", VerificationManagerConfig.GetTempRequestsPath());
    NJson::InsertField(result, "yt_clusters", JoinStrings(VerificationManagerConfig.GetYtClusters(), ","));
    NJson::InsertField(result, "geo", JoinStrings(VerificationManagerConfig.GetGeoVector(), ","));
    NJson::InsertField(result, "max_pool_size", VerificationManagerConfig.GetMaxPoolSize());
    NJson::InsertField(result, "recheck_tag_name", ResendToYangTagName);
    NJson::InsertField(result, "document_types", VerificationManagerConfig.GetDocumentTypes());

    return result;
}

bool TTagsOnlyCreator::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }

    JREAD_STRING(jsonInfo, "requests_path", VerificationManagerConfig.MutableRequestsPath());
    JREAD_STRING_OPT(jsonInfo, "recheck_tag_name", ResendToYangTagName);
    JREAD_UINT(jsonInfo, "max_pool_size", VerificationManagerConfig.MutableMaxPoolSize());

    {
        TString ytClustersStr;
        JREAD_STRING(jsonInfo, "yt_clusters", ytClustersStr);
        VerificationManagerConfig.MutableYtClusters() = SplitString(ytClustersStr, ",");
    }

    {
        TString geoStr;
        JREAD_STRING(jsonInfo, "geo", geoStr);
        auto geoVector = SplitString(geoStr, ",");
        auto& geoSet = VerificationManagerConfig.MutableGeo();
        for (auto&& token : geoVector) {
            geoSet.insert(std::move(token));
        }
    }

    if (!TJsonProcessor::ReadContainer(jsonInfo, "document_types", VerificationManagerConfig.MutableDocumentTypes())) {
        return false;
    }

    return NJson::ParseField(jsonInfo["temp_requests_path"], VerificationManagerConfig.MutableTempRequestsPath());
}

// TDocumentsVerificationYangTasksCreator

TExpectedState TCreator::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();
    const TCreatorState* state = dynamic_cast<const TCreatorState*>(stateExt.Get());

    auto lastAssignmentsCreated = state ? state->GetLastProcessedInstant() : ModelingNow() - TDuration::Hours(5);
    TInstant nextAssignmentsCreated = TInstant::Now();
    if (!server || !server->GetDriveAPI() || !server->GetDriveAPI()->HasDocumentPhotosManager()) {
        return MakeUnexpected<TString>("user photo manager undefined");
    }
    TDocumentsVerificationManager manager(GetVerificationManagerConfig(), server->GetDriveAPI()->GetDocumentPhotosManager());

    auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
    if (!manager.CreateNewAssignments(lastAssignmentsCreated - LastAssignmentFreshness, server, session) || !session.Commit()) {
        ERROR_LOG << "unable to create yang assignments" << Endl;
        return stateExt;
    }

    TTagsOnlyBase::ResendTagsToYang(server, manager);

    THolder<TCreatorState> result(new TCreatorState);
    result->SetLastProcessedInstant(nextAssignmentsCreated);
    return result.Release();
}

NDrive::TScheme TCreator::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TTagsOnlyBase::DoGetScheme(server);

    scheme.Add<TFSVariants>("task_type", "Тип задания").InitVariants<TDocumentsVerificationConfig::ETaskType>();

    auto impl = server.GetAs<NDrive::IServer>();
    if (impl) {
        scheme.Add<TFSVariants>("required_origin_chats", "Требуемый чат-источник фотографий")
            .SetVariants(NContainer::Keys(impl->GetChatRobots()))
            .SetEditable(true)
            .SetMultiSelect(true);
    } else {
        scheme.Add<TFSVariants>("required_origin_chats", "Требуемый чат-источник фотографий").SetEditable(true).SetMultiSelect(true);
    }

    scheme.Add<TFSArray>("required_chat_nodes", "В каких нодах должен находиться чат").SetElement<TFSString>();
    scheme.Add<TFSBoolean>("require_recognizer_meta", "Использовать только фотографии после предзаполнения CV");
    scheme.Add<TFSString>("last_assignment_fresheness", "Искать новые задания за последние").SetRequired(false);

    return scheme;
}

NJson::TJsonValue TCreator::DoSerializeToJson() const {
    NJson::TJsonValue result = TTagsOnlyBase::DoSerializeToJson();

    NJson::InsertField(result, "task_type", ToString(GetVerificationManagerConfig().GetTaskType()));
    NJson::InsertField(result, "required_origin_chats", GetVerificationManagerConfig().GetRequiredOriginChats());
    NJson::InsertField(result, "required_chat_nodes", GetVerificationManagerConfig().GetRequiredChatNodes());
    NJson::InsertField(result, "require_recognizer_meta", GetVerificationManagerConfig().GetRequireRecognizerMeta());
    NJson::InsertField(result, "last_assignment_fresheness", NJson::Hr(LastAssignmentFreshness));

    return result;
}

bool TCreator::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TTagsOnlyBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }

    JREAD_FROM_STRING_OPT(jsonInfo, "task_type", MutableProtectedVerificationManagerConfig().MutableTaskType());

    return
        NJson::ParseField(jsonInfo["required_origin_chats"], MutableProtectedVerificationManagerConfig().MutableRequiredOriginChats()) &&
        NJson::ParseField(jsonInfo["required_chat_nodes"], MutableProtectedVerificationManagerConfig().MutableRequiredChatNodes()) &&
        NJson::ParseField(jsonInfo["require_recognizer_meta"], MutableProtectedVerificationManagerConfig().MutableRequireRecognizerMeta()) &&
        NJson::ParseField(jsonInfo["last_assignment_fresheness"], NJson::Hr(LastAssignmentFreshness));
}
