#include "processor.h"

#include <drive/backend/rt_background/manager/state.h>

#include <drive/backend/data/user_tags.h>
#include <drive/backend/data/user_insurance.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/library/unistat/cache.h>


TRTUserInsuranceWatcher::TFactory::TRegistrator<TRTUserInsuranceWatcher> TRTUserInsuranceWatcher::Registrator(TRTUserInsuranceWatcher::GetTypeName());

TExpectedState TRTUserInsuranceWatcher::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const TExecutionContext& context) const {
    const auto& server = context.GetServerAs<NDrive::IServer>();
    const auto& driveApi = *Yensured(server.GetDriveAPI());
    const auto& tagsManager = driveApi.GetTagsManager();

    TSet<TString> tagsToRemove;
    {
        auto tx = tagsManager.GetUserTags().BuildTx<NSQL::ReadOnly>();
        auto dbTags = tagsManager.GetUserTags().RestoreTagsRobust({}, MakeVector(TagNames), tx);
        if (!dbTags) {
            return MakeUnexpected<TString>("Fail to fetch tags: " + tx.GetStringReport() );
        }

        TMap<TString, TInstant> tagsToCheck;
        for (auto&& dbTag : *dbTags) {
            auto tag = dbTag.GetTagAs<TUserKaskoTag>();
            if (!tag) {
                ERROR_LOG << GetRobotId() << ": wrong tag type: " << dbTag.GetTagId() << Endl;
                continue;
            }
            auto status = tag->HasKaskoData() ? tag->GetKaskoDataRef().GetStatus() : NDrive::NRenins::EKaskoStatus::NoData;
            if (status != NDrive::NRenins::EKaskoStatus::NoData
                && status != NDrive::NRenins::EKaskoStatus::Signed
                && tag->GetKaskoDataRef().GetInsuranceStart() < ModelingNow())
            {
                tagsToRemove.insert(dbTag.GetTagId());
                continue;
            }
            if (auto timeout = CleanMap.FindPtr(ToString(status))) {
                tagsToCheck.emplace(dbTag.GetTagId(), ModelingNow() - *timeout);
            }
        }

        if (!tagsToCheck.empty()) {
            IEntityTagsManager::TQueryOptions options;
            options.SetTagIds(MakeVector(NContainer::Keys(tagsToCheck)));
            options.SetActions({ EObjectHistoryAction::Add });
            auto events = tagsManager.GetUserTags().GetHistoryManager().GetEventsByRanges({}, {}, tx, options);
            if (!events) {
                return MakeUnexpected<TString>("Fail to fetch tags history: " + tx.GetStringReport() );
            }
            for (auto&& event : *events) {
                if (auto ptr = tagsToCheck.FindPtr(event.GetTagId()); ptr && event.GetHistoryInstant() < *ptr) {
                    tagsToRemove.insert(event.GetTagId());
                }
            }
        }
    }

    if (!tagsToRemove.empty()) {
        auto tx = tagsManager.GetUserTags().BuildTx<NSQL::Writable>();
        auto dbTags = tagsManager.GetUserTags().RestoreTags(tagsToRemove, tx);
        if (!dbTags) {
            return MakeUnexpected<TString>("Fail to fetch tags to remove: " + tx.GetStringReport() );
        }
        if (!tagsManager.GetUserTags().RemoveTags(*dbTags, GetRobotUserId(), &server, tx, /* force = */ true) || !tx.Commit()) {
            return MakeUnexpected<TString>("Fail to remove tags: " + tx.GetStringReport() );
        }
    }

    return stateExt;
}

NDrive::TScheme TRTUserInsuranceWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    auto& frServer = *Yensured(server.GetAs<NDrive::IServer>());
    scheme.Add<TFSVariants>("tag_names", "Теги для обработки").SetVariants(
        NContainer::Keys(frServer.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::User, { TUserKaskoTag::Type() }))
    ).SetMultiSelect(true);

    auto& item = scheme.Add<TFSArray>("clean_map", "Таблица удаления тегов").SetElement<NDrive::TScheme>();
    item.Add<TFSVariants>("status", "Статус").SetVariants(GetEnumAllValues<NDrive::NRenins::EKaskoStatus>()).SetRequired(true);
    item.Add<TFSDuration>("timeout", "Время жизни после расчёта").SetRequired(true);
    return scheme;
}

bool TRTUserInsuranceWatcher::DoDeserializeFromJson(const NJson::TJsonValue& json) {
    for (auto&& item : json["clean_map"].GetArray()) {
        std::pair<TString, TDuration> ttl;
        if (!NJson::ParseField(item, "status", NJson::Stringify(ttl.first), /* required = */ true)
            || !NJson::ParseField(item, "timeout", ttl.second, /* required = */ true))
        {
            return false;
        }
        CleanMap.insert(ttl);
    }
    return TBase::DoDeserializeFromJson(json)
        && NJson::ParseField(json, "tag_names", TagNames, /* required = */ true)
        && !TagNames.empty()
        && !CleanMap.empty();
}

NJson::TJsonValue TRTUserInsuranceWatcher::DoSerializeToJson() const {
    NJson::TJsonValue json = TBase::DoSerializeToJson();
    NJson::InsertField(json, "tag_names", TagNames);
    auto& map = json.InsertValue("clean_map", NJson::JSON_ARRAY);
    for (auto&& [status, timeout] : CleanMap) {
        auto& item = map.AppendValue(NJson::JSON_MAP);
        NJson::InsertField(item, "status", ToString(status));
        NJson::InsertField(item, "timeout", NJson::ToJson(NJson::Hr(timeout)));
    }
    return json;
}
