#include "config.h"

#include <drive/backend/data/chargable.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/data/notifications_abstract.h>
#include <drive/backend/data/user_origin.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/device_snapshot/snapshots/tag.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/history_iterator/history_iterator.h>

#include <drive/library/cpp/scheme/scheme.h>

TRTUserPushSenderCounter::TFactory::TRegistrator<TRTUserPushSenderCounter> TRTUserPushSenderCounter::Registrator("user_push_sender");

class TSendInfo {
private:
    R_READONLY(TTagDescription::TConstPtr, Description);
    R_FIELD(TVector<TDBTag>, Tags);

public:
    TSendInfo(TTagDescription::TConstPtr description)
        : Description(description)
    {
    }
};

class TMessageSendTasks {
private:
    using TResultPtr = NDrive::INotifier::TResult::TPtr;

    class TTask {
    private:
        NDrive::INotifier::TMessage Message;
        TVector<TUserContacts> Recipients;
        TVector<TString> ExternalRecipients;

    public:
        TTask(const NDrive::INotifier::TMessage& message)
            : Message(message)
        {
        }

        void AddExternalRecipient(const TString& userId) {
            ExternalRecipients.push_back(userId);
        }
        void AddRecipient(const TUserContacts& contact) {
            Recipients.emplace_back(contact);
        }

        TResultPtr Send(NDrive::INotifier::TPtr notifier, const IServerBase* server = nullptr) const {
            if (notifier) {
                return notifier->Notify(Message, NDrive::INotifier::TContext()
                    .SetExternalRecipients(ExternalRecipients)
                    .SetRecipients(Recipients)
                    .SetServer(server)
                );
            } else {
                return nullptr;
            }
        }
    };

    NDrive::INotifier::TPtr Notifier;
    TMap<TString, TTask> Tasks;

public:
    TMessageSendTasks(NDrive::INotifier::TPtr notifier)
        : Notifier(notifier)
    {
    }

    void AddMessage(const NDrive::INotifier::TMessage& message, const TUserContacts& contacts) {
        const TString hashMessage = message.GetMessageHash();
        auto it = Tasks.find(hashMessage);
        if (it == Tasks.end()) {
            it = Tasks.emplace(hashMessage, message).first;
        }
        it->second.AddRecipient(contacts);
    }
    void AddMessage(const NDrive::INotifier::TMessage& message, const TString& externalUserId) {
        const TString hashMessage = message.GetMessageHash();
        auto it = Tasks.find(hashMessage);
        if (it == Tasks.end()) {
            it = Tasks.emplace(hashMessage, message).first;
        }
        it->second.AddExternalRecipient(externalUserId);
    }

    TMap<TString, TResultPtr> Send(const IServerBase* server = nullptr) const {
        TMap<TString, TResultPtr> result;
        for (auto&& i : Tasks) {
            if (auto ptr = i.second.Send(Notifier, server); ptr && ptr->GetTransitId()) {
                result.emplace(ptr->GetTransitId(), ptr);
            }
        }
        return result;
    }
};

TExpectedState TRTUserPushSenderCounter::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    NDrive::INotifier::TPtr notifier = server.GetNotifier(GetNotifier());
    if (!notifier) {
        return MakeUnexpected<TString>("incorrect notifier for PushSender: " + GetNotifier());
    }
    NDrive::INotifier::TPtr externalNotifier;
    if (ExternalNotifier) {
        externalNotifier = server.GetNotifier(ExternalNotifier);
        if (!externalNotifier) {
            return MakeUnexpected("incorrect external notifier: " + ExternalNotifier);
        }
    }

    auto sessionBuilder = server.GetDriveDatabase().GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", Now());

    TMap<TString, TSendInfo> tagsSendInfo;
    TVector<TDBTag> tags;
    const TInstant now = ModelingNow();
    TMap<TString, TVector<TString>> usersByTagName;
    ui32 incorrectTags = 0;
    TSet<TString> userIds;
    TSet<TString> unsubscribedUserIds, undefinedSubscriptionUserIds;
    {
        ITagsMeta::TTagDescriptions td = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(MessageTagType);
        TSet<TString> tagNames;
        for (auto&& i : td) {
            if (MessageTagNames.size() && !MessageTagNames.contains(i->GetName())) {
                continue;
            }
            tagNames.emplace(i->GetName());
            tagsSendInfo.emplace(i->GetName(), i);
        }

        auto session = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        IEntityTagsManager::TQueryOptions queryOptions;
        queryOptions.SetLimit(PackSize * MaxPacksPerTag);
        queryOptions.SetTags(tagNames);
        queryOptions.SetOrderBy({
            "priority",
            "tag_id"
        });
        auto optionalTags = server.GetDriveDatabase().GetTagsManager().GetUserTags().RestoreTags(session, std::move(queryOptions));
        if (!optionalTags) {
            return MakeUnexpected<TString>("cannot restore tags: " + session.GetStringReport());
        }
        tags = std::move(*optionalTags);

        for (auto&& i : tags) {
            const INotificationTag* notificationTag = i.GetTagAs<INotificationTag>();
            if (!notificationTag) {
                continue;
            }
            if (notificationTag->GetSendInstant() > now) {
                continue;
            }
            auto it = tagsSendInfo.find(i->GetName());
            if (it == tagsSendInfo.end()) {
                ++incorrectTags;
            } else {
                it->second.MutableTags().emplace_back(i);
                userIds.emplace(i.GetObjectId());
                if (!unsubscribedUserIds.contains(i.GetObjectId())) {
                    TUserSetting userSettings(i.GetObjectId());
                    auto subscription = userSettings.GetValue(SubscriptionStatusTag, SubscriptionStatusField, session);
                    if (!subscription && subscription.GetError() != TUserSettingStatus::NotFound) {
                        ERROR_LOG << "Can't get subscription info for user " << i.GetObjectId()  << ", error: " << session.GetStringReport() << Endl;
                        undefinedSubscriptionUserIds.emplace(i.GetObjectId());
                        session = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                    } else if (subscription && IsFalse(subscription.GetRef())) {
                        unsubscribedUserIds.emplace(i.GetObjectId());
                    }
                }
            }
        }
    }

    auto deletedUserIds = server.GetDriveAPI()->GetUsersData()->GetUsersDeletedByTags();

    if (incorrectTags) {
        WARNING_LOG << "Incorrect tags for send push: " << incorrectTags << Endl;
    }

    auto locale = ELocalization::Rus;
    auto session = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
    auto gUsers = server.GetDriveAPI()->GetUsersData()->FetchInfo(userIds, session);
    for (auto&& [tagName, tagInfo] : tagsSendInfo) {
        const INotificationsTagDescription* pushTag = dynamic_cast<const INotificationsTagDescription*>(tagInfo.GetDescription().Get());
        if (!pushTag) {
            return nullptr;
        }
        if (pushTag->HasActivityInterval()) {
            if (!pushTag->GetActivityIntervalRef().IsActualNow(now)) {
                INFO_LOG << GetRobotId() << ": tag " << tagName << " is inactive now" << Endl;
                continue;
            }
        }
        bool ignoreSubscription = pushTag->GetCommunicationChannelSettings().GetIgnoreSubscriptionStatus();
        ui32 packsCount = 0;
        for (ui32 i = 0; i < tagInfo.GetTags().size(); ++packsCount) {
            if (server.GetRTBackgroundManager() && !server.GetRTBackgroundManager()->IsActive()) {
                return MakeUnexpected<TString>("Stopped with RTBackgroundManager finishing");
            }

            if (GetMaxPacksPerTag() && packsCount >= GetMaxPacksPerTag()) {
                INFO_LOG << "Pack limit exceeded for " << pushTag->GetName() << Endl;
                break;
            }

            const ui32 nextI = Min<ui32>(tagInfo.GetTags().size(), i + GetPackSize());

            TMessageSendTasks sendTasks(notifier);
            TMessageSendTasks externalTasks(externalNotifier);
            TVector<TDBTag> tagsForRemove;
            TMap<TString, TAtomicSharedPtr<TTagSnapshot>> sentSnapshots;
            for (ui32 j = i; j < nextI; ++j) {
                TDBTag dbTag = tagInfo.GetTags()[j];
                const INotificationTag* notificationTag = dbTag.GetTagAs<INotificationTag>();
                if (!notificationTag) {
                    ERROR_LOG << "Incorrect notification tag for " << dbTag->GetName() << Endl;
                    continue;
                }
                const auto& userId = dbTag.GetObjectId();
                if (!ignoreSubscription && undefinedSubscriptionUserIds.contains(userId)) {
                    INFO_LOG << "Skipped tag " << pushTag->GetName() << " for " << userId << " because subscription status is unknown" << Endl;
                    continue;
                }

                if (!ignoreSubscription && unsubscribedUserIds.contains(userId)) {
                    auto tagSnapshot = MakeAtomicShared<TTagSnapshot>(TTagSnapshot::EActionReason::UserUnsubscribedFromCommunitation);
                    dbTag->SetObjectSnapshot(tagSnapshot);
                    tagsForRemove.emplace_back(dbTag);
                    INFO_LOG << "Skipped tag " << pushTag->GetName() << " for " << userId << " because user was unsubscribed" << Endl;
                    continue;
                }

                auto userData = gUsers.GetResultPtr(dbTag.GetObjectId());
                if (!userData) {
                    auto tagSnapshot = MakeAtomicShared<TTagSnapshot>(TTagSnapshot::EActionReason::WrongUser);
                    dbTag->SetObjectSnapshot(tagSnapshot);
                    tagsForRemove.emplace_back(dbTag);
                    ERROR_LOG << "Incorrect user for tag " << dbTag.GetTagId() << Endl;
                    continue;
                }

                auto message = notificationTag->BuildMessage(locale, server, *pushTag, *userData);
                if (!message) {
                    NDrive::TEventLog::Log("NotificationSenderSkip", NJson::TMapBuilder
                        ("reason", message.GetError())
                        ("tag_id", dbTag.GetTagId())
                        ("user_id", dbTag.GetObjectId())
                    );
                    continue;
                }
                message->SetTransitId(dbTag.GetTagId());

                TString externalUserId;
                if (pushTag->IsUseOrigin()) {
                    auto optionalExternalUserId = TUserOriginTag::GetExternalUserId(userData->GetUserId(), server, session);
                    if (optionalExternalUserId) {
                        externalUserId = *optionalExternalUserId;
                    }
                }
                if (externalNotifier && !externalUserId) {
                    externalUserId = notificationTag->GetExternalUserId();
                }
                if (externalNotifier && !externalUserId && pushTag->IsEnabled() && pushTag->IsExternalEnabled()) {
                    if (!sessionBuilder) {
                        return MakeUnexpected<TString>("session builder is missing");
                    }
                    auto timestamp = notificationTag->GetSendInstant();
                    if (!timestamp) {
                        timestamp = Now();
                    }
                    auto userSessions = sessionBuilder->GetUserSessions(userId);
                    for (auto&& userSession : userSessions) {
                        if (!userSession) {
                            continue;
                        }
                        if (userSession->GetStartTS() > timestamp) {
                            continue;
                        }
                        if (userSession->GetClosed() && userSession->GetLastTS() < timestamp) {
                            continue;
                        }
                        auto billingSession = std::dynamic_pointer_cast<TBillingSession>(userSession);
                        if (!billingSession) {
                            ERROR_LOG << GetRobotId() << ": cannot cast session " << userSession->GetInstanceId() << " to BillingSession" << Endl;
                            continue;
                        }
                        auto currentOffer = billingSession->GetCurrentOffer();
                        externalUserId = currentOffer ? currentOffer->GetExternalUserId() : TString();
                        break;
                    }
                    if (!externalUserId && pushTag->IsFetchExternalUserIdFromHistorySessions()) {
                        THistoryRidesContext context(server);
                        auto ydbTx = server.GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("rtuser_push_sender_counter", &server);
                        R_ENSURE(context.InitializeUser(userId, session, ydbTx, TInstant::Max(), 1), {}, "cannot initialize HistoryRidesContext");
                        if (const auto& sessions = context.GetSessions(TInstant::Max(), 1)) {
                            R_ENSURE(sessions.ysize() == 1, {}, "cannot fetch last history session");
                            const auto& lastSession = sessions.back();
                            if (lastSession.GetBillingSession() && lastSession.GetBillingSession()->GetCurrentOffer()) {
                                externalUserId = lastSession.GetBillingSession()->GetCurrentOffer()->GetExternalUserId();
                                NDrive::TEventLog::Log("FetchExternalUserIdFromHistorySessions", NJson::TMapBuilder
                                    ("user_id", userId)
                                    ("external_user_id", externalUserId)
                                );
                            }
                        }
                    }
                }
                if (!deletedUserIds.contains(userData->GetUserId()) && notificationTag->GetDeadline() > now) {
                    if (externalUserId) {
                        externalTasks.AddMessage(*message, externalUserId);
                    } else {
                        sendTasks.AddMessage(*message, *userData);
                    }
                    if (pushTag->GetEnabled()) {
                        auto snapshot = MakeAtomicShared<TTagSnapshot>(TTagSnapshot::EActionReason::NotificationSent);
                        dbTag->SetObjectSnapshot(snapshot);
                        sentSnapshots.emplace(message->GetTransitId(), snapshot);
                    } else {
                        dbTag->SetObjectSnapshot(MakeAtomicShared<TTagSnapshot>(TTagSnapshot::EActionReason::NotificationDisabled));
                    }
                } else if (notificationTag->GetDeadline() <= now) {
                    dbTag->SetObjectSnapshot(MakeAtomicShared<TTagSnapshot>(TTagSnapshot::EActionReason::NotificationTooOld));
                } else {
                    dbTag->SetObjectSnapshot(MakeAtomicShared<TTagSnapshot>(TTagSnapshot::EActionReason::DeletedUser));
                }
                tagsForRemove.emplace_back(dbTag);
            }
            if (pushTag->GetEnabled()) {
                for (auto&& [tagId, taskResult] : sendTasks.Send(&server)) {
                    if (!taskResult) {
                        continue;
                    }
                    if (auto snapshotPtr = sentSnapshots.FindPtr(tagId)) {
                        if (!snapshotPtr->Get()) {
                            continue;
                        }
                        if (taskResult->HasErrors()) {
                            snapshotPtr->Get()->AddHistoryActionReason(TTagSnapshot::EActionReason::NotificationError);
                        }
                        snapshotPtr->Get()->SetMeta(taskResult->SerializeToJson());
                    }
                }
                externalTasks.Send();
            } else {
                NOTICE_LOG << "Omitting push of " << tagName << " for " << nextI - i << " recipients" << Endl;
            }
            for (ui32 att = 0; att < 7; ++att) {
                auto session = server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
                if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTagsSimple(tagsForRemove, GetRobotUserId(), session, true) || !session.Commit()) {
                    WARNING_LOG << att << "attempt for remove user push tags: " << session.GetStringReport() << Endl;
                    continue;
                } else {
                    break;
                }
            }
            if (nextI != tagInfo.GetTags().size()) {
                Sleep(GetPacksInterval());
            }
            i = nextI;
        }
    }
    return new IRTBackgroundProcessState();
}

NDrive::TScheme TRTUserPushSenderCounter::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("notifier", "Название агента посылки сообщений").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSVariants>("external_notifier").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSNumeric>("pack_size", "Размер пакета сообщений").SetDefault(1000);
    scheme.Add<TFSNumeric>("max_packs_per_tag", "Число пакетов одного типа на вызов робота").SetDefault(MaxPacksPerTag);
    scheme.Add<TFSDuration>("packs_interval", "Интервал между отправками пакетов").SetDefault(TDuration::Seconds(30));

    TSet<TString> tagTypes;
    TSet<TString> tagNames, dictionaryTagNames;
    TSet<TString> tagKeys;
    ITag::TFactory::GetRegisteredKeys(tagKeys);
    for (auto&& i : tagKeys) {
        THolder<ITag> tag(ITag::TFactory::Construct(i));
        const INotificationTag* nTag = dynamic_cast<const INotificationTag*>(tag.Get());
        if (nTag) {
            tagTypes.emplace(i);
            auto tags = server.GetAs<NDrive::IServer>()->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(i);
            for (auto&& j : tags) {
                tagNames.emplace(j->GetName());
            }
        }
    }
    auto dictionaryTags = server.GetAs<NDrive::IServer>()->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(TUserDictionaryTag::TypeName);
    for (auto&& dictTag : dictionaryTags) {
        dictionaryTagNames.emplace(dictTag->GetName());
    }

    scheme.Add<TFSVariants>("message_tag_type", "Тип обрабатываемых писем").SetVariants(tagTypes);
    scheme.Add<TFSVariants>("message_tag_names", "Именования тегов для выборочной обработки").SetVariants(tagNames);
    scheme.Add<TFSVariants>("subscription_status_tag", "Тег с информацией об отписке").SetVariants(dictionaryTagNames).SetEditable(true);
    scheme.Add<TFSString>("subscription_status_field", "Поле тега с информацией об отписке");
    return scheme;
}

NJson::TJsonValue TRTUserPushSenderCounter::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    TJsonProcessor::Write(result, "notifier", Notifier);
    TJsonProcessor::Write(result, "external_notifier", ExternalNotifier);
    TJsonProcessor::Write(result, "pack_size", PackSize);
    TJsonProcessor::Write(result, "max_packs_per_tag", MaxPacksPerTag);
    TJsonProcessor::WriteDurationString(result, "packs_interval", PacksInterval);
    TJsonProcessor::Write(result, "message_tag_type", MessageTagType);
    TJsonProcessor::WriteContainerString(result, "message_tag_names", MessageTagNames);
    NJson::InsertField(result, "subscription_status_tag", SubscriptionStatusTag);
    NJson::InsertField(result, "subscription_status_field", SubscriptionStatusField);
    return result;
}

bool TRTUserPushSenderCounter::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }

    JREAD_STRING(jsonInfo, "notifier", Notifier);
    JREAD_STRING_OPT(jsonInfo, "external_notifier", ExternalNotifier);
    JREAD_UINT_OPT(jsonInfo, "pack_size", PackSize);
    JREAD_UINT_OPT(jsonInfo, "max_packs_per_tag", MaxPacksPerTag);
    JREAD_DURATION_OPT(jsonInfo, "packs_interval", PacksInterval);
    JREAD_STRING_OPT(jsonInfo, "message_tag_type", MessageTagType);
    TJsonProcessor::ReadContainer(jsonInfo, "message_tag_names", MessageTagNames);
    return
        NJson::ParseField(jsonInfo, "subscription_status_tag", SubscriptionStatusTag) &&
        NJson::ParseField(jsonInfo, "subscription_status_field", SubscriptionStatusField);
}
