#include "process.h"

#include <drive/backend/compiled_riding/compiled_riding.h>
#include <drive/backend/database/config.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/history/event.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTYdbDumperWatcher> TRTYdbDumperWatcher::Registrator(TRTYdbDumperWatcher::GetTypeName());
TRTYdbDumperState::TFactory::TRegistrator<TRTYdbDumperState> TRTYdbDumperState::Registrator(TRTYdbDumperWatcher::GetTypeName());

TString TRTYdbDumperState::GetType() const {
    return TRTYdbDumperWatcher::GetTypeName();
}

TRTYdbDumperWatcher::TSchema TRTYdbDumperWatcher::GetEntitySchema() const {
    if (GetDBType() == "trace_tags") {
        return TDBTag::GetYDBSchema();
    }
    ythrow yexception() << "Unsupported DBType!";
}

TRTYdbDumperWatcher::TSchema TRTYdbDumperWatcher::GetHistorySchema() const {
    if (GetDBType() == "compiled_rides") {
        return TObjectEvent<TCompiledRiding>::GetYDBSchema();
    }
    if (GetDBType() == "trace_tags") {
        return TObjectEvent<TDBTag>::GetYDBSchema();
    }
    ythrow yexception() << "Unsupported DBType!";
}

NDrive::TScheme TRTYdbDumperWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    TVector<TString> dbNames = server.GetDatabaseNames();
    dbNames.erase(std::remove_if(dbNames.begin(), dbNames.end(),
        [&server](TString name) {
            return std::dynamic_pointer_cast<TYdbDatabase>(server.GetDatabase(name)) == nullptr;}),
        dbNames.end());

    scheme.Add<TFSVariants>("destination_db_name", "База данных для записи").SetVariants(dbNames);
    scheme.Add<TFSVariants>("db_type", "Тип данных в таблице").SetVariants({
        "compiled_rides", "trace_tags"
    });
    scheme.Add<TFSString>("destination_table_name", "Название таблицы с историей для записи");
    scheme.Add<TFSString>("destination_entiry_table_name", "Название entity таблицы для записи (опционально)");
    scheme.Add<TFSString>("source_entiry_table_name", "Название entity таблицы для чтения (опционально)");
    return scheme;
}

bool TRTYdbDumperWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }

    return NJson::ParseField(jsonInfo["db_type"], DBType) &&
        NJson::ParseField(jsonInfo["destination_db_name"], DestinationDBName) &&
        NJson::ParseField(jsonInfo["destination_table_name"], DestinationTableName) &&
        NJson::ParseField(jsonInfo["destination_entiry_table_name"], DestinationEntityTableName, false) &&
        NJson::ParseField(jsonInfo["source_entiry_table_name"], SourceEntityTableName, false);
}

NJson::TJsonValue TRTYdbDumperWatcher::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["db_type"] = DBType;
    result["destination_db_name"] = DestinationDBName;
    result["destination_table_name"] = DestinationTableName;
    if (DestinationEntityTableName) {
        result["destination_entiry_table_name"] = *DestinationEntityTableName;
    }
    if (SourceEntityTableName) {
        result["source_entiry_table_name"] = *SourceEntityTableName;
    }
    return result;
}

TString TRTYdbDumperWatcher::ConvertTypes(
    const TSet<TString>& columnNames,
    const NStorage::TTableRecord& record,
    const TSchema& schema,
    NStorage::ITransaction* tx) const {

    TStringBuilder values;
    for (const TString& name : columnNames) {
        if (values) {
            values << ", ";
        }

        auto typePointer = schema.find(name);
        EPrimitiveType type = typePointer->second;

        if (type == EPrimitiveType::Int64 ||
            type == EPrimitiveType::Int32) {
            values << record.Get(name);
            continue;
        }
        if (type == EPrimitiveType::String) {
            values << tx->Quote(record.Get(name));
            continue;
        }
        if (type == EPrimitiveType::JsonDocument) {
            TString json = record.Get(name);
            if (!json) {
                json = "null";
            }
            values << "JsonDocument(" << tx->Quote(json) << ")";
            continue;
        }
    }
    return values;
}

bool TRTYdbDumperWatcher::CheckSchema(const TRecordsSet& records, const TSchema& schema) const {
    Y_ENSURE(!records.empty());
    auto inputColumnNames = records.GetRecords().front().GetKeysSet();

    TSet<TString> correctColumnNames;
    for (const TString& name : inputColumnNames) {
        if (!schema.contains(name)) {
            ythrow yexception() << "Column with name " << name << " doesn't exist!";
        } else {
            correctColumnNames.insert(name);
        }
    }
    if (correctColumnNames.size() != schema.size()) {
        ythrow yexception() << "There are " << schema.size() - correctColumnNames.size() << " missing columns in records!";
    }
    return true;
}

TString TRTYdbDumperWatcher::MakeHeader(const TSet<TString>& columnNames) const {
    TStringBuilder queryString;
    for (const TString& name : columnNames) {
        if (queryString) {
            queryString << ", ";
        }
        queryString << name;
    }
    return queryString;
}

TString TRTYdbDumperWatcher::MakeUpsertQuery(const TRecordsSet& records, const TString& tableName, const TSchema& schema, NStorage::ITransaction::TPtr tx) const {
    TStringBuilder query;
    Y_ENSURE(!records.empty());
    auto columnNames = records.GetRecords().front().GetKeysSet();
    query << "UPSERT INTO " << tableName << " (" << MakeHeader(columnNames) << ")";
    query << " VALUES ";

    TStringBuilder values;
    for (auto&& record : records.GetRecords()) {
        if (values) {
            values << " ,";
        }
        values << "(" << ConvertTypes(columnNames, record, schema, tx.Get()) << ")";
    }
    query << values;
    return query;
}

TString TRTYdbDumperWatcher::MakeDeleteQuery(const TString& tableName, const TSet<TString>& tagIds, NStorage::ITransaction::TPtr tx) const {
    return TStringBuilder() << "DELETE FROM " << tableName << " WHERE tag_id IN (" << Yensured(tx)->Quote(tagIds) << ")";
}

TRecordsSet TRTYdbDumperWatcher::GetTags(const TSet<TString>& tagIds, NStorage::ITransaction::TPtr tx, const TString& tableName) const {
        TRecordsSet records;
        NSQL::TQueryOptions options;
        options.SetGenericCondition("tag_id", tagIds);

        auto query = options.PrintQuery(*Yensured(tx), tableName);
        auto queryResult = Yensured(tx)->Exec(query, &records);
        if (!queryResult || !queryResult->IsSucceed()) {
            return {};
        }
        return records;
}

void TRTYdbDumperWatcher::DumpHistoryData(NStorage::IDatabase::TPtr destinationDatabase, const TRecordsSet& records, ui64& lastEventId) const {
    NStorage::TCreateTransactionOptions options;
    auto tx = Yensured(destinationDatabase->CreateTransaction(options));

    CheckSchema(records, GetHistorySchema());

    const auto table = GetTable();
    Y_ENSURE(table);
    const auto eventIdFieldName = table->GetEventIdFieldName();
    for (auto&& record : records.GetRecords()) {
        ui64 eventId = 0;
        if (!record.TryGet(eventIdFieldName, eventId)) {
            ALERT_LOG << "Invalid record: " << record.BuildSet(*tx.Get());
            continue;
        }
        lastEventId = Max<ui64>(eventId, lastEventId);
    }
    auto result = tx->Exec(MakeUpsertQuery(records, DestinationTableName, GetHistorySchema(), tx));
    if (!result->IsSucceed()) {
        ythrow yexception() << "Invalid Exec Operation. Status: " << tx->GetErrors().GetStringReport();
    }

    if (!tx->Commit()) {
        ythrow yexception() << "Commit failed: " << tx->GetErrors().GetStringReport();
    }
}

void TRTYdbDumperWatcher::DumpEntityData(NStorage::IDatabase::TPtr sourceDatabase, NStorage::IDatabase::TPtr destinationDatabase, const TRecordsSet& records) const {
    TSet<TString> toRemove;
    TSet<TString> toUpsert;

    for (auto&& record : records.GetRecords()) {
        EObjectHistoryAction historyAction;
        record.TryGet("history_action", historyAction);
        TString tagId = record.Get("tag_id");

        if (historyAction == EObjectHistoryAction::Remove) {
            if (toUpsert.contains(tagId)) {
                toUpsert.erase(tagId);
            } else {
                toRemove.insert(tagId);
            }
        } else {
            if (toRemove.contains(tagId)) {
                toRemove.erase(tagId);
            } else {
                toUpsert.insert(tagId);
            }
        }
    }

    if (!toUpsert.empty()) {
        auto readSourceTransaction = Yensured(sourceDatabase->CreateTransaction(false));
        TRecordsSet entityRecords = GetTags(toUpsert, readSourceTransaction, *SourceEntityTableName);

        if (!entityRecords.empty()) {
            CheckSchema(entityRecords, GetEntitySchema());
            auto tx = Yensured(destinationDatabase->CreateTransaction(NStorage::TCreateTransactionOptions()));
            auto result = tx->Exec(MakeUpsertQuery(entityRecords, *DestinationEntityTableName, GetEntitySchema(), tx));
            if (!result->IsSucceed()) {
                ythrow yexception() << "Invalid Exec Operation. Status: " << tx->GetErrors().GetStringReport();
            }

            if (!tx->Commit()) {
                ythrow yexception() << "Commit failed: " << tx->GetErrors().GetStringReport();
            }
        }
    }

    if (!toRemove.empty()) {
        auto tx = Yensured(destinationDatabase->CreateTransaction(NStorage::TCreateTransactionOptions()));

        auto result = tx->Exec(MakeDeleteQuery(*DestinationEntityTableName, toRemove, tx));
        if (!result->IsSucceed()) {
            ythrow yexception() << "Invalid Exec Operation. Status: " << tx->GetErrors().GetStringReport();
        }

        if (!tx->Commit()) {
            ythrow yexception() << "Commit failed: " << tx->GetErrors().GetStringReport();
        }
    }
}

bool TRTYdbDumperWatcher::ProcessRecords(const TRecordsSet& records, const NDrive::IServer& server, ui64& lastEventId, TMessagesCollector& errors) const {
    TUnistatSignalsCache::SignalAdd("ydb_dumper-" + GetRTProcessName(), "records", 0);
    TUnistatSignalsCache::SignalAdd("ydb_dumper-" + GetRTProcessName(), "errors", 0);
    try {
        if (records.empty()) {
            TUnistatSignalsCache::SignalAdd("ydb_dumper-" + GetRTProcessName(), "records", 0);
            return true;
        }

        const auto destinationDatabase = server.GetDatabase(DestinationDBName);
        if (!destinationDatabase) {
            ythrow yexception() << "database " + DestinationDBName + " is missing";
        }

        DumpHistoryData(destinationDatabase, records, lastEventId);

        if (DestinationEntityTableName && SourceEntityTableName) {
            const auto table = GetTable();
            const auto sourceDatabase = table ? table->GetDatabasePtr() : nullptr;
            if (!sourceDatabase) {
                ythrow yexception() << "Sequential table for " + GetTableName() + " is missing";
            }
            DumpEntityData(sourceDatabase, destinationDatabase, records);
        }
        TUnistatSignalsCache::SignalAdd("ydb_dumper-" + GetRTProcessName(), "records", records.GetRecords().size());
    } catch (const std::exception& e) {
        errors.AddMessage("YdbError", FormatExc(e));
        TUnistatSignalsCache::SignalAdd("ydb_dumper-" + GetRTProcessName(), "errors", 1);
        return false;
    }
    return true;
}
