#include "process.h"

#include <library/cpp/testing/unittest/registar.h>

#include <drive/backend/ut/library/helper.h>
#include <drive/backend/ut/library/helper2.h>
#include <drive/backend/compiled_riding/manager.h>

#include <util/system/env.h>

TString compiledRides = "compiled_rides";

using namespace NYdb;
using namespace NTable;

TDriver InitDriver(){
    auto driverConfig = TDriverConfig()
        .SetEndpoint(GetEnv("YDB_ENDPOINT"))
        .SetDatabase(GetEnv("YDB_DATABASE"));

    TDriver driver(driverConfig);
    return driver;
}

TTableDescription CreateTableDescription() {
    TTableDescription ydbDecsription = TTableBuilder()
            .AddNullableColumn("history_event_id", EPrimitiveType::Int64)
            .AddNullableColumn("history_user_id", EPrimitiveType::String)
            .AddNullableColumn("history_originator_id", EPrimitiveType::String)
            .AddNullableColumn("history_action", EPrimitiveType::String)
            .AddNullableColumn("history_timestamp", EPrimitiveType::Int32)
            .AddNullableColumn("history_comment", EPrimitiveType::String)
            .AddNullableColumn("session_id", EPrimitiveType::String)
            .AddNullableColumn("object_id", EPrimitiveType::String)
            .AddNullableColumn("price", EPrimitiveType::Int32)
            .AddNullableColumn("duration", EPrimitiveType::Int32)
            .AddNullableColumn("start", EPrimitiveType::Int32)
            .AddNullableColumn("finish", EPrimitiveType::Int32)
            .AddNullableColumn("meta", EPrimitiveType::JsonDocument)
            .AddNullableColumn("meta_proto", EPrimitiveType::String)
            .AddNullableColumn("hard_proto", EPrimitiveType::String)
            .SetPrimaryKeyColumn("history_event_id")
            .AddSecondaryIndex("compiled_rides_history_user_id_index", TVector<TString>({"history_user_id", "history_timestamp"}))
            .AddSecondaryIndex("compiled_rides_session_id_index", "session_id")
            .AddSecondaryIndex("compiled_rides_object_id_index", TVector<TString>({"object_id", "history_timestamp"}))
            .AddSecondaryIndex("compiled_rides_history_timestamp_index", "history_timestamp")
            .Build();
    return ydbDecsription;
}

Y_UNIT_TEST_SUITE(YdbDumper) {
    Y_UNIT_TEST(CompiledRidesExportCheck) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        configGenerator.SetNeedBackground(0);
        UNIT_ASSERT(configGenerator.SetYDBConfig());
        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);

        script.Add<NDrive::NTest::TCommonChecker>([&](NDrive::NTest::TRTContext& context) {
            Y_UNUSED(context);
            TDriver driver = InitDriver();
            TTableClient client(driver);

            auto sessionResult = client.GetSession().GetValueSync();

            UNIT_ASSERT_VALUES_EQUAL(client.GetActiveSessionCount(), 1);
            UNIT_ASSERT_C(sessionResult.IsSuccess(), sessionResult.GetIssues().ToString());

            TSession session = sessionResult.GetSession();

            auto result = session.CreateTable(GetEnv("YDB_DATABASE") + "/compiled_rides", CreateTableDescription()).GetValueSync();
            UNIT_ASSERT_C(result.IsSuccess(), result.GetIssues().ToString());
        });

        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        script.Add<NDrive::NTest::TCreateCar>().SetPosition(from);
        script.Add<NDrive::NTest::TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("pack_offer_constructor").SetUserPosition(from);
        script.Add<NDrive::NTest::TAccept>(TDuration::Zero());
        script.Add<NDrive::NTest::TRide>(TDuration::Zero());
        script.Add<NDrive::NTest::TDrop>(TDuration::Minutes(2)).SetCarPosition(to);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TCommonChecker>([&](NDrive::NTest::TRTContext& context) {
            auto process = MakeHolder<TRTYdbDumperWatcher>();
            process->SetPeriod(TDuration::Seconds(5));
            process->SetEnabled(true);
            process->SetRobotUserId(USER_ROOT_DEFAULT);
            process->SetTableName(compiledRides);
            process->SetDBType(compiledRides);
            process->SetDestinationDBName("testing-drive-ydb");
            process->SetDestinationTableName(compiledRides);

            TRTBackgroundProcessContainer container(process.Release());
            container.SetName("test_ydb_dumper");
            UNIT_ASSERT(context.GetConfigGenerator().ForceUpsertRTBackground(container, USER_ROOT_DEFAULT));
        });
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(20));
        script.Add<NDrive::NTest::TCommonChecker>([&](NDrive::NTest::TRTContext& context) {
            NDrive::TEntitySession sessionForYDB = context.GetServer()->GetYDB()->BuildSession(true);
            NDrive::TEntitySession session = context.GetDriveAPI().GetMinimalCompiledRides().BuildSession(true);

            auto queryOptions = NSQL::TQueryOptions();
            auto defaultTx = NDrive::TEntitySession();
            auto optionalCompiledSessions = context.GetDriveAPI().GetMinimalCompiledRides().GetUser<TMinimalCompiledRiding>({ USER_ID_DEFAULT }, session, defaultTx);
            TSet<TString> compiledSessionsReports;
            for (auto&& compiledSession : *optionalCompiledSessions) {
                compiledSessionsReports.insert(compiledSession.SerializeToJson().GetString());
            }

            optionalCompiledSessions = context.GetDriveAPI().GetMinimalCompiledRides().GetUser<TMinimalCompiledRiding>({ USER_ID_DEFAULT }, session, sessionForYDB);
            UNIT_ASSERT(optionalCompiledSessions);
            TSet<TString> ydbCompiledSessionsReports;
            for (auto&& compiledSession : *optionalCompiledSessions) {
                ydbCompiledSessionsReports.insert(compiledSession.SerializeToJson().GetString());
            }
            UNIT_ASSERT(compiledSessionsReports.size() != 0);
            UNIT_ASSERT(ydbCompiledSessionsReports.size() != 0);
            UNIT_ASSERT(std::equal(compiledSessionsReports.begin(), compiledSessionsReports.end(), ydbCompiledSessionsReports.begin(), ydbCompiledSessionsReports.end()));
        });
        UNIT_ASSERT(script.Execute());
    }
}
