#include "process.h"

#include <drive/backend/rt_background/manager/state.h>

#include <drive/backend/billing/manager.h>
#include <drive/backend/chat_robots/state_storage.h>
#include <drive/backend/chat_robots/state/robot_state.pb.h>
#include <drive/backend/compiled_riding/compiled_riding.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/images/database.h>
#include <drive/backend/support_center/manager.h>

#include <drive/library/cpp/yt/common/writer.h>

#include <library/cpp/yson/node/node_io.h>

#include <rtline/library/unistat/cache.h>

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTYTDumperWatcher> TRTYTDumperWatcher::Registrator(TRTYTDumperWatcher::GetTypeName());
TRTYTDumperState::TFactory::TRegistrator<TRTYTDumperState> TRTYTDumperState::Registrator(TRTYTDumperWatcher::GetTypeName());

TString TRTYTDumperState::GetType() const {
    return TRTYTDumperWatcher::GetTypeName();
}

NYT::TTableSchema TRTYTDumperWatcher::GetYtSchema() const {
    if (TYtProcessTraits::HasYtSchema()) {
        return TYtProcessTraits::GetYtSchema();
    }
    if (DBDefineSchema) {
        if (DBType == "tags_history") {
            TSchema schema = {
                { "data",                   NYT::VT_STRING },
                { "history_action",         NYT::VT_STRING },
                { "history_comment",        NYT::VT_STRING },
                { "history_event_id",       NYT::VT_UINT64 },
                { "history_originator_id",  NYT::VT_STRING },
                { "history_timestamp",      NYT::VT_UINT64 },
                { "history_user_id",        NYT::VT_STRING },
                { "object_id",              NYT::VT_STRING },
                { "performer",              NYT::VT_STRING },
                { "priority",               NYT::VT_UINT64 },
                { "snapshot",               NYT::VT_STRING },
                { "tag",                    NYT::VT_STRING },
                { "tag_id",                 NYT::VT_STRING },
                { "unpacked_data",          NYT::VT_ANY },
                { "unpacked_snapshot",      NYT::VT_ANY },
            };
            return TYtProcessTraits::GetYtSchema(schema);
        }
    }
    return {};
}

void TRTYTDumperWatcher::DumpToYtNode(const NStorage::TTableRecord& tableRecord, NYT::TNode& recordNode, const NDrive::IServer& server) const {
    auto dumper = [&recordNode](const TString& key, const TString& value) {
        char first = value ? value[0] : '\0';
        if (first == '{' || first == '[') {
            NJson::TJsonValue json(NJson::JSON_NULL);
            if (NJson::ReadJsonFastTree(value, &json)) {
                try {
                    recordNode[key] = NYT::NodeFromJsonValue(json);
                    return;
                } catch (const std::exception& e) {
                    WARNING_LOG << "cannot convert Json " << value << " to Yson: " << FormatExc(e) << Endl;
                }
            }
        }
        recordNode[key] = value;
    };
    tableRecord.Scan(dumper);

    const TString& type = DBType;
    if (type == "tags_history") {
        TObjectEvent<TDBTag> tag;
        if (tag.DeserializeFromTableRecord(tableRecord, &server.GetDriveAPI()->GetTagsManager().GetContext())) {
            {
                NJson::TJsonValue data(NJson::JSON_MAP);
                tag->SerializeSpecialDataToJson(data);
                recordNode["unpacked_data"] = NYT::NodeFromJsonValue(data);
            }
            if (auto snapshot = tag->GetObjectSnapshot()) {
                recordNode["unpacked_snapshot"] = NYT::NodeFromJsonValue(snapshot->SerializeToJson());
            } else {
                recordNode["unpacked_snapshot"] = NYT::TNode::CreateEntity();
            }
            if (HasYtSchema() || DBDefineSchema) {
                for (auto&&[field, value] : recordNode.AsMap()) {
                    if (value.IsString() && value.AsString().empty()) {
                        value = NYT::TNode::CreateEntity();
                    }
                }
            }
        }
    } else if (type == "compiled_rides") {
        TFullCompiledRiding riding;
        if (TBaseDecoder::DeserializeFromTableRecord(riding, tableRecord)) {
            auto locale = DefaultLocale;
            NJson::TJsonValue data = riding.GetReport(locale, NDriveSession::ReportAll, server);
            recordNode["unpacked_data"] = NYT::NodeFromJsonValue(data);
        }
    } else if (type == "compiled_bills") {
        TCompiledBill bill;
        if (TBaseDecoder::DeserializeFromTableRecord(bill, tableRecord)) {
            NJson::TJsonValue data = bill.GetReport();
            recordNode["unpacked_data"] = NYT::NodeFromJsonValue(data);
            recordNode["unpacked_full_data"] = NYT::NodeFromJsonValue(bill.GetFullReport());
        }
    } else if (type == "chat_states") {
        TChatRobotSerializedState stateRecord;
        if (TBaseDecoder::DeserializeFromTableRecord(stateRecord, tableRecord)) {
            NChatRobotState::TChatRobotState state;
            {
                Y_ENSURE(state.ParseFromString(Base64Decode(stateRecord.GetSerializedState())));
                recordNode["chat_node"] = state.GetCurrentStep();
                recordNode["resubmit_mask"] = state.GetResubmitMask();
                recordNode["original_resubmit_mask"] = state.GetOriginalResubmitMask();
                recordNode["queued_resubmit_mask"] = state.GetQueuedResubmitMask();
                auto talkId = stateRecord.GetTalkId();
                if (talkId.size() > 37) {
                    recordNode["user_id"] = talkId.substr(0, 36);
                    talkId.erase(0, 37);
                    auto sep = talkId.find("-");
                    if (sep == TString::npos) {
                        recordNode["chat_id"] = talkId;
                        recordNode["topic"] = "";
                    } else {
                        recordNode["chat_id"] = talkId.substr(0, sep);
                        recordNode["topic"] = talkId.substr(sep + 1);
                    }
                }
            }
        }
    } else if (type == "drive_images") {
        TCommonImageData imageData;
        if (TBaseDecoder::DeserializeFromTableRecord(imageData, tableRecord)) {
            NJson::TJsonValue data = imageData.BuildReport();
            recordNode["unpacked_data"] = NYT::NodeFromJsonValue(data);
        }
    } else if (type == "support_request_categorization") {
        const auto categorizer = Yensured(server.GetSupportCenterManager())->GetSupportRequestCategorizer();
        const auto categoryTreeManager = Yensured(categorizer)->GetTreeManager();
        const auto& category = tableRecord.Get("category");
        auto categoryNode = Yensured(categoryTreeManager)->GetNode(category);
        auto serializedCategoryMeta = categoryNode ? categoryNode->SerializeMeta() : NJson::JSON_NULL;
        recordNode["category_meta"] = NYT::NodeFromJsonValue(serializedCategoryMeta);
    }
}

bool TRTYTDumperWatcher::ProcessRecords(const TRecordsSet& records, const NDrive::IServer& server, ui64& lastEventId, TMessagesCollector& errors) const {
    auto table = GetTable();
    Y_ENSURE(table);
    auto eventIdFieldName = table->GetEventIdFieldName();
    TUnistatSignalsCache::SignalAdd("yt_dumper-" + GetRTProcessName(), "records", 0);
    TUnistatSignalsCache::SignalAdd("yt_dumper-" + GetRTProcessName(), "errors", 0);
    try {
        NYT::IClientPtr ytClient = NYT::CreateClient(YTCluster);
        NYT::TTableSchema schema = GetYtSchema();
        TYTWritersSet<TTableSelectorDay> writers(ytClient, YTDir, schema);
        for (auto&& record : records.GetRecords()) {
            ui64 eventId = 0;
            if (!record.TryGet(eventIdFieldName, eventId)) {
                continue;
            }
            lastEventId = Max<ui64>(eventId, lastEventId);

            NYT::TNode recordNode;
            DumpToYtNode(record, recordNode, server);
            recordNode = Schematize(std::move(recordNode), schema);
            writers.GetWriter(Now())->AddRow(recordNode);
            TUnistatSignalsCache::SignalAdd("yt_dumper-" + GetRTProcessName(), "records", 1);
        }
        writers.Finish();
    } catch (const std::exception& e) {
        errors.AddMessage("YTError", FormatExc(e));
        TUnistatSignalsCache::SignalAdd("yt_dumper-" + GetRTProcessName(), "errors", 1);
        return false;
    }
    return true;
}

NDrive::TScheme TRTYTDumperWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    TYtProcessTraits::FillScheme(scheme);
    scheme.Add<TFSString>("yt_cluster", "Кластер YT").SetDefault("hahn");
    scheme.Add<TFSString>("yt_dir", "Директория для экспорта");
    scheme.Add<TFSVariants>("db_type", "Тип данных в таблице").SetVariants({
        "chat_states",
        "compiled_bills",
        "compiled_rides",
        "drive_images",
        "support_request_categorization",
        "tags_history",
    });
    scheme.Add<TFSBoolean>("db_define_schema", "Тип данных в таблице задает YT схему");
    return scheme;
}

bool TRTYTDumperWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }
    if (!TYtProcessTraits::Deserialize(jsonInfo)) {
        return false;
    }
    JREAD_STRING(jsonInfo, "yt_cluster", YTCluster);
    JREAD_STRING(jsonInfo, "yt_dir", YTDir);
    JREAD_STRING_OPT(jsonInfo, "db_type", DBType);
    JREAD_BOOL_OPT(jsonInfo, "db_define_schema", DBDefineSchema);
    return true;
}

NJson::TJsonValue TRTYTDumperWatcher::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    TYtProcessTraits::Serialize(result);
    result["yt_dir"] = YTDir;
    result["yt_cluster"] = YTCluster;
    result["db_type"] = DBType;
    result["db_define_schema"] = DBDefineSchema;
    return result;
}
