#include "config.h"

#include <library/cpp/mediator/global_notifications/system_status.h>
#include <util/string/join.h>

TRTLineAPIConfig::TRTLineAPIConfig() {
}

void TRTLineAPIConfig::Init(const TYandexConfig::Section* section, const TMap<TString, NSimpleMeta::TConfig>* requestPolicy) {
    const TYandexConfig::Directives& directives = section->GetDirectives();
    {
        Host = directives.Value("Host", Host);
        AssertCorrectConfig(!!Host, "no 'Host' field");
        Port = directives.Value("Port", Port);
        AssertCorrectConfig(!!Port, "no 'Port' field");
        ServiceName = directives.Value("ServiceName", ServiceName);
        AssertCorrectConfig(!!ServiceName, "no 'ServiceName' field");

        TString balancerTimeoutTable;
        if (directives.GetValue("BalancerTimeoutTable", balancerTimeoutTable) && !balancerTimeoutTable.empty()) {
            BalancerOptions.BalancerTimeoutTable.clear();
            for (auto&& i : StringSplitter(balancerTimeoutTable).SplitBySet(" ,").SkipEmpty()) {
                BalancerOptions.BalancerTimeoutTable.push_back(TDuration::Parse(i.Token()));
            }
        }

        if (directives.GetValue("RequestPolicyName", RequestPolicyName)) {
            CHECK_WITH_LOG(requestPolicy);
            auto it = requestPolicy->find(RequestPolicyName);
            AssertCorrectConfig(it != requestPolicy->end(), "Incorrect policy %s", RequestPolicyName.data());
            SearchRequestConfig = it->second;
        } else {
            const TYandexConfig::TSectionsMap children = section->GetAllChildren();
            auto it = children.find("RequestConfig");
            AssertCorrectConfig(it != children.end(), "Incorrect RequestConfig section for rtline api");
            SearchRequestConfig.InitFromSection(it->second);
        }
        SelfClientId = directives.Value("SelfClientId", SelfClientId);
        DestinationTvmId = directives.Value("DestinationTvmId", DestinationTvmId);
        SpMetaSearch = directives.Value("SpMetaSearch", SpMetaSearch);
    }
    {
        IndexingHost = directives.Value("IndexingHost", IndexingHost);
        IndexingPort = directives.Value("IndexingPort", IndexingPort);
        IndexingToken = directives.Value("IndexingToken", IndexingToken);
        if (IndexingHost) {
            AssertCorrectConfig(IndexingPort != 0, "no 'IndexingPort' field");
            AssertCorrectConfig(!IndexingToken.empty(), "no 'IndexingToken' field");
            if (directives.GetValue("IndexingPolicyName", IndexingPolicyName)) {
                CHECK_WITH_LOG(requestPolicy);
                auto it = requestPolicy->find(IndexingPolicyName);
                AssertCorrectConfig(it != requestPolicy->end(), "Incorrect policy %s", IndexingPolicyName.data());
                IndexingRequestConfig = it->second;
            } else {
                const TYandexConfig::TSectionsMap children = section->GetAllChildren();
                auto it = children.find("IndexingRequestConfig");
                AssertCorrectConfig(it != children.end(), "Incorrect IndexingRequestConfig section for rtline api");
                IndexingRequestConfig.InitFromSection(it->second);
            }
        }
    }
}

void TRTLineAPIConfig::ToString(IOutputStream& os) const {
    {
        os << "Host: " << Host << Endl;
        os << "Port: " << Port << Endl;
        os << "ServiceName: " << ServiceName << Endl;
        if (!RequestPolicyName) {
            os << "<RequestConfig>" << Endl;
            SearchRequestConfig.ToString(os);
            os << "</RequestConfig>" << Endl;
        } else {
            os << "RequestPolicyName: " << RequestPolicyName << Endl;
        }
        os << "BalancerTimeoutTable: " << JoinSeq(" ", BalancerOptions.BalancerTimeoutTable) << Endl;
        if (SelfClientId) {
            os << "SelfClientId: " << SelfClientId << Endl;
            os << "DestinationTvmId: " << DestinationTvmId << Endl;
        }
        if (SpMetaSearch) {
            os << "SpMetaSearch: " << SpMetaSearch << Endl;
        }
    }

    if (!!IndexingHost) {
        os << "IndexingHost: " << IndexingHost << Endl;
        os << "IndexingPort: " << IndexingPort << Endl;
        os << "IndexingToken: " << IndexingToken << Endl;
        if (!IndexingPolicyName) {
            os << "<IndexingRequestConfig>" << Endl;
            IndexingRequestConfig.ToString(os);
            os << "</IndexingRequestConfig>" << Endl;
        } else {
            os << "IndexingPolicyName: " << IndexingPolicyName << Endl;
        }
    }
}
