#include "config.h"

#include <drive/backend/chat/engine.h>
#include <drive/backend/database/config.h>
#include <drive/backend/distributing_block_storage/common.h>
#include <drive/backend/doc_packages/manager.h>
#include <drive/backend/saas/api.h>
#include <drive/backend/tags/config.h>
#include <drive/telematics/api/server/config.h>

#include <drive/library/cpp/taxi/config.h>
#include <drive/library/cpp/user_events_api/client.h>

namespace {
    void Init(NClickHouse::TAsyncClientOptions& options, const TYandexConfig::Directives& directives) {
        options.Endpoints = StringSplitter(
            directives.Value("Endpoints", JoinStrings(options.Endpoints, " "))
        ).SplitBySet(" ,").SkipEmpty();
        options.Database = directives.Value("Database", options.Database);
        options.Password = directives.Value("Password", options.Password);
        options.User = directives.Value("User", options.User);
        {
            size_t maxMemoryUsage;
            if (directives.GetValue("MaxMemoryUsage", maxMemoryUsage)) {
                options.MaxMemoryUsage = maxMemoryUsage;
            }
        }
    }

    void Print(const NClickHouse::TAsyncClientOptions& options, IOutputStream& out) {
        out << "Endpoints: " << JoinStrings(options.Endpoints, " ") << Endl;
        out << "Database: " << options.Database << Endl;
        out << "Password: " << options.Password << Endl;
        out << "User: " << options.User << Endl;
        if (options.MaxMemoryUsage) {
            out << "MaxMemoryUsage: " << options.MaxMemoryUsage << Endl;
        }
    }
}

namespace NDrive {

    void THttpRewriteConfig::Init(const TYandexConfig::Directives& directives) {
        Pattern = directives.Value("Pattern", Pattern);
        Method = directives.Value("Method", Method);
        HandlerPath = directives.Value("HandlerPath", HandlerPath);
        OverrideQuery = directives.Value("OverrideQuery", OverrideQuery);
    }

    void THttpRewriteConfig::ToString(IOutputStream& os) const {
        os << "Pattern: " << Pattern << Endl;
        os << "Method: " << Method << Endl;
        os << "HandlerPath: " << HandlerPath << Endl;
        os << "OverrideQuery: " << OverrideQuery << Endl;
    }

}

void NDrive::TServerConfig::Init(const TYandexConfig::Section* section) {
    try {
        const TYandexConfig::Directives& directives = section->GetDirectives();
        const TYandexConfig::TSectionsMap serverSections = section->GetAllChildren();

        TransactionNamesEnabled = directives.Value("TransactionNamesEnabled", TransactionNamesEnabled);
        YangClientType = directives.Value("YangClientType", YangClientType);

        {
            auto p = serverSections.find("Chat");
            if (p != serverSections.end()) {
                ChatConfig = MakeHolder<NDrive::NChat::TEngineConfig>();
                ChatConfig->Init(p->second);
            }
        }

        {
            auto itFuelingManager = serverSections.find("FuelingManager");
            if (itFuelingManager != serverSections.end()) {
                FuelingManagerConfig.Reset(new TFuelingManagerConfig);
                FuelingManagerConfig->Init(itFuelingManager->second);
            }
        }

        {
            auto itDriveAPI = serverSections.find("DriveAPI");
            if (itDriveAPI != serverSections.end()) {
                DriveAPIConfig.Reset(new TDriveAPIConfig);
                DriveAPIConfig->Init(itDriveAPI->second, &RequestPolicy);
            }
        }

        {
            auto it = serverSections.find("WeatherAPI");
            if (it != serverSections.end()) {
                WeatherAPIConfig.Reset(new TWeatherAPIConfig);
                WeatherAPIConfig->Init(it->second);
            }
        }

        {
            auto it = serverSections.find("SurgeConstructor");
            if (it != serverSections.end()) {
                SurgeConstructorConfig.Reset(new TSurgeConstructorConfig);
                SurgeConstructorConfig->Init(it->second);
            }
        }

        {
            auto it = serverSections.find("UsersController");
            if (it != serverSections.end()) {
                UsersControllerConfig.Reset(new TUsersControllerConfig);
                UsersControllerConfig->Init(it->second);
            }
        }

        {
            auto p = serverSections.find("TelematicsApi");
            if (p != serverSections.end()) {
                TelematicsApiConfig = MakeHolder<TTaskExecutorConfig>();
                TelematicsApiConfig->Init(p->second);
            }
        }

        {
            auto p = serverSections.find("TelematicsClient");
            if (p != serverSections.end()) {
                TelematicsClientConfig = MakeHolder<TTelematicsClientConfig>();
                TelematicsClientConfig->Init(*p->second);
                AssertCorrectConfig(
                    !TelematicsClientConfig->GetSelfClientId() || GetTvmConfigs().contains(TelematicsClientConfig->GetSelfClientId()),
                    "SelfClientId %d is not registered", TelematicsClientConfig->GetSelfClientId()
                );
            }
        }

        {
            auto p = serverSections.find("SensorApi");
            if (p != serverSections.end()) {
                SensorApiConfig = MakeHolder<TSensorApiConfig>();
                SensorApiConfig->Init(p->second);
            }
        }

        {
            auto p = serverSections.find("SensorHistory");
            if (p != serverSections.end()) {
                const auto& d = p->second->GetDirectives();
                SensorHistoryConfig = MakeHolder<TSensorHistoryConfig>();
                ::Init(*SensorHistoryConfig, d);
            }
        }

        {
            auto p = serverSections.find("RidesInfo");
            if (p != serverSections.end()) {
                const auto& d = p->second->GetDirectives();
                RidesInfoConfig = MakeHolder<TRidesInfoConfig>();
                ::Init(*RidesInfoConfig, d);
            }
        }

        {
            auto p = serverSections.find("TrackClient");
            if (p != serverSections.end()) {
                const auto& d = p->second->GetDirectives();
                TrackClientConfig = MakeHolder<TTrackClientConfig>();
                ::Init(*TrackClientConfig, d);
            }
        }

        {
            auto p = serverSections.find("UserDevicesManager");
            if (p != serverSections.end()) {
                const TString type = p->second->GetDirectives().Value<TString>("Type", "");
                if (type == "passport") {
                    UserDevicesManagerConfig = MakeHolder<TPassportUserDevicesManagerConfig>();
                } else if (type == "fake") {
                    UserDevicesManagerConfig = MakeHolder<TFakeUserDevicesManagerConfig>();
                } else {
                    AssertCorrectConfig(false, "Incorrect UserDevicesManager.Type");
                }
                UserDevicesManagerConfig->Init(p->second);
            }
        }

        {
            auto p = serverSections.find("ModelsStorage");
            if (p != serverSections.end()) {
                const auto& d = p->second->GetDirectives();
                ModelsStorageConfig = MakeHolder<TModelsStorageConfig>();
                ModelsStorageConfig->Name = d.Value("Name", ModelsStorageConfig->Name);
            }
        }

        {
            auto it = serverSections.find("ChatRobots");
            if (it != serverSections.end()) {
                ChatRobotsConfig.Reset(new TChatRobotsConfig);
                ChatRobotsConfig->Init(it->second);
            }
        }

        {
            auto it = serverSections.find("UserRegistrationManager");
            if (it != serverSections.end()) {
                UserRegistrationManagerConfig.Reset(new TUserRegistrationManagerConfig);
                UserRegistrationManagerConfig->Init(it->second);
            }
        }

        {
            auto it = serverSections.find("PromoCodesManager");
            if (it != serverSections.end()) {
                PromoCodesManagerConfig.Reset(new TPromoCodesManagerConfig);
                PromoCodesManagerConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("UserEventsApi");
            if (it != serverSections.end()) {
                UserEventsApiConfig.Reset(new NDrive::TUserEventsApiConfig);
                UserEventsApiConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("FeaturesClient");
            if (it != serverSections.end()) {
                FeaturesClientConfig.Reset(new NDrive::TUserEventsApiConfig);
                FeaturesClientConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("ParkingZonesManager");
            if (it != serverSections.end()) {
                ParkingZonesManagerConfig = MakeHolder<NDrive::TParkingZonesManagerConfig>();
                ParkingZonesManagerConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("SupportCenterManager");
            if (it != serverSections.end()) {
                SupportCenterManagerConfig.Reset(new TSupportCenterManagerConfig);
                SupportCenterManagerConfig->Init(it->second, &RequestPolicy);
            }
        }
        {
            auto it = serverSections.find("SocialAPIClient");
            if (it != serverSections.end()) {
                SocialAPIClientConfig.Reset(new TSocialAPIClientConfig);
                SocialAPIClientConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("TaxiRouteHistory");
            if (it != serverSections.end()) {
                const auto& d = it->second->GetDirectives();
                TaxiRouteHistoryClientConfig = MakeHolder<TTaxiRouteHistoryConfig>();
                TaxiRouteHistoryClientConfig->Endpoint = d.Value("Endpoint", TaxiRouteHistoryClientConfig->Endpoint);
                TaxiRouteHistoryClientConfig->DestinationClientId = d.Value("DestinationClientId", TaxiRouteHistoryClientConfig->DestinationClientId);
                TaxiRouteHistoryClientConfig->SelfClientId = d.Value("SelfClientId", TaxiRouteHistoryClientConfig->SelfClientId);
            }
        }
        {
            auto it = serverSections.find("TaxiSuggest");
            if (it != serverSections.end()) {
                const auto& d = it->second->GetDirectives();
                TaxiSuggestConfig = MakeHolder<TTaxiSuggestConfig>();
                TaxiSuggestConfig->SelfClientId = d.Value("SelfClientId", TaxiSuggestConfig->SelfClientId);
            }
        }
        {
            auto it = serverSections.find("TaxiSupportClassifier");
            if (it != serverSections.end()) {
                const auto& d = it->second->GetDirectives();
                TaxiSupportClassifierConfig = MakeHolder<TTaxiSupportClassifierConfig>();
                TaxiSupportClassifierConfig->SelfClientId = d.Value("SelfClientId", TaxiSupportClassifierConfig->SelfClientId);
            }
        }
        {
            auto it = serverSections.find("GeoFeatures");
            if (it != serverSections.end()) {
                const auto& d = it->second->GetDirectives();
                GeoFeaturesConfig = MakeHolder<TGeoFeaturesConfig>();
                GeoFeaturesConfig->Name = d.Value("Name", GeoFeaturesConfig->Name);
            }
        }
        {
            auto it = serverSections.find("ScootersSurge");
            if (it != serverSections.end()) {
                ScootersSurgeConfig = MakeHolder<NDrive::TScootersSurgeConfig>();
                ScootersSurgeConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("AccountEmailBinder");
            if (it != serverSections.end()) {
                AccountEmailBinderConfig.Reset(new TAccountEmailBinderConfig);
                AccountEmailBinderConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("SelfHttpRequester");
            if (it != serverSections.end()) {
                SelfHttpRequesterConfig.Reset(new TSelfHttpRequesterConfig);
                SelfHttpRequesterConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("DocumentsManager");
            if (it != serverSections.end()) {
                DocumentsManagerConfig.Reset(new TDocumentsManagerConfig);
                DocumentsManagerConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("TagsManager");
            if (it != serverSections.end()) {
                TagsManagerConfig.Reset(new TTagsManagerConfig);
                TagsManagerConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("CovidPassMoscow");
            if (it != serverSections.end()) {
                CovidPassMoscowConfig.Reset(new TCovidPassMoscowConfig);
                CovidPassMoscowConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("CovidPassMoscowRegion");
            if (it != serverSections.end()) {
                CovidPassMoscowRegionConfig.Reset(new TCovidPassMoscowRegionConfig);
                CovidPassMoscowRegionConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("CovidPassGosuslugi");
            if (it != serverSections.end()) {
                CovidPassGosuslugiConfig.Reset(new TCovidPassGosuslugiConfig);
                CovidPassGosuslugiConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("ClckClient");
            if (it != serverSections.end()) {
                ClckClientConfig.Reset(new TClckClientConfig());
                ClckClientConfig->Init(it->second, &RequestPolicy);
            }
        }
        {
            auto it = serverSections.find("TaxiChatClient");
            if (it != serverSections.end()) {
                TaxiChatClientConfig.Reset(new TTaxiChatClientConfig());
                TaxiChatClientConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("TaxiRouteInfoClient");
            if (it != serverSections.end()) {
                TaxiRouteInfoClientConfig = MakeHolder<TTaxiModuleConfig>();
                TaxiRouteInfoClientConfig->Init(*it->second);
            }
        }
        {
            auto it = serverSections.find("Geobase");
            if (it != serverSections.end()) {
                const auto& sectionDirectives = it->second->GetDirectives();
                GeobaseConfig = MakeHolder<TGeobaseConfig>();
                GeobaseConfig->Path = sectionDirectives.Value("Path", GeobaseConfig->Path);
            }
        }
        {
            auto it = serverSections.find("BalanceClient");
            if (it != serverSections.end()) {
                BalanceClientConfig = MakeHolder<TBalanceClientConfig>();
                BalanceClientConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("YaDocClient");
            if (it != serverSections.end()) {
                YaDocClientConfig = MakeHolder<TYaDocClientConfig>();
                YaDocClientConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("DistributingBlockEventsStorage");
            if (it != serverSections.end()) {
                DistributingBlockEventsStorageConfig = IDistributingBlockEventsStorageConfig::ConstructConfig(*it->second);
            }
        }
        {
            auto it = serverSections.find("TaxiSupportChatSuggestClient");
            if (it != serverSections.end()) {
                TaxiSupportChatSuggestConfig = MakeHolder<TTaxiSupportChatSuggestConfig>();
                TaxiSupportChatSuggestConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("SendrClient");
            if (it != serverSections.end()) {
                SendrConfig = MakeHolder<TSendrConfig>();
                SendrConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("ClearWebClient");
            if (it != serverSections.end()) {
                ClearWebClientConfig = MakeHolder<TClearWebClientConfig>();
                ClearWebClientConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("TaxiSurgeCalculator");
            if (it != serverSections.end()) {
                TaxiSurgeCalculatorConfig = MakeHolder<NDrive::TTaxiSurgeCalculatorConfig>();
                TaxiSurgeCalculatorConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("SaturnClient");
            if (it != serverSections.end()) {
                SaturnClientConfig = MakeHolder<NDrive::TSaturnClientConfig>();
                SaturnClientConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("TaxiPromocodesClient");
            if (it != serverSections.end()) {
                TaxiPromocodesClientConfig = MakeHolder<NDrive::TTaxiPromocodesClientConfig>();
                TaxiPromocodesClientConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("TaxiAntifraudClient");
            if (it != serverSections.end()) {
                TaxiAntifraudConfig = MakeHolder<TTaxiAntifraudConfig>();
                TaxiAntifraudConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("TaxiDriverProfilesClient");
            if (it != serverSections.end()) {
                TaxiDriverProfilesConfig = MakeHolder<TTaxiDriverProfilesConfig>();
                TaxiDriverProfilesConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("TaxiDriverStatusClient");
            if (it != serverSections.end()) {
                TaxiDriverStatusConfig = MakeHolder<TTaxiDriverStatusConfig>();
                TaxiDriverStatusConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("TaxiSignalqDrivematicsApiClient");
            if (it != serverSections.end()) {
                TaxiSignalqDrivematicsApiConfig = MakeHolder<TTaxiSignalqDrivematicsApiConfig>();
                TaxiSignalqDrivematicsApiConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("TaxiFleetVehiclesClient");
            if (it != serverSections.end()) {
                TaxiFleetVehiclesConfig = MakeHolder<TTaxiFleetVehiclesConfig>();
                TaxiFleetVehiclesConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("LPMClient");
            if (it != serverSections.end()) {
                LPMClientConfig = MakeHolder<TLPMClientConfig>();
                LPMClientConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("UserDocumentsChecks");
            if (it != serverSections.end()) {
                UserDocumentsChecksConfig = MakeHolder<TUserDocumentsCheckConfig>();
                UserDocumentsChecksConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("RadarGeohash");
            if (it != serverSections.end()) {
                RadarGeohashConfig = MakeHolder<TDBEntitiesManagerConfig>();
                RadarGeohashConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("YDB");
            if (it != serverSections.end()) {
                YDBConfig = MakeHolder<TDBEntitiesManagerConfig>();
                YDBConfig->Init(it->second);
            }
        }
        if (auto it = serverSections.find("ParkingManager"); it != serverSections.end()) {
            ParkingConfig = MakeHolder<NDrive::TParkingAggregatorsManagerConfig>();
            ParkingConfig->Init(it->second);
        }
        {
            DevicesSnapshotManagerConfig = MakeHolder<NDevicesSnapshotManager::TOptions>();
            auto it = serverSections.find("DevicesSnapshotManager");
            if (it != serverSections.end()) {
                DevicesSnapshotManagerConfig->Init(*it->second);
            }
        }
        {
            TaxiTrackStoryConfig = MakeHolder<TTaxiTrackStoryConfig>();
            auto it = serverSections.find("TaxiTrackStoryConfig");
            if (it != serverSections.end()) {
                TaxiTrackStoryConfig->Init(*it->second);
            }
        }
        {
            auto configIt = serverSections.find("SupportAI");
            if (configIt != serverSections.end()) {
                SupportAIApiConfig.Reset(new NDrive::TSupportAIApiConfig);
                SupportAIApiConfig->Init(configIt->second);
            }
        }
        {
            auto configIt = serverSections.find("YangClient");
            if (configIt != serverSections.end()) {
                YangClientConfig =  MakeHolder<TYangClientConfig>();
                YangClientConfig->Init(configIt->second);
            }
        }
        {
            auto configIt = serverSections.find("ReninsKaskoConfig");
            if (configIt != serverSections.end()) {
                ReninsApiKaskoConfig.Reset(new NDrive::NRenins::TReninsConfig);
                ReninsApiKaskoConfig->Init(configIt->second);
            }
        }
        if (auto it = serverSections.find("ProcessorRewrites"); it != serverSections.end()) {
            for (auto&& i : it->second->GetAllChildren()) {
                // We do not allow any mistakes in naming of subsections.
                AssertCorrectConfig(i.first == "Rewrite", "'ProcessorRewrites' only supports 'Rewrite' sections");
                THttpRewriteConfig config;
                config.Init(i.second->GetDirectives());
                ProcessorRewritesConfig.push_back(std::move(config));
            }
        }
    } catch (...) {
        auto error = CurrentExceptionInfo(/*forceBacktrace=*/true).GetStringRobust();
        AbortFromCorruptedConfig("ServerConfig::Init: cannot parse config: %s", error.c_str());
    }
}

NDrive::TServerConfig::TServerConfig(const TServerConfigConstructorParams& params)
    : TServerBaseConfig(params)
{
    TAnyYandexConfig config;
    CHECK_WITH_LOG(config.ParseMemory(params.GetText().data()));
    const TYandexConfig::Section* root = config.GetRootSection();
    const TYandexConfig::TSectionsMap rootSections = root->GetAllChildren();
    auto itServer = rootSections.find("Server");
    AssertCorrectConfig(itServer != rootSections.end(), "No 'Server' section in config");
    Init(itServer->second);
}

NDrive::TServerConfig::~TServerConfig() {
}

void NDrive::TServerConfig::DoToString(IOutputStream& os) const {
    if (FuelingManagerConfig) {
        os << "<FuelingManager>" << Endl;
        FuelingManagerConfig->ToString(os);
        os << "</FuelingManager>" << Endl;
    }

    if (DriveAPIConfig) {
        os << "<DriveAPI>" << Endl;
        DriveAPIConfig->ToString(os);
        os << "</DriveAPI>" << Endl;
    }

    if (TelematicsApiConfig) {
        os << "<TelematicsApi>" << Endl;
        TelematicsApiConfig->ToString(os);
        os << "</TelematicsApi>" << Endl;
    }

    if (TelematicsClientConfig) {
        os << "<TelematicsClient>" << Endl;
        TelematicsClientConfig->ToString(os);
        os << "</TelematicsClient>" << Endl;
    }

    if (!!UsersControllerConfig) {
        os << "<UsersController>" << Endl;
        UsersControllerConfig->ToString(os);
        os << "</UsersController>" << Endl;
    }

    if (!!SurgeConstructorConfig) {
        os << "<SurgeConstructor>" << Endl;
        SurgeConstructorConfig->ToString(os);
        os << "</SurgeConstructor>" << Endl;
    }

    if (TaxiChatClientConfig) {
        os << "<TaxiChatClient>" << Endl;
        TaxiChatClientConfig->ToString(os);
        os << "</TaxiChatClient>" << Endl;
    }

    if (TaxiSignalqDrivematicsApiConfig) {
        os << "<TaxiSignalqDrivematicsApiClient>" << Endl;
        TaxiSignalqDrivematicsApiConfig->ToString(os);
        os << "</TaxiSignalqDrivematicsApiClient>" << Endl;
    }

    if (TaxiRouteInfoClientConfig) {
        os << "<TaxiRouteInfoClientConfig>" << Endl;
        TaxiRouteInfoClientConfig->ToString(os);
        os << "</TaxiRouteInfoClientConfig>" << Endl;
    }

    if (WeatherAPIConfig) {
        os << "<WeatherAPI>" << Endl;
        WeatherAPIConfig->ToString(os);
        os << "</WeatherAPI>" << Endl;
    }

    if (UserDevicesManagerConfig) {
        os << "<UserDevicesManager>" << Endl;
        UserDevicesManagerConfig->ToString(os);
        os << "</UserDevicesManager>" << Endl;
    }

    if (ModelsStorageConfig) {
        os << "<ModelsStorage>" << Endl;
        os << "Name: " << ModelsStorageConfig->Name << Endl;
        os << "</ModelsStorage>" << Endl;
    }

    if (SensorApiConfig) {
        os << "<SensorApi>" << Endl;
        SensorApiConfig->ToString(os);
        os << "</SensorApi>" << Endl;
    }

    if (SensorHistoryConfig) {
        os << "<SensorHistory>" << Endl;
        ::Print(*SensorHistoryConfig, os);
        os << "</SensorHistory>" << Endl;
    }

    if (RidesInfoConfig) {
        os << "<RidesInfo>" << Endl;
        ::Print(*RidesInfoConfig, os);
        os << "</RidesInfo>" << Endl;
    }

    if (TrackClientConfig) {
        os << "<TrackClient>" << Endl;
        ::Print(*TrackClientConfig, os);
        os << "</TrackClient>" << Endl;
    }

    if (ChatRobotsConfig) {
        os << "<ChatRobots>" << Endl;
        ChatRobotsConfig->ToString(os);
        os << "</ChatRobots>" << Endl;
    }

    if (UserRegistrationManagerConfig) {
        os << "<UserRegistrationManager>" << Endl;
        UserRegistrationManagerConfig->ToString(os);
        os << "</UserRegistrationManager>" << Endl;
    }

    if (PromoCodesManagerConfig) {
        os << "<PromoCodesManager>" << Endl;
        PromoCodesManagerConfig->ToString(os);
        os << "</PromoCodesManager>" << Endl;
    }

    if (UserEventsApiConfig) {
        os << "<UserEventsApi>" << Endl;
        UserEventsApiConfig->ToString(os);
        os << "</UserEventsApi>" << Endl;
    }

    if (FeaturesClientConfig) {
        os << "<FeaturesClient>" << Endl;
        FeaturesClientConfig->ToString(os);
        os << "</FeaturesClient>" << Endl;
    }

    if (ScootersSurgeConfig) {
        os << "<ScootersSurge>" << Endl;
        ScootersSurgeConfig->ToString(os);
        os << "</ScootersSurge>" << Endl;
    }

    if (GeoFeaturesConfig) {
        os << "<GeoFeaturesConfig>" << Endl;
        os << "Name: " << GeoFeaturesConfig->Name << Endl;
        os << "</GeoFeaturesConfig>" << Endl;
    }

    if (AccountEmailBinderConfig) {
        os << "<AccountEmailBinder>" << Endl;
        AccountEmailBinderConfig->ToString(os);
        os << "</AccountEmailBinder>" << Endl;
    }

    if (SelfHttpRequesterConfig) {
        os << "<SelfHttpRequester>" << Endl;
        SelfHttpRequesterConfig->ToString(os);
        os << "</SelfHttpRequester>" << Endl;
    }

    if (ParkingZonesManagerConfig) {
        os << "<ParkingZonesManager>" << Endl;
        ParkingZonesManagerConfig->ToString(os);
        os << "</ParkingZonesManager>" << Endl;
    }

    if (DocumentsManagerConfig) {
        os << "<DocumentsManager>" << Endl;
        DocumentsManagerConfig->ToString(os);
        os << "</DocumentsManager>" << Endl;
    }

    if (CovidPassMoscowConfig) {
        os << "<CovidPassMoscow>" << Endl;
        CovidPassMoscowConfig->ToString(os);
        os << "</CovidPassMoscow>" << Endl;
    }

    if (CovidPassMoscowRegionConfig) {
        os << "<CovidPassMoscowRegion>" << Endl;
        CovidPassMoscowRegionConfig->ToString(os);
        os << "</CovidPassMoscowRegion>" << Endl;
    }

    if (CovidPassGosuslugiConfig) {
        os << "<CovidPassGosuslugi>" << Endl;
        CovidPassGosuslugiConfig->ToString(os);
        os << "</CovidPassGosuslugi>" << Endl;
    }
    if (ClckClientConfig) {
        os << "<ClckClient>" << Endl;
        ClckClientConfig->ToString(os);
        os << "</ClckClient>" << Endl;
    }

    if (GeobaseConfig) {
        os << "<Geobase>" << Endl;
        os << "Path: " << GeobaseConfig->Path << Endl;
        os << "</Geobase>" << Endl;
    }
    if (BalanceClientConfig) {
        os << "<BalanceClient>" << Endl;
        BalanceClientConfig->ToString(os);
        os << "</BalanceClient>" << Endl;
    }
    if (TaxiSurgeCalculatorConfig) {
        os << "<TaxiSurgeCalculator>" << Endl;
        TaxiSurgeCalculatorConfig->ToString(os);
        os << "</TaxiSurgeCalculator>" << Endl;
    }
    if (SaturnClientConfig) {
        os << "<SaturnClient>" << Endl;
        SaturnClientConfig->ToString(os);
        os << "</SaturnClient>" << Endl;
    }
    if (TaxiPromocodesClientConfig) {
        os << "<TaxiPromocodesClient>" << Endl;
        TaxiPromocodesClientConfig->ToString(os);
        os << "</TaxiPromocodesClient>" << Endl;
    }
    if (TaxiAntifraudConfig) {
        os << "<TaxiAntifraudClient>" << Endl;
        TaxiAntifraudConfig->ToString(os);
        os << "</TaxiAntifraudClient>" << Endl;
    }
    if (LPMClientConfig) {
        os << "<LPMClient>" << Endl;
        LPMClientConfig->ToString(os);
        os << "</LPMClient>" << Endl;
    }
    if (UserDocumentsChecksConfig) {
        os << "<UserDocumentsChecks>" << Endl;
        UserDocumentsChecksConfig->ToString(os);
        os << "</UserDocumentsChecks>" << Endl;
    }
    if (RadarGeohashConfig) {
        os << "<RadarGeohash>" << Endl;
        RadarGeohashConfig->ToString(os);
        os << "</RadarGeohash>" << Endl;
    }
    if (YDBConfig) {
        os << "<YDB>" << Endl;
        YDBConfig->ToString(os);
        os << "</YDB>" << Endl;
    }
    if (ParkingConfig) {
        os << "<ParkingManager>" << Endl;
        ParkingConfig->ToString(os);
        os << "</ParkingManager>" << Endl;
    }
    if (DevicesSnapshotManagerConfig) {
        os << "<DevicesSnapshotManager>" << Endl;
        DevicesSnapshotManagerConfig->ToString(os);
        os << "</DevicesSnapshotManager>" << Endl;
    }
    if (YangClientConfig) {
        os << "<YangClient>" << Endl;
        YangClientConfig->ToString(os);
        os << "</YangClient>" << Endl;
    }
    os << "YangClientType: " << YangClientType << Endl;
    if (ReninsApiKaskoConfig) {
        os << "<ReninsKaskoConfig>" << Endl;
        ReninsApiKaskoConfig->ToString(os);
        os << "</ReninsKaskoConfig>" << Endl;
    }
    os << "TransactionNamesEnabled: " << TransactionNamesEnabled << Endl;
    if (!ProcessorRewritesConfig.empty()) {
        os << "<ProcessorRewrites>" << Endl;
        for (auto&& rewrite : ProcessorRewritesConfig) {
            os << "<Rewrite>" << Endl;
            rewrite.ToString(os);
            os << "</Rewrite>" << Endl;
        }
        os << "</ProcessorRewrites>" << Endl;
    }
}

const NDrive::TUserEventsApiConfig* NDrive::TServerConfig::GetFeaturesClientConfig() const {
    return FeaturesClientConfig.Get();
}

const NDrive::TScootersSurgeConfig* NDrive::TServerConfig::GetScootersSurgeConfig() const {
    return ScootersSurgeConfig.Get();
}

const NDrive::TTaxiSurgeCalculatorConfig* NDrive::TServerConfig::GetTaxiSurgeCalculatorConfig() const {
    return TaxiSurgeCalculatorConfig.Get();
}

const NDrive::TSaturnClientConfig* NDrive::TServerConfig::GetSaturnClientConfig() const {
    return SaturnClientConfig.Get();
}

const TDBEntitiesManagerConfig* NDrive::TServerConfig::GetYDBConfig() const {
    return YDBConfig.Get();
}

const NDrive::TParkingAggregatorsManagerConfig* NDrive::TServerConfig::GetParkingConfig() const {
    return ParkingConfig.Get();
}

const NDevicesSnapshotManager::TOptions* NDrive::TServerConfig::GetDevicesSnapshotManagerConfig() const {
    return DevicesSnapshotManagerConfig.Get();
}

const TVector<NDrive::THttpRewriteConfig>& NDrive::TServerConfig::GetProcessorRewritesConfig() const {
    return ProcessorRewritesConfig;
}

const NDrive::TTaxiPromocodesClientConfig* NDrive::TServerConfig::GetTaxiPromocodesClientConfig() const {
    return TaxiPromocodesClientConfig.Get();
}
