#include "server.h"

#include <drive/backend/areas/areas.h>
#include <drive/backend/billing/redis_proxy_cache.h>
#include <drive/backend/billing/trust_cache.h>
#include <drive/backend/chat/engine.h>
#include <drive/backend/chat_robots/channel/bot.h>
#include <drive/backend/chat_robots/registration/bot.h>
#include <drive/backend/chat_robots/support/bot.h>
#include <drive/backend/database/config.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/distributing_block_storage/common.h>
#include <drive/backend/localization/localization.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/models/storage.h>
#include <drive/backend/notifications/native_chat/chat.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/promo_codes/manager.h>
#include <drive/backend/settings/settings.h>
#include <drive/backend/tracks/client.h>
#include <drive/backend/users_controller/ucontroller.h>

#include <drive/library/cpp/geofeatures/client.h>
#include <drive/library/cpp/maps_router/linker.h>
#include <drive/library/cpp/maps_router/router.h>
#include <drive/library/cpp/taxi/request.h>
#include <drive/library/cpp/taxi/routehistory/client.h>
#include <drive/library/cpp/taxi/signalq_drivematics_api/client.h>
#include <drive/library/cpp/taxi/suggest/client.h>
#include <drive/library/cpp/taxi/support_classifier/client.h>
#include <drive/library/cpp/taxi/scooters_surge/client.h>
#include <drive/library/cpp/user_events_api/client.h>
#include <drive/telematics/api/sensor/local.h>
#include <drive/telematics/api/sensor/history.h>
#include <drive/telematics/api/server/config.h>
#include <drive/telematics/api/track/client.h>

#include <kernel/daemon/base_controller.h>
#include <kernel/daemon/common/time_guard.h>

#include <library/cpp/geobase/lookup.hpp>

#include <rtline/library/storage/redis/abstract.h>

#include <util/digest/fnv.h>
#include <util/string/hex.h>
#include <util/system/env.h>

namespace {
    class TDropCacheCommand: public NController::TController::TCommandProcessor {
    private:
        static NController::TController::TCommandProcessor::TFactory::TRegistrator<TDropCacheCommand> Registrator;
    protected:
        TString GetName() const override {
            return GetTypeName();
        }

        void DoProcess() override {
            TSet<TString> entities;
            StringSplitter(GetCgi().Get("entities")).SplitBySet(", ").SkipEmpty().Collect(&entities);
            SendGlobalMessage<NDrive::TCacheRefreshMessage>(entities);
            Write("result", "OK");
        }
    public:
        static TString GetTypeName() {
            return "drop_cache";
        }
    };
    NController::TController::TCommandProcessor::TFactory::TRegistrator<TDropCacheCommand> TDropCacheCommand::Registrator(TDropCacheCommand::GetTypeName());
}

using TDurationGuard = TTimeGuardImpl<false, TLOG_INFO, true>;

NDrive::TServer::TServer(const TConfig& config)
    : TServerBase(config)
    , Config(config)
{
}

NDrive::TServer::~TServer() {
}

bool NDrive::TServer::Process(IMessage* message) {
    if (TServerBase::Process(message)) {
        return true;
    }
    TSessionCorruptedGlobalMessage* mess = dynamic_cast<TSessionCorruptedGlobalMessage*>(message);
    if (mess) {
        mess->SetNeedAlert(NDrive::HasServer());
        return true;
    }
    return false;
}

void NDrive::TServer::DoRun() {
    NDrive::DeregisterServer();
    AssertCorrectConfig(Config.GetDriveAPIConfig(), "No DriveAPI configuration");

    if (Config.IsTransactionNamesEnabled()) {
        TDatabaseSessionConstructor::EnableTransactionNames();
    }

    if (Config.GetChatConfig()) {
        auto db = GetDatabase(Config.GetChatConfig()->GetDBName());
        AssertCorrectConfig(!!db, "no such database: " + Config.GetChatConfig()->GetDBName());
        ChatHistoryContext.Reset(new THistoryContext(db));
    }

    {
        TDurationGuard dg("ConstructComponent TagsManager");

        const TTagsManagerConfig* tagsManagerConfig = nullptr;
        TString dbName;
        if (Config.GetTagsManagerConfig()) {
            tagsManagerConfig = Config.GetTagsManagerConfig();
            dbName = tagsManagerConfig->GetTagsDBName();
        } else if (Config.GetDriveAPIConfig()) {
            tagsManagerConfig = Config.GetDriveAPIConfig();
            dbName = !!tagsManagerConfig->GetTagsDBName() ? tagsManagerConfig->GetTagsDBName() : Config.GetDriveAPIConfig()->GetDBName();
        }

        if (tagsManagerConfig) {
            auto database = GetDatabase(dbName);
            AssertCorrectConfig(!!database, "No such DataBase: " + dbName);
            TagsManager.Reset(new TDriveTagsManager(database, *tagsManagerConfig));
            Y_ENSURE_BT(TagsManager->Start());
        }
    }

    {
        auto threadCount = FromStringWithDefault<size_t>(GetEnv("SERVER_START_THREADS"), 4);
        auto threadPool = CreateThreadPool(threadCount);
        TVector<NThreading::TFuture<void>> futures;

        auto constructDriveApi = NThreading::Async([this] {
            TDurationGuard dg("ConstructComponent DriveApi");
            AssertCorrectConfig(!!TagsManager, "Creating DriveAPI without tags manager");
            DriveAPI.Destroy();
            DriveAPI.Reset(new TDriveAPI(*Config.GetDriveAPIConfig(), Databases, &GetSettings(), *this, TagsManager));
        }, *threadPool);
        auto startDriveApi = constructDriveApi.Apply([this](const NThreading::TFuture<void>& w) {
            w.GetValue();
            TDurationGuard dg("StartComponent DriveApi");
            Y_ENSURE(DriveAPI->Start());
        });
        futures.push_back(constructDriveApi);
        futures.push_back(startDriveApi);

        auto startChatEngine = NThreading::TFuture<void>();
        if (!!Config.GetChatConfig()) {
            startChatEngine = NThreading::Async([this] {
                TDurationGuard dg("ConstructComponent ChatEngine");

                ChatMetaManagerHistoryReader.Reset(new TCallbackSequentialTableImpl<TObjectEvent<NDrive::NChat::TChat>, TString>(*ChatHistoryContext, "chats_history", Config.GetChatConfig()->GetChatMetaManagerConfig()));
                Y_ENSURE(ChatMetaManagerHistoryReader->Start());
                ChatMetaManager.Reset(new NDrive::NChat::TChatsMetaManager(ChatMetaManagerHistoryReader, ChatHistoryContext->GetDatabase()));
                Y_ENSURE(ChatMetaManager->RebuildCache());

                ChatEngine = MakeHolder<NDrive::NChat::TEngine>(*ChatMetaManager.Get(), *Config.GetChatConfig(), *ChatHistoryContext);
                Y_ENSURE(ChatEngine->StartEngine());
            }, *threadPool);
            futures.push_back(startChatEngine);
        }
        if (Config.GetChatRobotsConfig()) {
            AssertCorrectConfig(startChatEngine.Initialized(), "you cannot configure chat robots without chat engine");
            auto startChatRobots = NThreading::WaitAll(constructDriveApi, startChatEngine).Apply([this](const NThreading::TFuture<void>& w) {
                w.GetValue();
                TDurationGuard dg("ConstructComponent ChatRobotsManager");
                AssertCorrectConfig(ChatEngine != nullptr, "you cannot configure chat robots without chat engine");
                ChatRobotsManager = MakeHolder<TChatRobotsManager>(this, *Config.GetChatRobotsConfig());
            });
            futures.push_back(startChatRobots);
        }

        if (!!Config.GetUserDevicesManagerConfig()) {
            futures.push_back(NThreading::Async([this]() {
                TDurationGuard dg("ConstructComponent UserDevicesManager");
                UserDevicesManager = Config.GetUserDevicesManagerConfig()->BuildManager(*this);
            }, *threadPool));
        }
        if (auto msc = Config.GetModelsStorageConfig(); msc && msc->Name) {
            futures.push_back(NThreading::Async([this, msc]() {
                TDurationGuard dg("ConstructComponent ModelsStorage");
                auto storage = GetVersionedStorage(msc->Name);
                AssertCorrectConfig(!!storage, "cannot find storage %s", msc->Name.c_str());
                ModelsStorage = MakeHolder<NDrive::TModelsStorage>(storage);
            }, *threadPool));
        }

        NThreading::WaitAll(futures).GetValueSync();
    }

    if (!!Config.GetPromoCodesManagerConfig()) {
        TDurationGuard dg("ConstructComponent PromoCodesManager");
        auto it = Databases.find(Config.GetPromoCodesManagerConfig()->GetDBName());
        AssertCorrectConfig(it != Databases.end(), "Incorrect DBName for promo codes manager");
        PromoHistoryContext.Reset(new THistoryContext(it->second));
        PromoCodesManager.Reset(new TPromoCodesManager(*PromoHistoryContext, *Config.GetPromoCodesManagerConfig(), *this));
    }

    if (!!Config.GetFuelingManagerConfig()) {
        TDurationGuard dg("ConstructComponent FuelingManager");
        FuelingManager.Reset(new TFuelingManager(*Config.GetFuelingManagerConfig(), this));
        Y_ENSURE_BT(FuelingManager->Start());
    }

    if (Config.GetTelematicsApiConfig()) {
        TDurationGuard dg("ConstructComponent TelematicsApi");
        TelematicsApi = MakeHolder<NDrive::TTelematicsApi>(*Config.GetTelematicsApiConfig(), this);
    }

    if (Config.GetTelematicsClientConfig()) {
        TDurationGuard dg("ConstructComponent TelematicsClient");
        auto config = Config.GetTelematicsClientConfig();
        auto tvm = GetTvmClient(config->GetSelfClientId());
        AssertCorrectConfig(!config->GetSelfClientId() || !!tvm, "SelfClientId %d cannot be acquired", config->GetSelfClientId());

        NDrive::TTelematicsClient::TOptions options;
        options.DestinationClientId = config->GetDestinationClientId();
        options.InfoFetchHost = config->GetFetchHost();
        if (config->GetFetchPort()) {
            options.InfoFetchPort = config->GetFetchPort();
        }
        options.InfoFetchService = config->GetFetchService();
        options.InfoPushHost = config->GetPushHost();
        if (config->GetPushPort()) {
            options.InfoPushPort = config->GetPushPort();
        }
        options.InfoPushToken = config->GetPushToken();
        options.InfoSpMetaSearch = config->GetSpMetaSearch();
        options.InfoSaasTvmId = config->GetSaasTvmId();
        auto sdc = config->GetServiceDiscoveryConfig();
        auto sd = sdc ? sdc->Construct() : nullptr;
        if (!sd) {
            auto shards = config->GetShards();
            sd = MakeHolder<NDrive::TStaticServiceDiscovery>(std::move(shards));
        }
        TelematicsClient = MakeHolder<NDrive::TTelematicsClient>(std::move(sd), options, tvm);
    }

    if (Config.GetSensorApiConfig()) {
        TDurationGuard dg("ConstructComponent Sensors");
        const auto config = Config.GetSensorApiConfig();
        if (config->GetSensorApiName()) {
            const auto api = GetRTLineAPI(config->GetSensorApiName());
            AssertCorrectConfig(!!api, "%s is not found", config->GetSensorApiName().c_str());
            const auto& search = api->GetSearchClient();
            auto pusherType = config->GetPusherType();
            switch (pusherType) {
                case TServerBaseConfig::TSensorApiConfig::EPusherType::IndexingClient: {
                    const auto* indexing = api->GetIndexingClient();
                    AssertCorrectConfig(!!indexing, "IndexingClient is not configured for %s", config->GetSensorApiName().c_str());
                    SensorApi = MakeHolder<NDrive::TSensorApi>(config->GetSensorApiName(), search, *indexing);
                    break;
                }
                case TServerBaseConfig::TSensorApiConfig::EPusherType::CustomPusher: {
                    THolder<NDrive::IPusher> pusher = config->GetPusherOptions() ? config->GetPusherOptions()->BuildPusher(nullptr) : nullptr;
                    AssertCorrectConfig(pusher.Get(), "Pusher is not configured");
                    SensorApi = MakeHolder<NDrive::TSensorApi>(config->GetSensorApiName(), search, pusher.Release());
                    break;
                }
            }
            if (config->GetHistoryApiName()) {
                const auto historyApi = GetRTLineAPI(config->GetHistoryApiName());
                AssertCorrectConfig(!!historyApi, "%s is not found", config->GetHistoryApiName().c_str());
                SensorHistoryApi = MakeHolder<NDrive::TSensorHistoryApi>(historyApi->GetSearchClient(), config->GetHistoryApiTimeout());
            }
        } else {
            SensorApi = MakeHolder<TLocalSensorApi>();
        }
    }
    if (auto sensorHistoryConfig = Config.GetSensorHistoryConfig()) {
        SensorHistoryClient = MakeHolder<NDrive::TSensorHistoryClient>(*sensorHistoryConfig);
    }
    if (auto ridesInfoConfig = Config.GetRidesInfoConfig()) {
        RidesInfoClient = MakeHolder<NDrive::TClickHouseBackendClient>(*ridesInfoConfig);
    }
    if (auto trackClientConfig = Config.GetTrackClientConfig()) {
        auto client = MakeAtomicShared<NDrive::TClickHouseTrackClient>(*trackClientConfig);
        TrackClient = MakeHolder<TSessionTrackClient>(client, *this);
    }

    if (Config.GetDriveAPIConfig()->GetOffersStorageName() == "offers_manager") {
        OffersStorage = MakeHolder<TDBOffersStorage>(*this, Config.GetDriveAPIConfig()->GetDBName());
    } else if (Config.GetDriveAPIConfig()->GetOffersStorageName() == "") {
        OffersStorage = MakeHolder<TFakeOfferStorage>();
    } else {
        OffersStorage = MakeHolder<TRTYOfferStorage>(*this, Config.GetDriveAPIConfig()->GetOffersStorageName());
    }

    if (Config.GetWeatherAPIConfig()) {
        TDurationGuard dg("ConstructComponent Weather");
        WeatherAPI = MakeHolder<TWeatherAPI>(*Config.GetWeatherAPIConfig());
        Y_ENSURE_BT(WeatherAPI->Start());
    }

    if (!!Config.GetUsersControllerConfig()) {
        TDurationGuard dg("ConstructComponent UsersController");
        UsersController.Reset(new TUsersController(*Config.GetUsersControllerConfig(), this));
    }

    {
        TDurationGuard dg("ConstructComponent SnapshotsManager");
        Y_ENSURE(Config.GetDevicesSnapshotManagerConfig());
        SnapshotsManager = MakeHolder<TDevicesSnapshotManager>(*this, *Config.GetDevicesSnapshotManagerConfig());
        Y_ENSURE_BT(SnapshotsManager->Start());
    }

    if (!!Config.GetSurgeConstructorConfig()) {
        TDurationGuard dg("ConstructComponent SurgeConstructor");
        SurgeConstructor = new TRTFactorsConstructor(*Config.GetSurgeConstructorConfig(), this);
        Y_ENSURE_BT(SurgeConstructor->Start());
    }

    if (!!Config.GetUserRegistrationManagerConfig()) {
        TDurationGuard dg("ConstructComponent UserRegistrationManager");
        UserRegistrationManager = MakeHolder<TUserRegistrationManager>(*Config.GetUserRegistrationManagerConfig(), this);
    }

    if (Config.GetUserEventsApiConfig()) {
        const TString kvApiName = Config.GetUserEventsApiConfig()->GetKVRTLineApiName();
        const TRTLineAPI* kvConfig = GetRTLineAPI(kvApiName);
        AssertCorrectConfig(kvConfig, "Missing kv config for UserEventsApi: " + kvApiName);
        const TString historyApiName = Config.GetUserEventsApiConfig()->GetHistoryRTLineApiName();
        const TRTLineAPI* historyConfig = GetRTLineAPI(historyApiName);
        AssertCorrectConfig(historyConfig, "Missing history config for UserEventsApi: " + historyApiName);
        UserEventsApi.Reset(new NDrive::TUserEventsApi(*Config.GetUserEventsApiConfig(),
                                                   kvConfig->GetSearchClient(),
                                                   historyConfig->GetSearchClient()));
    }

    if (auto config = Config.GetFeaturesClientConfig()) {
        const TString kvApiName = config->GetKVRTLineApiName();
        const TRTLineAPI* kvConfig = GetRTLineAPI(kvApiName);
        AssertCorrectConfig(kvConfig, "Missing kv config for FeaturesClientConfig: " + kvApiName);
        const TString historyApiName = config->GetHistoryRTLineApiName();
        const TRTLineAPI* historyConfig = GetRTLineAPI(historyApiName);
        AssertCorrectConfig(historyConfig, "Missing history config for UserEventsApi: " + historyApiName);
        FeaturesClient = MakeHolder<NDrive::TUserEventsApi>(
            *config, kvConfig->GetSearchClient(), historyConfig->GetSearchClient()
        );
    }

    if (auto gbc = Config.GetGeobaseConfig()) {
        AssertCorrectConfig(!gbc->Path.empty(), "empty Geobase Path");
        Geobase = MakeHolder<NGeobase::TLookup>(gbc->Path);
    }

    if (auto cfg = Config.GetGeoFeaturesConfig()) {
        const auto api = GetRTLineAPI(cfg->Name);
        AssertCorrectConfig(!!api, "cannot find API %s", cfg->Name.c_str());
        RtmrClient = MakeHolder<NDrive::TRtmrSurgeClient>();
        Y_ENSURE_BT(RtmrClient);
        THolder<NDrive::TScootersSurge> scootersSurgeClient;
        if (auto scootersSurgeCfg = Config.GetScootersSurgeConfig()) {
            auto tvmClient = scootersSurgeCfg->SelfClientId ? GetTvmClient(scootersSurgeCfg->SelfClientId) : nullptr;
            auto tvmAuth = tvmClient ? MakeMaybe<NDrive::TTvmAuth>(tvmClient, scootersSurgeCfg->DestinationTvmId) : Nothing();
            scootersSurgeClient = MakeHolder<NDrive::TScootersSurge>(scootersSurgeCfg->Options, std::move(tvmAuth));
        }
        GeoFeaturesClient = MakeHolder<NDrive::TGeoFeaturesClient>(api->GetSearchClient(), *RtmrClient, std::move(scootersSurgeClient));
    }

    if (auto cfg = Config.GetParkingZonesManagerConfig()) {
        ParkingZonesManager = MakeHolder<NDrive::TParkingZonesManager>(*cfg, *this);
    }

    if (Config.GetSupportCenterManagerConfig()) {
        TDurationGuard dg("ConstructComponent SupportCenterManager");
        SupportCenterManager.Reset(new TSupportCenterManager(this, *Config.GetSupportCenterManagerConfig()));
        SupportCenterManager->Start();
    }

    if (Config.GetSocialAPIClientConfig()) {
        SocialAPIClient.Reset(new TSocialAPIClient(*Config.GetSocialAPIClientConfig(), GetTvmClient(Config.GetSocialAPIClientConfig()->GetSelfTvmId())));
    }

    if (auto trhc = Config.GetTaxiRouteHistoryClientConfig()) {
        auto tvm = GetTvmClient(trhc->SelfClientId);
        AssertCorrectConfig(!trhc->SelfClientId || !!tvm, "SelfClientId %d cannot be acquired", trhc->SelfClientId);
        NDrive::TTaxiRouteHistoryClient::TOptions options;
        if (trhc->Endpoint) {
            options.Endpoint = trhc->Endpoint;
        }
        if (trhc->DestinationClientId) {
            options.DestinationClientId = trhc->DestinationClientId;
        }
        TaxiRouteHistoryClient = MakeHolder<NDrive::TTaxiRouteHistoryClient>(options, tvm);
    }

    if (auto tsc = Config.GetTaxiSuggestConfig()) {
        auto tvm = GetTvmClient(tsc->SelfClientId);
        AssertCorrectConfig(!tsc->SelfClientId || !!tvm, "SelfClientId %d cannot be acquired", tsc->SelfClientId);
        NDrive::TTaxiSuggestClient::TOptions options;
        TaxiSuggestClient = MakeHolder<NDrive::TTaxiSuggestClient>(options, tvm);
    }

    if (auto tsc = Config.GetTaxiSupportClassifierConfig()) {
        auto tvm = GetTvmClient(tsc->SelfClientId);
        AssertCorrectConfig(!tsc->SelfClientId || !!tvm, "SelfClientId %d cannot be acquired", tsc->SelfClientId);
        NDrive::TTaxiSupportClassifierClient::TOptions options;
        TaxiSupportClassifierClient = MakeHolder<NDrive::TTaxiSupportClassifierClient>(options, tvm);
    }

    {
        TDurationGuard dg("ConstructComponent WaitSnapshotsManager");
        SnapshotsManager->Wait(Now() + Config.GetDevicesSnapshotWaitTime());
    }

    if (Config.GetAccountEmailBinderConfig()) {
        auto tvmId = Config.GetAccountEmailBinderConfig()->GetSelfTvmId();
        auto tvm = GetTvmClient(tvmId);
        AssertCorrectConfig(!tvmId || !!tvm, "SelfClientId %d cannot be acquired", tvmId);
        AccountEmailBinder.Reset(new TAccountEmailBinder(*Config.GetAccountEmailBinderConfig(), tvm));
    }

    if (Config.GetSelfHttpRequester()) {
        SelfHttpRequester.Reset(new TSelfHttpRequester(*Config.GetSelfHttpRequester()));
    }

    if (Config.GetDocumentsManagerConfig()) {
        auto it = Databases.find(Config.GetDocumentsManagerConfig()->GetDBName());
        AssertCorrectConfig(it != Databases.end(), "Incorrect DBName for documents manager");
        DocumentsManagerHistoryContext.Reset(new THistoryContext(it->second));
        DocumentsManager = MakeHolder<TDocumentsManager>(*DocumentsManagerHistoryContext, *Config.GetDocumentsManagerConfig(), *this);
    }

    if (Config.GetCovidPassMoscowConfig()) {
        CovidPassMoscowClient = MakeHolder<TCovidPassMoscowClient>(*Config.GetCovidPassMoscowConfig());
    }

    if (Config.GetCovidPassMoscowRegionConfig()) {
        CovidPassMoscowRegionClient = MakeHolder<TCovidPassMoscowRegionClient>(*Config.GetCovidPassMoscowRegionConfig());
    }

    if (Config.GetCovidPassGosuslugiConfig()) {
        CovidPassGosuslugiClient = MakeHolder<TCovidPassGosuslugiClient>(*Config.GetCovidPassGosuslugiConfig());
    }

    if (Config.GetClckClientConfig()) {
        ClckClient = MakeHolder<TClckClient>(*Config.GetClckClientConfig());
    }

    if (Config.GetTaxiChatClientConfig()) {
        TaxiChatClient = ITaxiChatClient::ConstructTaxiChatClient(*Config.GetTaxiChatClientConfig(), GetTvmClient(Config.GetTaxiChatClientConfig()->GetSelfTvmId()));
    }

    if (const auto& config = Config.GetTaxiRouteInfoClientConfig()) {
        TaxiRouteInfoClient = MakeHolder<TSimpleTaxiClient>(*config);
    }

    if (const auto config = Config.GetMapsLinkerConfig()) {
        THolder<NGraph::ILinker> relinker;
        const auto& relinkerName = config->GetRelinkerName();
        if (relinkerName) {
            auto relinkerApi = GetRTLineAPI(relinkerName);
            AssertCorrectConfig(!!relinkerApi, "cannot find API %s", relinkerName.c_str());
            relinker = MakeHolder<NGraph::TRouter>(relinkerApi->GetSearchClient());
        }

        const auto& endpoint = config->GetEndpoint();
        AssertCorrectConfig(!!endpoint, "MapsLinker endpoint is not set");
        MapsLinker = MakeAtomicShared<NDrive::TMapsLinker>(endpoint, std::move(relinker));
    }
    if (const auto& config = Config.GetMapsRouterConfig()) {
        auto selfClientId = config->GetSelfClientId();
        auto tvmClient = selfClientId ? GetTvmClient(config->GetSelfClientId()) : nullptr;
        auto tvmAuth = selfClientId ? MakeMaybe<NDrive::TTvmAuth>(tvmClient, config->GetDestinationTvmId()) : Nothing();
        if (config->GetVehicleType() == "scooter") {
            MapsRouter = MakeHolder<NDrive::TScooterRouter>(
                config->GetHostUrl(),
                std::move(tvmAuth),
                config->GetRequestTimeout()
            );
        } else {
            MapsRouter = MakeHolder<NDrive::TVehicleRouter>(
                config->GetHostUrl(),
                std::move(tvmAuth),
                config->GetRequestTimeout()
            );
        }
    }
    if (const auto& config = Config.GetPedestrianRouterConfig()) {
        auto selfClientId = config->GetSelfClientId();
        auto tvmClient = selfClientId ? GetTvmClient(config->GetSelfClientId()) : nullptr;
        auto tvmAuth = selfClientId ? MakeMaybe<NDrive::TTvmAuth>(tvmClient, config->GetDestinationTvmId()) : Nothing();
        PedestrianRouter = MakeHolder<NDrive::TPedestrianRouter>(
            config->GetHostUrl(), std::move(tvmAuth), config->GetRequestTimeout()
        );
    }
    if (const auto& config = Config.GetTaxiSurgeCalculatorConfig()) {
        auto tvmClient = config->SelfClientId ? GetTvmClient(config->SelfClientId) : nullptr;
        auto tvmAuth = tvmClient ? MakeMaybe<NDrive::TTvmAuth>(tvmClient, config->DestinationTvmId) : Nothing();
        TaxiSurgeCalculator = MakeHolder<NDrive::TTaxiSurgeCalculator>(config->Options, std::move(tvmAuth));
    }
    if (const auto& config = Config.GetSaturnClientConfig()) {
        auto tvmClient = config->SelfClientId ? GetTvmClient(config->SelfClientId) : nullptr;
        auto tvmAuth = tvmClient ? MakeMaybe<NDrive::TTvmAuth>(tvmClient, config->DestinationTvmId) : Nothing();
        SaturnClient = MakeHolder<NDrive::TSaturnClient>(config->Options, std::move(tvmAuth));
    }

    if (Config.GetDistributingBlockEventsStorageConfig()) {
        const auto* config = Config.GetDistributingBlockEventsStorageConfig();
        DistributingBlockEventsStorage = config->BuildStorage(*this);
    }

    if (Config.GetBalanceClientConfig()) {
        BalanceClient = MakeHolder<TBalanceClient>(*Config.GetBalanceClientConfig(), GetTvmClient(Config.GetBalanceClientConfig()->GetSelfTvmId()));
    }

    if (Config.GetYaDocClientConfig()) {
        YaDocClient = MakeHolder<TYaDocClient>(*Config.GetYaDocClientConfig(), GetTvmClient(Config.GetYaDocClientConfig()->GetSelfTvmId()));
    }

    if (auto config = Config.GetTaxiSupportChatSuggestConfig()) {
        auto tvm = GetTvmClient(config->GetSelfTvmId());
        AssertCorrectConfig(!config->GetSelfTvmId() || !!tvm, "SelfClientId %d cannot be acquired", config->GetSelfTvmId());
        TaxiSupportChatSuggestClient = MakeHolder<TTaxiSupportChatSuggestClient>(*config, tvm);
    }

    if (const auto& config = Config.GetSendrConfig()) {
        SendrClient = MakeHolder<TSendrClient>(*config);
    }

    if (auto config = Config.GetClearWebClientConfig()) {
        ClearWebClient = MakeHolder<TClearWebClient>(*config);
    }

    if (auto config = Config.GetTaxiAntifraudConfig()) {
        auto tvm = GetTvmClient(config->GetSelfTvmId());
        AssertCorrectConfig(!config->GetSelfTvmId() || !!tvm, "SelfClientId %d cannot be acquired", config->GetSelfTvmId());
        TaxiAntidraudClient = MakeHolder<TTaxiAntifraudClient>(*config, tvm);
    }

    if (auto config = Config.GetTaxiDriverProfilesConfig()) {
        auto tvm = GetTvmClient(config->GetSelfTvmId());
        AssertCorrectConfig(!config->GetSelfTvmId() || !!tvm, "SelfClientId %d cannot be acquired", config->GetSelfTvmId());
        TaxiDriverProfilesClient = MakeHolder<TTaxiDriverProfilesClient>(*config, tvm);
    }

    if (auto config = Config.GetTaxiDriverStatusConfig()) {
        auto tvm = GetTvmClient(config->GetSelfTvmId());
        AssertCorrectConfig(!config->GetSelfTvmId() || !!tvm, "SelfClientId %d cannot be acquired", config->GetSelfTvmId());
        TaxiDriverStatusClient = MakeHolder<TTaxiDriverStatusClient>(*config, tvm);
    }

    if (auto config = Config.GetTaxiSignalqDrivematicsApiConfig()) {
        TaxiSignalqDrivematicsApiClient = ITaxiSignalqDrivematicsApiClient::ConstructTaxiSignalqDrivematicsApiClient(*config, GetTvmClient(config->GetSelfTvmId()));
    }

    if (auto config = Config.GetTaxiFleetVehiclesConfig()) {
        auto tvm = GetTvmClient(config->GetSelfTvmId());
        AssertCorrectConfig(!config->GetSelfTvmId() || !!tvm, "SelfClientId %d cannot be acquired", config->GetSelfTvmId());
        TaxiFleetVehiclesClient = MakeHolder<TTaxiFleetVehiclesClient>(*config, tvm);
    }

    if (auto config = Config.GetUserDocumentsCheckConfig()) {
        auto it = Databases.find(config->GetDBName());
        AssertCorrectConfig(it != Databases.end(), "Incorrect DBName for documents checks manager");
        UserDocumentsChecksContext = MakeHolder<THistoryContext>(it->second);
        UserDocumentsChecksManager = MakeHolder<TUserDocumentsChecksManager>(*UserDocumentsChecksContext, *config);
    }

    if (auto config = Config.GetRadarGeohashConfig()) {
        auto it = Databases.find(config->GetDBName());
        AssertCorrectConfig(it != Databases.end(), "Incorrect DBName for radar geohash");
        RadarGeohashContext = MakeHolder<THistoryContext>(it->second);
        RadarGeohashManager = MakeHolder<TRadarGeohashManager>(*RadarGeohashContext, *config);
    }
    if (auto config = Config.GetYDBConfig()) {
        auto it = Databases.find(config->GetDBName());
        AssertCorrectConfig(it != Databases.end(), "Incorrect DBName for YDB");
        YDB = MakeHolder<TDatabaseSessionConstructor>(it->second);
    }
    if (auto config = Config.GetYangClientConfig()) {
        if (Config.GetYangClientType() == "fake") {
            YangClient = MakeHolder<TFakeYangClient>();
        } else {
            if (config->GetAuthType() == TYangClientConfig::EAuthType::OAuth) {
                YangClient = MakeHolder<TYangClient>(*config);
            } else if (config->GetAuthType() == TYangClientConfig::EAuthType::TVM) {
                auto tvm = GetTvmClient(config->GetSelfTvmId());
                AssertCorrectConfig(!config->GetSelfTvmId() || !!tvm, "SelfClientId %d cannot be acquired", config->GetSelfTvmId());
                YangClient = MakeHolder<TYangClient>(*config, tvm);
            }
        }
    }
    if (auto config = Config.GetLPMClientConfig()) {
        auto tvmId = config->GetClientConfig().GetSelfTvmId();
        auto tvm = GetTvmClient(tvmId);
        AssertCorrectConfig(!tvmId || !!tvm, "SelfClientId %d cannot be acquired", tvmId);
        auto storageOptions = ConstructCacheStorageOptions(*config);
        LPMClient = config->Construct(std::move(storageOptions), tvm);
    }
    if (auto config = Config.GetParkingConfig()) {
        ParkingManager = MakeHolder<NDrive::TParkingAggregatorsManager>(*config);
    }
    if (auto config = Config.GetTaxiTrackStoryConfig()) {
        auto tvmId = config->GetSelfTvmId();
        auto tvm = GetTvmClient(tvmId);
        AssertCorrectConfig(!tvmId || !!tvm, "SelfClientId %d cannot be acquired", tvmId);
        TaxiTrackStoryClient = MakeHolder<TTaxiTrackStoryClient>(*config, tvm);
    }
    if (auto supportAI = Config.GetSupportAIApiConfig()) {
        TDurationGuard dg("ConstructComponent SupportAIClient");
        TAtomicSharedPtr<NTvmAuth::TTvmClient> tvm;
        if (supportAI->GetSelfClientId() && supportAI->GetDestinationTvmId()) {
            tvm = GetTvmClient(supportAI->GetSelfClientId());
        }
        SupportAIClient = MakeHolder<NDrive::TSupportAIClient>(*supportAI, tvm);
    }
    if (const auto& config = Config.GetReninsApiKaskoConfig()) {
        ReninsKaskoClient = MakeHolder<NDrive::NRenins::NKasko::TClient>(*config);
    }
    if (const auto& config = Config.GetProcessorRewritesConfig(); !config.empty()) {
        RewriteRouter = MakeHolder<NDrive::THttpRouter<THttpRewriteConfig>>();
        for (auto&& rewrite : config) {
            RewriteRouter->Add(rewrite.Pattern, rewrite.Method, rewrite);
        }
    }
    if (const auto& config = Config.GetTaxiPromocodesClientConfig()) {
        auto tvmClient = config->SelfClientId ? GetTvmClient(config->SelfClientId) : nullptr;
        auto tvmAuth = tvmClient ? MakeMaybe<NDrive::TTvmAuth>(tvmClient, config->DestinationTvmId) : Nothing();
        TaxiPromocodesClient = MakeHolder<NDrive::TTaxiPromocodesClient>(config->Options, std::move(tvmAuth));
    }
    NDrive::RegisterServer(*this);
}

TVector<THolder<NDrive::ITrustStorageOptions>> NDrive::TServer::ConstructCacheStorageOptions(const TLPMClientConfig& config) {
    TVector<THolder<NDrive::ITrustStorageOptions>> result;

    for (const auto& storageConfig: config.GetCacheStorageConfigs()) {
        if (auto redisPtr = dynamic_cast<NDriveRedis::NCache::TTrustRedisCacheConfig*>(storageConfig.Get())) {
            /* Construct redis storage */
            auto redisDB = Databases.find(redisPtr->GetExternalDBName());
            AssertCorrectConfig(redisDB != Databases.end(), "Incorrect DBName for Redis");
            NRTLine::TStorageOptions storageOps;
            auto storage = MakeAtomicShared<NDriveRedis::NKVAbstract::TRedisVersionedStorage>(storageOps, redisDB->second);
            result.push_back(MakeHolder<NDriveRedis::NCache::TTrustRedisStorageOptions>(storage));
        } else if (auto redisPtr = dynamic_cast<NDrive::NRedis::NCache::TTrustCacheConfig*>(storageConfig.Get())) {
            result.push_back(MakeHolder<NDrive::NRedis::NCache::TTrustStorageOptions>());
        } else if (auto rtyPtr = dynamic_cast<TRTYTrustStorageConfig*>(storageConfig.Get())) {
            const TRTLineAPI* rtlineApi = GetRTLineAPI(rtyPtr->GetAPIName());
            result.push_back(MakeHolder<TTrustRTYStorageOptions>(rtlineApi));
        } else if (auto localPtr = dynamic_cast<TLocalTrustStorageConfig*>(storageConfig.Get())) {
            result.push_back(MakeHolder<TLocalTrustStorageOptions>());
        } else {
            ERROR_LOG << "Unknown cache type " << storageConfig->GetCacheType() << ":" << storageConfig->GetCacheId() << Endl;
        }
    }

    return result;
}

void NDrive::TServer::DoStop(ui32 /*rigidStopLevel*/, const TCgiParameters* /*cgiParams*/ /*= nullptr*/) {
    if (!TagsManager->Stop()) {
        ERROR_LOG << "cannot stop TagsManager" << Endl;
    }
    if (!DriveAPI->Stop()) {
        ERROR_LOG << "cannot stop DriveAPI" << Endl;
    }
    if (!SnapshotsManager->Stop()) {
        ERROR_LOG << "cannot stop SnapshotsManager" << Endl;
    }
    if (ChatEngine && !ChatEngine->StopEngine()) {
        ERROR_LOG << "cannot stop ChatEngine" << Endl;
    }
    if (SurgeConstructor && !SurgeConstructor->Stop()) {
        ERROR_LOG << "cannot stop SurgeConstructor" << Endl;
    }
    if (WeatherAPI && !WeatherAPI->Stop()) {
        ERROR_LOG << "cannot stop WeatherAPI" << Endl;
    }
    if (FuelingManager && !FuelingManager->Stop()) {
        ERROR_LOG << "cannot stop FuelingManager" << Endl;
    }
    if (ChatMetaManagerHistoryReader && !ChatMetaManagerHistoryReader->Stop()) {
        ERROR_LOG << "cannot stop ChatMetaManagerHistoryReader" << Endl;
    }
    if (SupportCenterManager) {
        SupportCenterManager->Stop();
    }
    TelematicsApi.Destroy();
}

bool NDrive::TServer::RegisterChatRobot(const TChatRobotConfig& config) const {
    return ChatRobotsManager ? ChatRobotsManager->RegisterChatRobot(config) : false;
}

bool NDrive::TServer::UnregisterChatRobot(const TString& robotId) const {
    return ChatRobotsManager ? ChatRobotsManager->UnregisterChatRobot(robotId) : false;
}

TMap<TString, TAtomicSharedPtr<IChatRobot>> NDrive::TServer::GetChatRobots() const {
    if (!!ChatRobotsManager) {
        return ChatRobotsManager->GetChatRobots();
    } else {
        return TMap<TString, TAtomicSharedPtr<IChatRobot>>();
    }
}

const TCovidPassClient* NDrive::TServer::GetCovidPassClient(EPassApiType apiType) const {
    switch (apiType) {
    case EPassApiType::Moscow:
        return GetCovidPassMoscowClient();
    case EPassApiType::MoscowRegion:
        return GetCovidPassMoscowRegionClient();
    case EPassApiType::Gosuslugi:
        return GetCovidPassGosuslugiClient();
    case EPassApiType::Undefined:
        return GetCovidPassGosuslugiClient();
    }
}

TAtomicSharedPtr<NGraph::ILinker> NDrive::TServer::GetLinker(const TString& name) const {
    if (name == "maps_linker") {
        return GetMapsLinker();
    } else {
        auto api = GetRTLineAPI(name);
        return api ? MakeHolder<NGraph::TRouter>(api->GetSearchClient()) : nullptr;
    }
}

TAtomicSharedPtr<NDrive::TMapsLinker> NDrive::TServer::GetMapsLinker() const {
    return MapsLinker;
}

const NDrive::IRouter* NDrive::TServer::GetMapsRouter() const {
    return MapsRouter.Get();
}

const NDrive::TPedestrianRouter* NDrive::TServer::GetPedestrianRouter() const {
    return PedestrianRouter.Get();
}

TMaybe<TString> NDrive::TServer::GetOrCreateUserId(const TString& username, const TString& operatorUserId) const {
    if (!username) {
        return {};
    }
    auto users = GetDriveDatabase().GetUsersData();
    if (users) {
        auto session = users->BuildSession(true);
        auto optionalUserId = users->GetUserIdByLogin(username, session);
        if (optionalUserId && *optionalUserId) {
            return optionalUserId;
        }
    }
    if (users) {
        auto session = users->BuildSession();
        auto optionalUserId = users->GetUserIdByLogin(username, session);
        Y_ENSURE_BT(optionalUserId, session.GetStringReport());
        if (*optionalUserId) {
            return optionalUserId;
        }
        auto user = users->RegisterNewUser(operatorUserId, {}, username, session);
        Y_ENSURE_BT(user, session.GetStringReport());
        Y_ENSURE_BT(session.Commit(), session.GetStringReport());
        return user->GetUserId();
    }
    return {};

}

TMaybe<TString> NDrive::TServer::GetRtBackgroundRobotUserId(const TString& name) const {
    if (!name) {
        WARNING_LOG << "empty name for GetRtBackgroundRobotUserId" << Endl;
    }

    const size_t usernameSize = 64;
    auto desiredUsername =  "rtbg-" + name;
    auto username = StripInPlace(desiredUsername);
    if (username.size() > usernameSize) {
        ui16 hash = FnvHash<ui32>(desiredUsername);
        username.resize(usernameSize - sizeof(hash) * 2 - 1);
        username.append('-');
        username.append(HexEncode(&hash, sizeof(hash)));
    }

    return GetOrCreateUserId(username, "rtbg");
}

const IDBEntitiesWithPropositionsManager<TArea>* NDrive::TServer::GetAreasManager() const {
    return DriveAPI->GetAreasDB();
}

TSet<TString> NDrive::TServer::GetFullAreaTagsList() const {
    return GetDriveAPI()->GetAreasDB()->GetAreaTags();
}

TMaybe<NDrive::TGeoFeaturesClient> NDrive::TServer::GetGeoFeaturesClient(const TString& name) const {
    if (!name) {
        const auto client = GetGeoFeaturesClient();
        if (!client) {
            return Nothing();
        }
        return MakeMaybe<NDrive::TGeoFeaturesClient>(*client);
    }
    const auto api = GetRTLineAPI(name);
    if (!api || !RtmrClient) {
        return Nothing();
    }
    return MakeMaybe<NDrive::TGeoFeaturesClient>(api->GetSearchClient(), *RtmrClient);
}

const NDrive::TUserEventsApi* NDrive::TServer::GetFeaturesClient() const {
    return FeaturesClient.Get();
}

const NDrive::TTaxiSurgeCalculator* NDrive::TServer::GetTaxiSurgeCalculator() const {
    return TaxiSurgeCalculator.Get();
}

const NDrive::TSaturnClient* NDrive::TServer::GetSaturnClient() const {
    return SaturnClient.Get();
}

const TDatabaseSessionConstructor* NDrive::TServer::GetYDB() const {
    return YDB.Get();
}

const NDrive::TParkingAggregatorsManager* NDrive::TServer::GetParkingManager() const {
    return ParkingManager.Get();
}

TVersionedKey NDrive::TServer::GetVersionedKey(IReplyContext& context) const {
    if (!RewriteRouter) {
        return TServerBase::GetVersionedKey(context);
    }
    NDrive::THttpRouter<THttpRewriteConfig>::TMatch match;
    if (!RewriteRouter->Match(match, TString{context.GetUri()}, context.GetBuf().Empty() ? "GET" : "POST")) {
        return TServerBase::GetVersionedKey(context);
    }
    TCgiParameters cgi;
    cgi.Scan(match.Handler.OverrideQuery);
    for (auto&& i : cgi) {
        if (!i.second.empty() && i.second[0] == ':') {
            context.MutableCgiParameters().ReplaceUnescaped(i.first, match.Vars[i.second.substr(1)]);
        } else {
            context.MutableCgiParameters().ReplaceUnescaped(i.first, i.second);
        }
    }
    return TVersionedKey(match.Handler.HandlerPath);
}

NDrive::TTaxiPromocodesClient* NDrive::TServer::GetTaxiPromocodesClient() const {
    return TaxiPromocodesClient.Get();
}
