#pragma once

#include "config.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/background/manager/manager.h>
#include <drive/backend/base/server.h>
#include <drive/backend/chat_robots/robot_manager.h>
#include <drive/backend/registrar/manager.h>
#include <drive/backend/support_center/manager.h>
#include <drive/backend/user_devices/manager.h>
#include <drive/backend/clickhouse/client.h>

#include <drive/library/cpp/clear_web/client.h>
#include <drive/library/cpp/rtmr/client.h>
#include <drive/library/cpp/mux/router.h>
#include <drive/library/cpp/social_api/client.h>
#include <drive/library/cpp/trust/lpm.h>
#include <drive/library/cpp/weather/weather_api.h>
#include <drive/library/cpp/yadoc/client.h>
#include <drive/library/cpp/taxi/driver_trackstory/client.h>
#include <drive/library/cpp/taxi/signalq_drivematics_api/client.h>
#include <drive/telematics/api/client.h>
#include <drive/telematics/api/server/client.h>

#include <kernel/daemon/base_controller.h>
#include <kernel/daemon/messages.h>
#include <kernel/daemon/server.h>

#include <util/thread/pool.h>

namespace NDrive {

class TServer
    : public TServerBase
    , public IServer
{
private:
    const TServerConfig& Config;

    TAtomicSharedPtr<IDriveTagsManager> TagsManager;
    THolder<TDriveAPI> DriveAPI;
    THolder<TFuelingManager> FuelingManager;
    THolder<IUsersController> UsersController;
    THolder<NDrive::TModelsStorage> ModelsStorage;
    THolder<NDrive::TTelematicsApi> TelematicsApi;
    THolder<NDrive::TTelematicsClient> TelematicsClient;
    THolder<NDrive::ISensorApi> SensorApi;
    THolder<NDrive::ISensorHistoryApi> SensorHistoryApi;
    THolder<NDrive::ISensorHistoryClient> SensorHistoryClient;
    THolder<NDrive::TClickHouseBackendClient> RidesInfoClient;
    THolder<NDrive::ITrackClient> TrackClient;
    THolder<NGeobase::TLookup> Geobase;
    THolder<NDrive::TGeoFeaturesClient> GeoFeaturesClient;
    THolder<NDrive::TRtmrSurgeClient> RtmrClient;
    THolder<TWeatherAPI> WeatherAPI;
    THolder<TDevicesSnapshotManager> SnapshotsManager;
    TAtomicSharedPtr<TRTFactorsConstructor> SurgeConstructor;
    TAtomicSharedPtr<TCallbackSequentialTableImpl<TObjectEvent<NDrive::NChat::TChat>, TString>> ChatMetaManagerHistoryReader;
    THolder<NDrive::NChat::TChatsMetaManager> ChatMetaManager;
    THolder<NDrive::NChat::TEngine> ChatEngine;
    THolder<TChatRobotsManager> ChatRobotsManager;
    THolder<IUserDevicesManager> UserDevicesManager;
    THolder<IUserRegistrationManager> UserRegistrationManager;
    THolder<NDrive::TUserEventsApi> UserEventsApi;
    THolder<NDrive::TUserEventsApi> FeaturesClient;
    THolder<NDrive::TParkingZonesManager> ParkingZonesManager;
    THolder<ISupportCenterManager> SupportCenterManager;
    THolder<IHistoryContext> DocumentsManagerHistoryContext;
    THolder<TDocumentsManager> DocumentsManager;

    THolder<IHistoryContext> ChatHistoryContext;
    THolder<IHistoryContext> PromoHistoryContext;
    THolder<IHistoryContext> UserDocumentsChecksContext;
    THolder<IHistoryContext> RadarGeohashContext;
    THolder<IPromoCodesManager> PromoCodesManager;
    THolder<TClearWebClient> ClearWebClient;
    THolder<TSocialAPIClient> SocialAPIClient;
    THolder<NDrive::TTaxiSuggestClient> TaxiSuggestClient;
    THolder<NDrive::TTaxiSupportClassifierClient> TaxiSupportClassifierClient;
    THolder<TAccountEmailBinder> AccountEmailBinder;
    THolder<TSelfHttpRequester> SelfHttpRequester;
    mutable THolder<IOffersStorage> OffersStorage;
    THolder<TCovidPassMoscowRegionClient> CovidPassMoscowRegionClient;
    THolder<TCovidPassMoscowClient> CovidPassMoscowClient;
    THolder<TCovidPassGosuslugiClient> CovidPassGosuslugiClient;
    THolder<TClckClient> ClckClient;
    THolder<ITaxiChatClient> TaxiChatClient;
    THolder<TSimpleTaxiClient> TaxiRouteInfoClient;
    THolder<NDrive::TTaxiRouteHistoryClient> TaxiRouteHistoryClient;
    TAtomicSharedPtr<NDrive::TMapsLinker> MapsLinker;
    THolder<NDrive::IRouter> MapsRouter;
    THolder<NDrive::TPedestrianRouter> PedestrianRouter;
    THolder<TBalanceClient> BalanceClient;
    THolder<TYaDocClient> YaDocClient;
    THolder<IDistributingBlockEventsStorage> DistributingBlockEventsStorage;
    THolder<TTaxiSupportChatSuggestClient> TaxiSupportChatSuggestClient;
    THolder<TSendrClient> SendrClient;
    THolder<NDrive::TTaxiSurgeCalculator> TaxiSurgeCalculator;
    THolder<NDrive::TSaturnClient> SaturnClient;
    THolder<TTaxiAntifraudClient> TaxiAntidraudClient;
    THolder<TTaxiDriverProfilesClient> TaxiDriverProfilesClient;
    THolder<TTaxiDriverStatusClient> TaxiDriverStatusClient;
    THolder<ITaxiSignalqDrivematicsApiClient> TaxiSignalqDrivematicsApiClient;
    THolder<TTaxiFleetVehiclesClient> TaxiFleetVehiclesClient;
    THolder<TLPMClient> LPMClient;
    THolder<TUserDocumentsChecksManager> UserDocumentsChecksManager;
    THolder<TRadarGeohashManager> RadarGeohashManager;
    THolder<TDatabaseSessionConstructor> YDB;
    THolder<NDrive::TParkingAggregatorsManager> ParkingManager;
    THolder<TTaxiTrackStoryClient> TaxiTrackStoryClient;
    THolder<NDrive::TSupportAIClient> SupportAIClient;
    THolder<TYangClient> YangClient;
    THolder<NDrive::NRenins::NKasko::TClient> ReninsKaskoClient;
    // RewriteRouter represents rewrite implementation for rewrite router.
    THolder<NDrive::THttpRouter<THttpRewriteConfig>> RewriteRouter;
    THolder<NDrive::TTaxiPromocodesClient> TaxiPromocodesClient;

private:
    TVector<THolder<NDrive::ITrustStorageOptions>> ConstructCacheStorageOptions(const TLPMClientConfig& config);

protected:
    virtual void DoRun() override;
    virtual void DoStop(ui32 /*rigidStopLevel*/, const TCgiParameters* /*cgiParams*/ = nullptr) override;

public:
    using TConfig = TServerConfig;

public:
    TServer(const TConfig& config);
    ~TServer();

    const TConfig& GetConfigImpl() const {
        return Config;
    }

    virtual bool Process(IMessage* message) override;

    virtual const TLPMClient* GetLPMClient() const override {
        return LPMClient.Get();
    }

    virtual IOffersStorage* GetOffersStorage() const override {
        return OffersStorage.Get();
    }

    virtual const IPromoCodesManager* GetPromoCodesManager() const override {
        return PromoCodesManager.Get();
    }

    virtual const TClearWebClient* GetClearWebClient() const override {
        return ClearWebClient.Get();
    }

    virtual const IDBEntitiesWithPropositionsManager<TArea>* GetAreasManager() const override;

    virtual const NDrive::NChat::TEngine* GetChatEngine() const override {
        return ChatEngine.Get();
    }

    virtual const NDrive::TTaxiSuggestClient* GetTaxiSuggest() const override {
        return TaxiSuggestClient.Get();
    }

    virtual const NDrive::TTaxiSupportClassifierClient* GetTaxiSupportClassifier() const override {
        return TaxiSupportClassifierClient.Get();
    }

    virtual const TFuelingManager* GetFuelingManager() const override {
        return FuelingManager.Get();
    }

    virtual const IDriveDatabase& GetDriveDatabase() const override {
        return *Yensured(DriveAPI);
    }

    virtual const TDriveAPI* GetDriveAPI() const override {
        return DriveAPI.Get();
    }

    IUsersController* GetUsersController() const override {
        return UsersController.Get();
    }

    virtual NDrive::TModelsStorage* GetModelsStorage() const override {
        return ModelsStorage.Get();
    }

    virtual const NDrive::TTelematicsApi& GetTelematicsApi() const override {
        return *Yensured(TelematicsApi);
    }

    virtual const NDrive::TTelematicsClient& GetTelematicsClient() const override {
        return *Yensured(TelematicsClient);
    }

    virtual const NDrive::ISensorApi* GetSensorApi() const override {
        return SensorApi.Get();
    }

    const NDrive::ISensorHistoryApi* GetSensorHistoryApi() const override {
        return SensorHistoryApi.Get();
    }

    const NDrive::ISensorHistoryClient* GetSensorHistoryClient() const override {
        return SensorHistoryClient.Get();
    }

    const NDrive::TClickHouseBackendClient* GetRidesInfoClient() const override {
        return RidesInfoClient.Get();
    }

    const NDrive::ITrackClient* GetTrackClient() const override {
        return TrackClient.Get();
    }

    virtual TSet<TString> GetFullAreaTagsList() const override;

    virtual const TWeatherAPI* GetWeatherAPI() const override {
        return WeatherAPI.Get();
    }

    TAtomicSharedPtr<TRTFactorsConstructor> GetSurgeConstructor() const override {
        return SurgeConstructor;
    }

    const TDevicesSnapshotManager& GetSnapshotsManager() const override {
        return *SnapshotsManager;
    }

    const IUserDevicesManager* GetUserDevicesManager() const override {
        return UserDevicesManager.Get();
    }

    virtual const IUserRegistrationManager* GetUserRegistrationManager() const override {
        return UserRegistrationManager.Get();
    }

    virtual const TChatRobotsManager* GetChatRobotsManager() const override {
        return ChatRobotsManager.Get();
    }

    virtual const IChatRobot::TPtr GetChatRobot(const TString& name) const override {
        if (ChatRobotsManager) {
            return ChatRobotsManager->GetChatRobot(name);
        } else {
            return nullptr;
        }
    }

    virtual const NGeobase::TLookup* GetGeobase() const override {
        return Geobase.Get();
    }

    virtual const TS3Client* GetMDSClient() const override {
        if (!DriveAPI || !DriveAPI->HasMDSClient()) {
            return nullptr;
        }
        return &DriveAPI->GetMDSClient();
    }

    virtual const TStartrekClient* GetStartrekClient() const override {
        if (!DriveAPI || !DriveAPI->HasStartrekClient()) {
            return nullptr;
        }
        return &DriveAPI->GetStartrekClient();
    }

    virtual const TClckClient* GetClckClient() const override {
        return ClckClient.Get();
    }

    virtual const NDrive::TGeoFeaturesClient* GetGeoFeaturesClient() const override {
        return GeoFeaturesClient.Get();
    }

    virtual TMaybe<NDrive::TGeoFeaturesClient> GetGeoFeaturesClient(const TString& name) const override;

    const NDrive::TRtmrSurgeClient* GetRtmrClient() const override {
        return RtmrClient.Get();
    }

    virtual const NDrive::TUserEventsApi* GetUserEventsApi() const override {
        return UserEventsApi.Get();
    }

    virtual const NDrive::TUserEventsApi* GetFeaturesClient() const override;

    virtual const NDrive::TParkingZonesManager& GetParkingZonesManager() const override {
        Y_ENSURE(ParkingZonesManager);
        return *ParkingZonesManager;
    }

    virtual const ISupportCenterManager* GetSupportCenterManager() const override {
        return SupportCenterManager.Get();
    }

    virtual const TSocialAPIClient* GetSocialAPIClient() const override {
        return SocialAPIClient.Get();
    }

    virtual const TAccountEmailBinder* GetAccountEmailBinder() const override {
        return AccountEmailBinder.Get();
    }

    virtual const TSelfHttpRequester* GetSelfHttpRequester() const override {
        return SelfHttpRequester.Get();
    }

    virtual const TDocumentsManager* GetDocumentsManager() const override {
        return DocumentsManager.Get();
    }

    const TCovidPassMoscowClient* GetCovidPassMoscowClient() const {
        return CovidPassMoscowClient.Get();
    }

    const TCovidPassMoscowRegionClient* GetCovidPassMoscowRegionClient() const {
        return CovidPassMoscowRegionClient.Get();
    }

    const TCovidPassGosuslugiClient* GetCovidPassGosuslugiClient() const {
        return CovidPassGosuslugiClient.Get();
    }

    virtual const ITaxiChatClient* GetTaxiChatClient() const override {
        return TaxiChatClient.Get();
    }

    virtual const TSimpleTaxiClient* GetTaxiRouteInfoClient() const override {
        return TaxiRouteInfoClient.Get();
    }

    virtual const NDrive::TTaxiRouteHistoryClient* GetTaxiRouteHistoryClient() const override {
        return TaxiRouteHistoryClient.Get();
    }

    virtual const TBalanceClient* GetBalanceClient() const override {
        return BalanceClient.Get();
    }

    virtual const TYaDocClient* GetYaDocClient() const override {
        return YaDocClient.Get();
    }

    virtual const IDistributingBlockEventsStorage* GetDistributingBlockEventsStorage() const override {
        return DistributingBlockEventsStorage.Get();
    }

    virtual const TTaxiSupportChatSuggestClient* GetTaxiSupportChatSuggestClient() const override {
        return TaxiSupportChatSuggestClient.Get();
    }

    virtual const TSendrClient* GetSendrClient() const override {
        return SendrClient.Get();
    }

    virtual const TTaxiAntifraudClient* GetTaxiAntifraudClient() const override {
        return TaxiAntidraudClient.Get();
    }

    virtual const TTaxiDriverProfilesClient* GetTaxiDriverProfilesClient() const override {
        return TaxiDriverProfilesClient.Get();
    }

    virtual const TTaxiDriverStatusClient* GetTaxiDriverStatusClient() const override {
        return TaxiDriverStatusClient.Get();
    }

    virtual const ITaxiSignalqDrivematicsApiClient* GetTaxiSignalqDrivematicsApiClient() const override {
        return TaxiSignalqDrivematicsApiClient.Get();
    }

    virtual const TTaxiFleetVehiclesClient* GetTaxiFleetVehiclesClient() const override {
        return TaxiFleetVehiclesClient.Get();
    }

    virtual const TUserDocumentsChecksManager* GetUserDocumentsChecksManager() const override {
        return UserDocumentsChecksManager.Get();
    }

    virtual const TRadarGeohashManager* GetRadarGeohashManager() const override {
        return RadarGeohashManager.Get();
    }

    virtual const TTaxiTrackStoryClient* GetTaxiTrackStoryClient() const override {
        return TaxiTrackStoryClient.Get();
    }

    virtual const NDrive::TSupportAIClient* GetSupportAIClient() const override {
        return SupportAIClient.Get();
    }

    virtual const TYangClient* GetYangClient() const override {
        return YangClient.Get();
    }

    virtual const NDrive::NRenins::NKasko::TClient* GetReninsKaskoClient() const override {
        return ReninsKaskoClient.Get();
    }

    const NDrive::TTaxiSurgeCalculator* GetTaxiSurgeCalculator() const override;
    const NDrive::TSaturnClient* GetSaturnClient() const override;

    bool RegisterChatRobot(const TChatRobotConfig& config) const override;
    bool UnregisterChatRobot(const TString& robotId) const override;
    virtual TMap<TString, TAtomicSharedPtr<IChatRobot>> GetChatRobots() const override;

    virtual const TCovidPassClient* GetCovidPassClient(EPassApiType apiType) const override;
    virtual TAtomicSharedPtr<NGraph::ILinker> GetLinker(const TString& name) const override;
    virtual TAtomicSharedPtr<NDrive::TMapsLinker> GetMapsLinker() const override;
    const NDrive::IRouter* GetMapsRouter() const override;
    const NDrive::TPedestrianRouter* GetPedestrianRouter() const override;
    virtual TMaybe<TString> GetOrCreateUserId(const TString& username, const TString& operatorUserId) const override;
    virtual TMaybe<TString> GetRtBackgroundRobotUserId(const TString& name) const override;
    // GetYDB returns session constructor for YDB.
    const TDatabaseSessionConstructor* GetYDB() const override;
    const NDrive::TParkingAggregatorsManager* GetParkingManager() const override;
    TVersionedKey GetVersionedKey(IReplyContext& context) const override;
    NDrive::TTaxiPromocodesClient* GetTaxiPromocodesClient() const override;
};

} // namespace NDrive
