#include "billing.h"

#include <drive/backend/data/chargable.h>
#include <drive/backend/data/offer.h>
#include <drive/backend/logging/evlog.h>

TSessionManager::TSessionManager(const IDriveTagsManager& driveTagsManager)
    : TBase(driveTagsManager.GetDeviceTags(), MakeHolder<TBillingSessionSelector>(driveTagsManager))
    , DriveTagsManager(driveTagsManager)
    , DefaultCacheLifetime(TDuration::Seconds(1000))
    , SessionTagInfoCache(64 * 1024)
{
}

TSessionManager::TOptionalSessionTagInfos TSessionManager::GetCurrentSessions(NDrive::TEntitySession& tx) const {
    auto timestamp = Now();
    auto optionalTags = DriveTagsManager.GetTraceTags().RestoreTagsRobust({}, { TChargableSessionTag::Type() }, tx);
    if (!optionalTags) {
        return {};
    }
    TSessionTagInfos result;
    for (auto&& tag : *optionalTags) {
        auto impl = tag.GetTagAs<TChargableSessionTag>();
        if (!impl) {
            tx.SetErrorInfo("BillingSessionManager::GetCurrentSessions", TStringBuilder() << "cannot cast " << tag.GetTagId() << " as ChargableSessionTag");
            return {};
        }
        TSessionTagInfo info;
        info.TagId = ToString(impl->GetSessionTagId());
        info.TagEntityType = impl->GetSessionTagEntityType();
        info.Timestamp = timestamp;
        result.emplace(tag.GetObjectId(), std::move(info));
    }
    return result;
}

TCommonTagSessionManager::TOptionalConstSession TSessionManager::GetSession(const TString& id, NDrive::TEntitySession& tx) const {
    auto timestamp = Now();
    auto optionalSessionTagInfo = SessionTagInfoCache.find(id);
    if (optionalSessionTagInfo) {
        auto eventLogger = NDrive::GetThreadEventLogger();
        auto lifetimeKey = TStringBuilder() << "billing_session_manager.cache_lifetime";
        auto lifetime = NDrive::HasServer()
            ? NDrive::GetServer().GetSettings().GetValue<TDuration>(lifetimeKey)
            : Nothing();
        auto threshold = timestamp - lifetime.GetOrElse(DefaultCacheLifetime);
        if (optionalSessionTagInfo->Timestamp > threshold) {
            if (eventLogger) {
                eventLogger->AddEvent(NJson::TMapBuilder
                    ("event", "SessionTagInfoCacheHit")
                    ("id", id)
                    ("timestamp", NJson::ToJson(optionalSessionTagInfo->Timestamp))
                );
            }
        } else {
            optionalSessionTagInfo.Clear();
        }
    }
    if (!optionalSessionTagInfo || optionalSessionTagInfo->TagEntityType == NEntityTagsManager::EEntityType::User) {
        auto optionalChargableSessionTag = TChargableSessionTag::Get(id, DriveTagsManager, tx);
        if (!optionalChargableSessionTag) {
            return {};
        }
        auto chargableSessionTag = optionalChargableSessionTag->GetTagAs<TChargableSessionTag>();
        if (chargableSessionTag) {
            TSessionTagInfo info;
            info.TagId = ToString(chargableSessionTag->GetSessionTagId());
            info.TagEntityType = chargableSessionTag->GetSessionTagEntityType();
            info.Timestamp = timestamp;
            SessionTagInfoCache.update(id, info);
            optionalSessionTagInfo = std::move(info);
        } else {
            optionalSessionTagInfo.Clear();
        }
    }
    if (!optionalSessionTagInfo) {
        return nullptr;
    }
    if (optionalSessionTagInfo->TagEntityType == NEntityTagsManager::EEntityType::User) {
        return CreateOfferHolderSession(optionalSessionTagInfo->TagId, {}, DriveTagsManager.GetUserTags(), tx);
    }

    return GetTagSession(optionalSessionTagInfo->TagId, tx);
}


TSessionManager::TOptionalConstSessions TSessionManager::GetObjectSessions(const TString& objectId, NDrive::TEntitySession& tx) const {
    return GetObjectsSessions({ objectId }, tx);
}

TSessionManager::TOptionalConstSessions TSessionManager::GetObjectsSessions(const TVector<TString>& objectsIds, NDrive::TEntitySession& tx) const {
    TTaggedObjectsSnapshot snapshot;

    if (!GetTagManager().RestoreObjects(MakeSet(objectsIds), snapshot, tx)) {
        tx.AddErrorMessage("TSessionManager::GetObjectSessions", "cannot RestoreObjects");
        return {};
    }
    TConstSessions result;
    TVector<std::pair<TString, TString>> skipSessions;
    for (const auto& [_, object] : snapshot) {
        for (auto&& tag : object.GetTags()) {
            if (!tag) {
                continue;
            }
            if (tag->GetPerformer().empty()) {
                continue;
            }
            auto chargableTag = tag.GetTagAs<TChargableTag>();
            if (!chargableTag) {
                continue;
            }
            auto session = GetTagSession(tag.GetTagId(), tx);
            if (!session) {
                return {};
            }
            if (!session.GetRef()) {
                skipSessions.emplace_back(tag.GetTagId(), object.GetId());
                continue;
            }
            result.push_back(std::move(*session));
        }
    }
    if (!skipSessions.empty()) {
        NJson::TJsonValue skipSessionsJson;
        for (const auto& [tagId, objectId] : skipSessions) {
            skipSessionsJson.AppendValue(
                NJson::TMapBuilder
                    ("tag_id", tagId)
                    ("object_id", objectId)
            );
        }
        NDrive::TEventLog::Log("GetObjectSessionsSkip", std::move(skipSessionsJson));
    }

    return result;
}

TSessionManager::TOptionalConstSessions TSessionManager::GetUserSessions(const TString& userId, NDrive::TEntitySession& tx, TDuration finishedDepth) const {
    auto result = TBase::GetUserSessions(userId, tx, finishedDepth);
    if (!result) {
        return {};
    }

    const auto& userTagManager = DriveTagsManager.GetUserTags();
    auto optionalTaggedUser = userTagManager.RestoreObject(userId, tx);
    if (!optionalTaggedUser) {
        return {};
    }
    for (auto&& tag : optionalTaggedUser->GetTags()) {
        auto offerHolder = tag.GetTagAs<TOfferHolderUserTag>();
        auto offerHolderSession = offerHolder ? CreateOfferHolderSession(tag, {}, userTagManager, tx) : nullptr;
        if (offerHolderSession) {
            result->push_back(std::move(offerHolderSession));
            continue;
        }
    }

    return result;
}
