#include "common.h"


TCommonTagSessionManager::TOptionalConstSessions TCommonTagSessionManager::GetUserSessions(const TString& userId, NDrive::TEntitySession& tx, TDuration finishedDepth) const {
    TVector<TDBTag> performedTags;
    if (!TagManager.RestorePerformerTags({ userId }, performedTags, tx)) {
        return {};
    }
    TConstSessions result;
    for (auto&& tag : performedTags) {
        auto optionalSession = GetTagSession(tag.GetTagId(), tx);
        if (!optionalSession) {
            tx.AddErrorMessage("CommonTagSessionManager::GetUserSessions", "cannot GetTagSession for " + tag.GetTagId());
            return {};
        }
        auto session = *optionalSession;
        if (session) {
            Y_ASSERT(session->GetUserId() == userId);
            result.push_back(std::move(session));
        }
    }
    if (finishedDepth && result.empty()) {
        auto since = ModelingNow() - finishedDepth;
        auto optionalEvents = TagManager.GetEvents(since, tx, IEntityTagsManager::TQueryOptions()
            .SetActions({ EObjectHistoryAction::SetTagPerformer })
            .SetPerformers({ userId })
        );
        if (!optionalEvents) {
            return {};
        }
        for (auto&& ev : *optionalEvents) {
            auto optionalSession = GetTagSession(ev.GetTagId(), tx);
            if (!optionalSession) {
                return {};
            }
            auto session = *optionalSession;
            if (session) {
                result.push_back(std::move(session));
            }
        }
    }
    return result;
}

TCommonTagSessionManager::TOptionalConstSessions TCommonTagSessionManager::GetSessionsActualSinceId(TRange<TTagHistoryEvent::TEventId> idRange, NDrive::TEntitySession& tx) const {
    auto optionalEvents = TagManager.GetEvents(idRange, tx, {});
    if (!optionalEvents) {
        return {};
    }

    TSet<TString> tagIds;
    Transform(optionalEvents->begin(), optionalEvents->end(), std::inserter(tagIds, tagIds.begin()), [](const auto& ev) { return ev.GetTagId(); });

    auto optionalTagEvents = TagManager.GetEvents({}, {}, tx, IEntityTagsManager::TQueryOptions()
            .SetTagIds(std::move(tagIds))
    );
    if (!optionalTagEvents) {
        return {};
    }
    TMap<TString, TTagHistoryEvents> tagEvents;
    ForEach(optionalTagEvents->begin(), optionalTagEvents->end(), [&tagEvents](const auto& ev) {
        tagEvents[ev.GetTagId()].emplace_back(ev);
    });

    TConstSessions result;
    for (auto&& [id, events] : tagEvents) {
        auto session = ConsumeEvents(nullptr, std::move(events), false);
        if (session) {
            result.emplace_back(std::move(session));
        }
    }
    return result;
}

TCommonTagSessionManager::TOptionalConstSessions TCommonTagSessionManager::GetObjectSessionsByTagNames(const TString& objectId, const TVector<TString>& tagNames, NDrive::TEntitySession& tx, TDuration finishedDepth) const {
    TVector<TDBTag> restoredTags;
    if (!TagManager.RestoreTags({objectId}, tagNames, restoredTags, tx)) {
        return {};
    }
    TConstSessions result;
    for (auto&& tag : restoredTags) {
        auto optionalSession = GetTagSession(tag.GetTagId(), tx);
        if (!optionalSession) {
            tx.AddErrorMessage("CommonTagSessionManager::GetObjectSessions", "cannot GetTagSession for " + tag.GetTagId());
            return {};
        }
        auto session = *optionalSession;
        if (session) {
            Y_ASSERT(session->GetObjectId() == objectId);
            result.push_back(std::move(session));
        }
    }
    if (finishedDepth && result.empty()) {
        auto since = ModelingNow() - finishedDepth;
        auto optionalEvents = TagManager.GetEvents(since, tx, IEntityTagsManager::TQueryOptions()
            .SetActions({EObjectHistoryAction::Add})
            .SetObjectIds({objectId})
        );
        if (!optionalEvents) {
            return {};
        }
        for (auto&& ev : *optionalEvents) {
            auto optionalSession = GetTagSession(ev.GetTagId(), tx);
            if (!optionalSession) {
                return {};
            }
            auto session = *optionalSession;
            if (session) {
                result.push_back(std::move(session));
            }
        }
    }
    return result;
}

TCommonTagSessionManager::TOptionalConstSession TCommonTagSessionManager::GetTagSession(const TString& tagId, NDrive::TEntitySession& tx) const {
    auto session = TagIdIndex.find(tagId).GetOrElse(nullptr);
    if (session) {
        Y_ASSERT(session->GetInstanceId() == tagId);
    }
    bool valid = session && (session->GetCurrentState() != ISession::ECurrentState::Corrupted);
    auto since = valid ? (session->GetLastEventId() + 1) : 0;
    auto timestamp = ModelingNow();
    auto events = TagManager.GetEventsByTag(tagId, tx, since);
    if (!events) {
        return {};
    }
    if (events->empty() && valid) {
        session->SignalRefresh(timestamp);
        return session;
    }
    auto consumer = valid ? session->Clone() : nullptr;
    bool updateCache = !tx->IsWritable();
    auto result = ConsumeEvents(std::move(consumer), std::move(*events), updateCache);
    if (result) {
        result->SignalRefresh(timestamp);
    }
    return result;
}

TCommonTagSessionManager::TSessionPtr TCommonTagSessionManager::ConsumeEvents(THolder<TSession>&& consumer, TTagHistoryEvents&& events, bool updateCache) const {
    auto session = Share(std::move(consumer));
    for (auto&& ev : events) {
        if (session) {
            Y_ASSERT(session->GetInstanceId() == ev.GetTagId());
            if (session->GetInstanceId() != ev.GetTagId()) {
                ERROR_LOG << "bad event for session " << session->GetSessionId() << ": " << ev.SerializeToJson().GetStringRobust() << Endl;
                continue;
            }
        }
        auto category = Yensured(Selector)->Accept(ev);
        switch (category) {
            case NEventsSession::IgnoreExternal:
            case NEventsSession::IgnoreInternal:
                break;
            case NEventsSession::Switch:
                if (session && !session->GetClosed()) {
                    session->AddEvent(MakeAtomicShared<TTagHistoryEvent>(ev), NEventsSession::End);
                }
                [[fallthrough]];
            case NEventsSession::Begin:
                if (session) {
                    session->CheckClosed();
                }
                session = Selector->BuildSession();
                if (session) {
                    session->AddEvent(MakeAtomicShared<TTagHistoryEvent>(std::move(ev)), NEventsSession::Begin);
                } else {
                    ERROR_LOG << "selector " << Selector->GetName() << " did not create a session" << Endl;
                }
                break;
            default:
                if (session) {
                    session->AddEvent(MakeAtomicShared<TTagHistoryEvent>(std::move(ev)), category);
                }
        }
    }
    if (session && updateCache) {
        const auto& tagId = session->GetInstanceId();
        if (tagId) {
            TagIdIndex.update(tagId, session);
        }
    }
    return session;
}
