#include "dedicated_fleet.h"

#include <drive/backend/data/dedicated_fleet.h>
#include <drive/backend/data/dedicated_fleet_session.h>
#include <drive/backend/data/offer.h>

#include <drive/backend/logging/evlog.h>

TDedicatedFleetSessionManager::TDedicatedFleetSessionManager(const IDriveTagsManager& driveTagsManager)
    : TBase(driveTagsManager.GetAccountTags(), MakeHolder<TDedicatedFleetSessionSelector>()),
    CarTagsManager(driveTagsManager.GetDeviceTags())
{
}

TCommonTagSessionManager::TOptionalConstSessions TDedicatedFleetSessionManager::GetAccountCurrentSessions(const TString& accountId, NDrive::TEntitySession& tx) const {
    auto result = GetAccountSessions(accountId, tx);
    if (!result) {
        return {};
    }

    TConstSessions current;
    for (auto&& session : *result) {
        if (session && !session->GetClosed()) {
            current.emplace_back(std::move(session));
        }
    }

    return current;
}

TCommonTagSessionManager::TOptionalConstSessions TDedicatedFleetSessionManager::GetAccountSessions(const TString& accountId, NDrive::TEntitySession& tx) const {
    auto optionalTags = GetTagManager().RestoreEntityTags(accountId, {TDedicatedFleetOfferHolderTag::Type()}, tx);
    if (!optionalTags) {
        return {};
    }

    TConstSessions result;
    for (auto&& tag : *optionalTags) {
        auto optionalSession = GetTagSession(tag.GetTagId(), tx);
        if (!optionalSession) {
            tx.AddErrorMessage("TDedicatedFleetSessionManager::GetCurrentSessions", "cannot GetSession for " + tag.GetTagId());
            return {};
        }

        auto session = *optionalSession;
        if (session && !session->GetClosed()) {
            result.push_back(std::move(session));
        }
    }

    return result;
}

TCommonTagSessionManager::TOptionalConstSession TDedicatedFleetSessionManager::GetTagSession(const TString& tagId, NDrive::TEntitySession& tx) const {
    auto session = TagIdIndex.find(tagId).GetOrElse(nullptr);
    if (session) {
        Y_ASSERT(session->GetInstanceId() == tagId);
    }
    bool valid = session && (session->GetCurrentState() != ISession::ECurrentState::Corrupted);
    auto since = valid ? (session->GetLastEventId() + 1) : 0;
    auto timestamp = ModelingNow();
    auto optAccountEvents = TagManager.GetEventsByTag(tagId, tx, since);
    if (!optAccountEvents) {
        return {};
    }

    TSet<TString> carTags;
    for (const auto& event : *optAccountEvents) {
        auto holderTag = std::dynamic_pointer_cast<const TDedicatedFleetOfferHolderTag>(event.GetData());
        if (!holderTag) {
            return {};
        }

        for (auto&& id : holderTag->GetFleetTagIds()) {
            carTags.emplace(std::move(id));
        }
    }

    TVector<TTagHistoryEvents> events;
    events.reserve(carTags.size() + 1);
    events.push_back(std::move(*optAccountEvents));
    size_t totalSize = events.size();
    for (const auto& tagId : carTags) {
        auto carEvents = CarTagsManager.GetEventsByTag(tagId, tx, since);
        if (!carEvents) {
            return {};
        }

        totalSize += carEvents->size();
        events.push_back(std::move(*carEvents));
    }

    TTagHistoryEvents sortedEvents;
    sortedEvents.reserve(totalSize);
    TVector<size_t> indexes(events.size(), 0);
    while (true) {
        std::pair<size_t, size_t> currentIndex {0, 0};
        TInstant currentInstant = TInstant::Max();
        for (size_t i = 0; i < indexes.size(); ++i) {
            if (indexes[i] < events[i].size() && events[i][indexes[i]].GetHistoryInstant() < currentInstant) {
                currentInstant = events[i][indexes[i]].GetHistoryInstant();
                currentIndex = {i, indexes[i]};
            }
        }

        if (currentInstant == TInstant::Max()) {
            break;
        }

        sortedEvents.emplace_back(std::move(events[currentIndex.first][currentIndex.second]));
        indexes[currentIndex.first] = currentIndex.second + 1;
    }

    if (sortedEvents.empty() && valid) {
        session->SignalRefresh(timestamp);
        return session;
    }

    auto consumer = valid ? session->Clone() : nullptr;
    bool updateCache = !tx->IsWritable();
    auto result = ConsumeEvents(std::move(consumer), std::move(sortedEvents), updateCache);
    if (result) {
        result->SignalRefresh(timestamp);
    }

    return result;
}

TCommonTagSessionManager::TSessionPtr TDedicatedFleetSessionManager::ConsumeEvents(THolder<TSession>&& consumer, TTagHistoryEvents&& events, bool updateCache) const {
    auto session = Share(std::move(consumer));
    for (auto&& ev : events) {
        auto category = Yensured(Selector)->Accept(ev);
        switch (category) {
            case NEventsSession::IgnoreExternal:
            case NEventsSession::IgnoreInternal:
                break;
            case NEventsSession::Switch:
                if (session && !session->GetClosed()) {
                    session->AddEvent(MakeAtomicShared<TTagHistoryEvent>(ev), NEventsSession::End);
                }
                [[fallthrough]];
            case NEventsSession::Begin:
                if (session) {
                    session->CheckClosed();
                }
                session = std::dynamic_pointer_cast<TDedicatedFleetSession>(Selector->BuildSession());
                if (session) {
                    session->AddEvent(MakeAtomicShared<TTagHistoryEvent>(std::move(ev)), NEventsSession::Begin);
                } else {
                    ERROR_LOG << "selector " << Selector->GetName() << " did not create a session" << Endl;
                }
                break;
            default:
                if (session) {
                    session->AddEvent(MakeAtomicShared<TTagHistoryEvent>(std::move(ev)), category);
                }
        }
    }
    if (session && updateCache) {
        const auto& tagId = session->GetInstanceId();
        if (tagId) {
            TagIdIndex.update(tagId, session);
        }
    }
    return session;
}
