#pragma once

#include <library/cpp/object_factory/object_factory.h>

#include <rtline/util/types/accessor.h>

#include <util/datetime/base.h>
#include <util/generic/algorithm.h>
#include <util/generic/map.h>

namespace NDrive {
    class IServer;
}

template <typename T> class TObjectEvent;
class TFullCompiledRiding;

class TBillingEventsCompilation;

class TConstDBTag;
using TCarTagHistoryEvent = TObjectEvent<TConstDBTag>;

template <typename TEvent> class TSessionsBuilder;
using TCarSessionsBuilderPtr = TAtomicSharedPtr<TSessionsBuilder<TCarTagHistoryEvent>>;

template <typename TEvent> class IEventsSession;
using TCarTagHistoryEventsSessionPtr = TAtomicSharedPtr<IEventsSession<TCarTagHistoryEvent>>;

namespace NDrive::NSession {
    // similar enums and constants are here:
    // - ESessionState in drive/backend/offers/discount.h
    // - TChargableTag constants in drive/backend/data/chargable.h
    enum class ESessionState {
        Parking = 1 << 0 /* "old_state_parking" */,
        Riding = 1 << 1 /* "old_state_riding" */,
        Reservation = 1 << 2 /* "old_state_reservation" */,
        Acceptance = 1 << 3 /* "old_state_acceptance" */,
    };

    class TSessionBindingInfo {
    public:
        struct TBindingAlgoMetaInfo {
            TInstant EarliestRideStart;
            TInstant LatestRideStart;
            size_t SelectedRidesTotal = 0;
            i32 SkippedTotal = 0;

            explicit operator bool() const {
                return EarliestRideStart || LatestRideStart || SelectedRidesTotal || SkippedTotal;
            }
        };

        TSessionBindingInfo() = default;

        TSessionBindingInfo(const TString& sessionId, const TString& userId, const i32 skipped)
            : SessionId(sessionId)
            , UserId(userId)
            , Skipped(skipped)
        {
        }

        constexpr bool IsInitialized() const {
            return !!SessionId && !!UserId;
        }

    private:
        R_FIELD(TString, SessionId);
        R_FIELD(TString, UserId);
        R_FIELD(i32, Skipped, 0);

        R_FIELD(TBindingAlgoMetaInfo, CompiledRidesMatchingMetaInfo);
        R_FIELD(TBindingAlgoMetaInfo, BillingSessionsMatchingMetaInfo);
    };
    using TOptionalSessionBindingInfo = TMaybe<TSessionBindingInfo>;

    struct TMatchingOptions {
        ui32 RejectedSessionsSkipLimit = 2;
        ui32 MinRideDistanceMeters = 10;
    };

    class IMatchingConstraint {
    public:
        using TPtr = TAtomicSharedPtr<IMatchingConstraint>;
        using TFactory = NObjectFactory::TParametrizedObjectFactory<IMatchingConstraint, TString, const NDrive::IServer*, const TMatchingOptions&>;

        IMatchingConstraint(const NDrive::IServer* server, const TMatchingOptions& options);

        virtual ~IMatchingConstraint() = default;

        virtual bool Match(const TObjectEvent<TFullCompiledRiding>& objEvent, const TInstant timestamp) const = 0;
        virtual bool Match(TCarTagHistoryEventsSessionPtr objEventPtr, const TBillingEventsCompilation& eventsCompilation, const TInstant timestamp) const = 0;

    protected:
        const NDrive::IServer* Server;
        const TMatchingOptions Options;
    };

    class TMatchingConstraintsGroup {
    public:
        using TPtr = TAtomicSharedPtr<TMatchingConstraintsGroup>;

        using TConstraints = TMap<TString, IMatchingConstraint::TPtr>;
        using TFullCompiledRideContainer = TVector<TObjectEvent<TFullCompiledRiding>>;
        using TCarTagHistoryEventsSessionPtrContainer = TVector<TCarTagHistoryEventsSessionPtr>;

        static TPtr Construct(const NDrive::IServer* server, const TMatchingOptions& options, const TSet<TString>& constraintNames, const bool checkEmpty = true);

        TMatchingConstraintsGroup(const NDrive::IServer* server, const TMatchingOptions& options);
        TMatchingConstraintsGroup(const NDrive::IServer* server, const TMatchingOptions& options, std::initializer_list<TConstraints::value_type> il);
        TMatchingConstraintsGroup(const TMatchingConstraintsGroup&) = default;

        const TMatchingOptions& GetOptions() const {
            return Options;
        }

        TOptionalSessionBindingInfo MatchSession(const TString& carId, TInstant timestamp) const;

        constexpr bool Empty() const;
        bool Has(const TString& name) const;
        void Add(const TString& name, IMatchingConstraint::TPtr constraint);

        bool Match(const TObjectEvent<TFullCompiledRiding>& objEvent, const TInstant timestamp) const;
        bool Match(TCarTagHistoryEventsSessionPtr objEventPtr, const TInstant timestamp) const;

        bool MatchSession(const TString& carId, const TInstant timestamp, TSessionBindingInfo& sessionInfo) const;

        bool MatchCompiledRidesSession(const TString& carId, const TInstant timestamp, TSessionBindingInfo& sessionInfo) const;
        bool MatchCompiledRidesSession(const TFullCompiledRideContainer& fullCompiledRides, const TString& carId, const TInstant timestamp, TSessionBindingInfo& sessionInfo) const;

        bool MatchBillingSession(const TString& carId, const TInstant timestamp, TSessionBindingInfo& sessionInfo) const;
        bool MatchBillingSession(const TCarTagHistoryEventsSessionPtrContainer& vSessions, const TString& carId, const TInstant timestamp, TSessionBindingInfo& sessionInfo) const;

    private:
        const NDrive::IServer* Server;
        const TMatchingOptions Options;

        TConstraints Constraints;
    };

    class TViolationDuringSessionMatchingConstraint: public IMatchingConstraint {
        using TBase = IMatchingConstraint;

    public:
        using TBase::TBase;

        static TString GetTypeName();

        virtual bool Match(const TObjectEvent<TFullCompiledRiding>& objEvent, const TInstant timestamp) const override;
        virtual bool Match(TCarTagHistoryEventsSessionPtr objEventPtr, const TBillingEventsCompilation& eventsCompilation, const TInstant timestamp) const override;

    private:
        static TFactory::TRegistrator<TViolationDuringSessionMatchingConstraint> Registrator;
    };

    class TViolationDuringOrAfterSessionMatchingConstraint: public IMatchingConstraint {
        using TBase = IMatchingConstraint;

    public:
        using TBase::TBase;

        static TString GetTypeName();

        virtual bool Match(const TObjectEvent<TFullCompiledRiding>& objEvent, const TInstant timestamp) const override;
        virtual bool Match(TCarTagHistoryEventsSessionPtr objEventPtr, const TBillingEventsCompilation& eventsCompilation, const TInstant timestamp) const override;

    private:
        static TFactory::TRegistrator<TViolationDuringOrAfterSessionMatchingConstraint> Registrator;
    };

    class TViolationNotDuringServicingMatchingConstraint: public IMatchingConstraint {
        using TBase = IMatchingConstraint;

    public:
        using TBase::TBase;

        static TString GetTypeName();

        virtual bool Match(const TObjectEvent<TFullCompiledRiding>& objEvent, const TInstant timestamp) const override;
        virtual bool Match(TCarTagHistoryEventsSessionPtr objEventPtr, const TBillingEventsCompilation& eventsCompilation, const TInstant timestamp) const override;

    private:
        static TFactory::TRegistrator<TViolationNotDuringServicingMatchingConstraint> Registrator;
    };

    class IHasEventDuringSessionMatchingConstraint: public IMatchingConstraint {
        using TBase = IMatchingConstraint;

    protected:
        using TBase::TBase;

        virtual bool MatchImpl(const TObjectEvent<TFullCompiledRiding>& objEvent, const TInstant timestamp, const TSet<TString>& needleEvents) const final;
        virtual bool MatchImpl(TCarTagHistoryEventsSessionPtr objEventPtr, const TBillingEventsCompilation& eventsCompilation, const TInstant timestamp, const TSet<TString>& needleEvents) const final;
    };

    class THasRidingEventDuringSessionMatchingConstraint: public IHasEventDuringSessionMatchingConstraint {
        using TBase = IHasEventDuringSessionMatchingConstraint;

    public:
        using TBase::TBase;

        static TString GetTypeName();

        virtual bool Match(const TObjectEvent<TFullCompiledRiding>& objEvent, const TInstant timestamp) const override;
        virtual bool Match(TCarTagHistoryEventsSessionPtr objEventPtr, const TBillingEventsCompilation& eventsCompilation, const TInstant timestamp) const override;

    private:
        static TFactory::TRegistrator<THasRidingEventDuringSessionMatchingConstraint> Registrator;
    };

    class IHasEventBeforeViolationMatchingConstraint: public IMatchingConstraint {
        using TBase = IMatchingConstraint;

    protected:
        using TBase::TBase;

        virtual bool MatchImpl(const TObjectEvent<TFullCompiledRiding>& objEvent, const TInstant timestamp, const TSet<TString>& needleEvents) const final;
        virtual bool MatchImpl(TCarTagHistoryEventsSessionPtr objEventPtr, const TBillingEventsCompilation& eventsCompilation, const TInstant timestamp, const TSet<TString>& needleEvents) const final;
    };

    class THasRidingEventBeforeViolationMatchingConstraint: public IHasEventBeforeViolationMatchingConstraint {
        using TBase = IHasEventBeforeViolationMatchingConstraint;

    public:
        using TBase::TBase;

        static TString GetTypeName();

        virtual bool Match(const TObjectEvent<TFullCompiledRiding>& objEvent, const TInstant timestamp) const override;
        virtual bool Match(TCarTagHistoryEventsSessionPtr objEventPtr, const TBillingEventsCompilation& eventsCompilation, const TInstant timestamp) const override;

    private:
        static TFactory::TRegistrator<THasRidingEventBeforeViolationMatchingConstraint> Registrator;
    };

    class THasLocationChangeMatchingConstraint: public IMatchingConstraint {
        using TBase = IMatchingConstraint;

    public:
        using TBase::TBase;

        static TString GetTypeName();

        virtual bool Match(const TObjectEvent<TFullCompiledRiding>& objEvent, const TInstant timestamp) const override;
        virtual bool Match(TCarTagHistoryEventsSessionPtr objEventPtr, const TBillingEventsCompilation& eventsCompilation, const TInstant timestamp) const override;

    private:
        static TFactory::TRegistrator<THasLocationChangeMatchingConstraint> Registrator;
    };

    class IViolationInStateMatchingConstraint: public IMatchingConstraint {
        using TBase = IMatchingConstraint;

    protected:
        using TBase::TBase;

        virtual bool MatchImpl(const TObjectEvent<TFullCompiledRiding>& objEvent, const TInstant timestamp, const TSet<TString>& needleSessionStates) const final;
        virtual bool MatchImpl(TCarTagHistoryEventsSessionPtr objEventPtr, const TBillingEventsCompilation& eventsCompilation, const TInstant timestamp, const TSet<TString>& needleSessionStates) const final;
    };

    class TViolationInRidingMatchingConstraint: public IViolationInStateMatchingConstraint {
        using TBase = IViolationInStateMatchingConstraint;

    public:
        using TBase::TBase;

        static TString GetTypeName();

        virtual bool Match(const TObjectEvent<TFullCompiledRiding>& objEvent, const TInstant timestamp) const override;
        virtual bool Match(TCarTagHistoryEventsSessionPtr objEventPtr, const TBillingEventsCompilation& eventsCompilation, const TInstant timestamp) const override;

    private:
        static TFactory::TRegistrator<TViolationInRidingMatchingConstraint> Registrator;
    };
}
