#include <drive/backend/sessions/matcher/session_matcher.h>

#include <drive/backend/compiled_riding/compiled_riding.h>
#include <drive/backend/database/history/event.h>
#include <drive/backend/database/history/session.h>
#include <drive/backend/database/history/session_builder.h>
#include <drive/backend/tags/tags.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/generic/guid.h>
#include <util/generic/map.h>
#include <util/string/builder.h>

// NB. Consider to use EnvironmentGenerator

namespace {
    static const TMap<TString, TVector<TString>> MatchingRules = {
        {"12_09_2", {"violation_during_session", "has_riding_before_violation"}},
        {"12_16_5", {"violation_during_or_after_session", "has_riding_before_violation", "has_location_change"}}
    };

    static const ui32 MinRideDistanceMeters = 10;
    static const ui32 RejectedSessionsSkipLimit = 4;

    static const TString TargetCarId = "car";

    static auto GetMatchingConstraintsGroup(const TString& articleCode) {
        NDrive::NSession::TMatchingOptions options;
        options.RejectedSessionsSkipLimit = RejectedSessionsSkipLimit;
        options.MinRideDistanceMeters = MinRideDistanceMeters;

        TSet<TString> constraintNames;
        auto it = MatchingRules.equal_range(articleCode);
        for (auto pairPtr = it.first; pairPtr != it.second; ++pairPtr) {
            for (auto&& constraintName: pairPtr->second) {
                constraintNames.emplace(constraintName);
            }
        }

        return NDrive::NSession::TMatchingConstraintsGroup::Construct(nullptr, options, constraintNames);
    }

    enum ERideTraits : ui32 {
        Empty = 0,
        Acceptance = 1 << 0,
        Riding = 1 << 1,
    };

    static auto ConstructRideEvents(const ui32 rideTraits, const TInstant start, const TDuration rideDuration = TDuration::Seconds(180)) {
        static const TDuration minDuration = TDuration::Seconds(120);
        if (rideDuration < minDuration) {
            ythrow yexception() << "Invalid test ride duration: it's supposed to be greater than " << minDuration.Seconds() << " seconds";
        }

        TVector<TCompiledLocalEvent> events;
        events.emplace_back(TCompiledLocalEvent().SetInstant(start).SetHistoryAction(EObjectHistoryAction::SetTagPerformer).SetTagName(::ToString(NDrive::NSession::ESessionState::Reservation)));

        if (rideTraits & ERideTraits::Acceptance || rideTraits & ERideTraits::Riding) {
            events.emplace_back(TCompiledLocalEvent().SetInstant(start + TDuration::Seconds(10)).SetHistoryAction(EObjectHistoryAction::TagEvolve).SetTagName(::ToString(NDrive::NSession::ESessionState::Acceptance)));
            if (rideTraits & ERideTraits::Riding) {
                events.emplace_back(TCompiledLocalEvent().SetInstant(start + TDuration::Seconds(60)).SetHistoryAction(EObjectHistoryAction::TagEvolve).SetTagName(::ToString(NDrive::NSession::ESessionState::Riding)));
            }
            events.emplace_back(TCompiledLocalEvent().SetInstant(start + TDuration::Seconds(120)).SetHistoryAction(EObjectHistoryAction::TagEvolve).SetTagName(::ToString(NDrive::NSession::ESessionState::Reservation)));
        }

        events.emplace_back(TCompiledLocalEvent().SetInstant(start + rideDuration).SetHistoryAction(EObjectHistoryAction::DropTagPerformer).SetTagName(::ToString(NDrive::NSession::ESessionState::Reservation)));
        return events;
    }

    static auto ConstructCompiledRide(const ui32 rideTraits, const ui32 mileageMeters, const TInstant start, const TDuration rideDuration = TDuration::Seconds(180)) {
        TFullCompiledRiding ride;

        TVector<TCompiledLocalEvent> events = ConstructRideEvents(rideTraits, start, rideDuration);
        ride.SetLocalEvents(events);

        auto finish = events.back().GetInstant();
        ride.SetStartInstant(start);
        ride.SetFinishInstant(finish);

        TString targetCarId = TargetCarId;
        ride.SetObjectId(targetCarId);

        TString targetSessionId = ICommonOffer::CreateOfferId();
        ride.SetSessionId(targetSessionId);

        TSnapshotsDiff snapshotsDiff;
        snapshotsDiff.SetMileage(mileageMeters * 0.001);
        ride.SetSnapshotsDiff(snapshotsDiff);

        return TObjectEvent(ride, EObjectHistoryAction::Add, start, "", "", "");
    }

    static auto GetThreeRidesPreset() {
        NDrive::NSession::TMatchingConstraintsGroup::TFullCompiledRideContainer compiledRides;

        const auto now = TInstant::Now();

        compiledRides.emplace_back(ConstructCompiledRide(ERideTraits::Riding, 1000, now, TDuration::Seconds(150)));
        compiledRides.emplace_back(ConstructCompiledRide(ERideTraits::Riding, 1000, now + TDuration::Seconds(200), TDuration::Seconds(150)));
        compiledRides.emplace_back(ConstructCompiledRide(ERideTraits::Riding, 1000, now + TDuration::Seconds(400), TDuration::Seconds(150)));

        return compiledRides;
    }
}

Y_UNIT_TEST_SUITE(SessionMatcher) {
    Y_UNIT_TEST(MatchSuccess_12_09_2) {
        auto constraints = GetMatchingConstraintsGroup("12_09_2");
        UNIT_ASSERT(constraints);

        auto compiledRides = GetThreeRidesPreset();

        NDrive::NSession::TSessionBindingInfo info;
        UNIT_ASSERT(constraints->MatchCompiledRidesSession(compiledRides, TargetCarId, compiledRides[1].GetStartInstant() + TDuration::Seconds(90), info));
        UNIT_ASSERT_STRINGS_EQUAL(compiledRides[1].GetSessionId(), info.GetSessionId());
    }

    Y_UNIT_TEST(MatchFailed_AfterRide_12_09_2) {
        auto constraints = GetMatchingConstraintsGroup("12_09_2");
        UNIT_ASSERT(constraints);

        auto compiledRides = GetThreeRidesPreset();

        NDrive::NSession::TSessionBindingInfo info;
        UNIT_ASSERT(!constraints->MatchCompiledRidesSession(compiledRides, TargetCarId, compiledRides[1].GetStartInstant() + TDuration::Seconds(180), info));
    }

    Y_UNIT_TEST(MatchFailed_BeforeRide_12_09_2) {
        auto constraints = GetMatchingConstraintsGroup("12_09_2");
        UNIT_ASSERT(constraints);

        auto compiledRides = GetThreeRidesPreset();

        NDrive::NSession::TSessionBindingInfo info;
        UNIT_ASSERT(!constraints->MatchCompiledRidesSession(compiledRides, TargetCarId, compiledRides[1].GetStartInstant() - TDuration::Seconds(5), info));
    }

    Y_UNIT_TEST(MatchFailed_InAcceptanceState_12_09_2) {
        auto constraints = GetMatchingConstraintsGroup("12_09_2");
        UNIT_ASSERT(constraints);

        auto compiledRides = GetThreeRidesPreset();

        NDrive::NSession::TSessionBindingInfo info;
        UNIT_ASSERT(!constraints->MatchCompiledRidesSession(compiledRides, TargetCarId, compiledRides[1].GetStartInstant() + TDuration::Seconds(30), info));
    }

    Y_UNIT_TEST(MatchFailed_InParkingState_12_09_2) {
        auto constraints = GetMatchingConstraintsGroup("12_09_2");
        UNIT_ASSERT(constraints);

        auto compiledRides = GetThreeRidesPreset();

        NDrive::NSession::TSessionBindingInfo info;
        UNIT_ASSERT(!constraints->MatchCompiledRidesSession(compiledRides, TargetCarId, compiledRides[1].GetStartInstant() + TDuration::Seconds(5), info));
    }

    Y_UNIT_TEST(MatchFailed_WithoutRiding_12_09_2) {
        auto constraints = GetMatchingConstraintsGroup("12_09_2");
        UNIT_ASSERT(constraints);

        NDrive::NSession::TMatchingConstraintsGroup::TFullCompiledRideContainer compiledRides;

        const auto now = TInstant::Now();

        compiledRides.emplace_back(ConstructCompiledRide(ERideTraits::Riding, 1000, now, TDuration::Seconds(150)));
        compiledRides.emplace_back(ConstructCompiledRide(ERideTraits::Empty, 1000, now + TDuration::Seconds(200), TDuration::Seconds(150)));
        compiledRides.emplace_back(ConstructCompiledRide(ERideTraits::Riding, 1000, now + TDuration::Seconds(400), TDuration::Seconds(150)));

        NDrive::NSession::TSessionBindingInfo info;
        UNIT_ASSERT(!constraints->MatchCompiledRidesSession(compiledRides, TargetCarId, compiledRides[1].GetStartInstant() + TDuration::Seconds(90), info));
    }

    Y_UNIT_TEST(MatchSuccess_DuringSession_12_16_5) {
        auto constraints = GetMatchingConstraintsGroup("12_16_5");
        UNIT_ASSERT(constraints);

        auto compiledRides = GetThreeRidesPreset();

        NDrive::NSession::TSessionBindingInfo info;
        UNIT_ASSERT(constraints->MatchCompiledRidesSession(compiledRides, TargetCarId, compiledRides[1].GetStartInstant() + TDuration::Seconds(90), info));
        UNIT_ASSERT_STRINGS_EQUAL(compiledRides[1].GetSessionId(), info.GetSessionId());
    }

    Y_UNIT_TEST(MatchSuccess_AfterSession_12_16_5) {
        auto constraints = GetMatchingConstraintsGroup("12_16_5");
        UNIT_ASSERT(constraints);

        auto compiledRides = GetThreeRidesPreset();

        NDrive::NSession::TSessionBindingInfo info;
        UNIT_ASSERT(constraints->MatchCompiledRidesSession(compiledRides, TargetCarId, compiledRides[1].GetStartInstant() + TDuration::Seconds(180), info));
        UNIT_ASSERT_STRINGS_EQUAL(compiledRides[1].GetSessionId(), info.GetSessionId());
    }

    Y_UNIT_TEST(MatchSuccess_AfterSession_BeforeNextRiding_12_16_5) {
        auto constraints = GetMatchingConstraintsGroup("12_16_5");
        UNIT_ASSERT(constraints);

        auto compiledRides = GetThreeRidesPreset();

        NDrive::NSession::TSessionBindingInfo info;
        UNIT_ASSERT(constraints->MatchCompiledRidesSession(compiledRides, TargetCarId, compiledRides[1].GetStartInstant() + TDuration::Seconds(30), info));
        UNIT_ASSERT_STRINGS_EQUAL(compiledRides[0].GetSessionId(), info.GetSessionId());
    }

    Y_UNIT_TEST(MatchSuccess_AfterSession_NoLocationChange_12_16_5) {
        auto constraints = GetMatchingConstraintsGroup("12_16_5");
        UNIT_ASSERT(constraints);

        NDrive::NSession::TMatchingConstraintsGroup::TFullCompiledRideContainer compiledRides;

        const auto now = TInstant::Now();
        const auto emptyRideMileage = MinRideDistanceMeters / 2;

        compiledRides.emplace_back(ConstructCompiledRide(ERideTraits::Riding, 1000, now, TDuration::Seconds(150)));
        compiledRides.emplace_back(ConstructCompiledRide(ERideTraits::Riding, emptyRideMileage, now + TDuration::Seconds(200), TDuration::Seconds(150)));
        compiledRides.emplace_back(ConstructCompiledRide(ERideTraits::Riding, 1000, now + TDuration::Seconds(400), TDuration::Seconds(150)));

        NDrive::NSession::TSessionBindingInfo info;
        UNIT_ASSERT(constraints->MatchCompiledRidesSession(compiledRides, TargetCarId, compiledRides[1].GetStartInstant() + TDuration::Seconds(90), info));
        UNIT_ASSERT_STRINGS_EQUAL(compiledRides[0].GetSessionId(), info.GetSessionId());
    }

    Y_UNIT_TEST(MatchSuccess_AfterSession_TwoSkips_12_16_5) {
        auto constraints = GetMatchingConstraintsGroup("12_16_5");
        UNIT_ASSERT(constraints);

        NDrive::NSession::TMatchingConstraintsGroup::TFullCompiledRideContainer compiledRides;

        const auto now = TInstant::Now();
        const auto emptyRideMileage = MinRideDistanceMeters / 2;

        compiledRides.emplace_back(ConstructCompiledRide(ERideTraits::Riding, 1000, now, TDuration::Seconds(150)));
        compiledRides.emplace_back(ConstructCompiledRide(ERideTraits::Empty, 1000, now + TDuration::Seconds(200), TDuration::Seconds(150)));
        compiledRides.emplace_back(ConstructCompiledRide(ERideTraits::Riding, emptyRideMileage, now + TDuration::Seconds(400), TDuration::Seconds(150)));

        NDrive::NSession::TSessionBindingInfo info;
        UNIT_ASSERT(constraints->MatchCompiledRidesSession(compiledRides, TargetCarId, compiledRides[2].GetStartInstant() + TDuration::Seconds(90), info));
        UNIT_ASSERT_STRINGS_EQUAL(compiledRides[0].GetSessionId(), info.GetSessionId());
    }

    Y_UNIT_TEST(MatchFailed_TooMuchSkips_12_16_5) {
        auto constraints = GetMatchingConstraintsGroup("12_16_5");
        UNIT_ASSERT(constraints);

        NDrive::NSession::TMatchingConstraintsGroup::TFullCompiledRideContainer compiledRides;

        const auto now = TInstant::Now();
        const auto emptyRideMileage = MinRideDistanceMeters / 2;

        compiledRides.emplace_back(ConstructCompiledRide(ERideTraits::Riding, 1000, now, TDuration::Seconds(150)));

        for (size_t idx = 0; idx < 1 + RejectedSessionsSkipLimit / 2; ++idx) {
            auto rideStart = now + (idx + 1) * TDuration::Seconds(200);
            compiledRides.emplace_back(ConstructCompiledRide(ERideTraits::Empty, 1000, rideStart, TDuration::Seconds(150)));
            compiledRides.emplace_back(ConstructCompiledRide(ERideTraits::Riding, emptyRideMileage, rideStart + TDuration::Seconds(200), TDuration::Seconds(150)));
        }

        NDrive::NSession::TSessionBindingInfo info;
        UNIT_ASSERT(!constraints->MatchCompiledRidesSession(compiledRides, TargetCarId, compiledRides[compiledRides.size() - 1].GetStartInstant() + TDuration::Seconds(90), info));
    }
}
