#include "settings.h"

TString TSettingConditionConstructor::BuildCondition(const TSet<TString>& ids, NDrive::TEntitySession& session) {
    return "setting_key IN (" + session->Quote(ids) + ")";
}

NStorage::TTableRecord TSettingConditionConstructor::BuildCondition(const TString& id) {
    NStorage::TTableRecord trCondition;
    trCondition.Set("setting_key", id);
    return trCondition;
}

template <class T>
NStorage::TTableRecord TSettingConditionConstructor::BuildCondition(const T& object) {
    return BuildCondition(object.GetKey());
}

TSettingsDB::TSettingsDB(const IHistoryContext& context, const TSettingsConfig& config)
    : TBase(context, config.GetHistoryConfig())
    , Prefix(config.GetPrefix())
    , PrefixNewBehaviour(config.GetPrefixNewBehaviour())
    , Propositions(GetHistoryManager().GetContext(), config.GetPropositionsConfig())
{
    Y_ENSURE_BT(Start());
}

TSettingsDB::~TSettingsDB() {
    if (!Stop()) {
        ERROR_LOG << "cannot stop SettingsDB" << Endl;
    }
}

template <class T>
bool TSettingsDB::GetValueImpl(TStringBuf key, T& result, TInstant reqActuality) const {
    auto keys = NContainer::Scalar(key);
    bool found = false;
    const auto action = [&found, &result](const TSetting& setting) {
        found = TryFromString(setting.GetValue(), result);
    };

    if (!ForObjectsList(action, reqActuality, &keys)) {
        return false;
    }

    return found;
}

bool TSettingsDB::GetValueStr(TStringBuf key, TString& result, TInstant reqActuality) const {
    if (PrefixNewBehaviour) {
        return GetValueNew(key, result, reqActuality);
    } else {
        return GetValueOld(key, result, reqActuality);
    }
}

bool TSettingsDB::GetValueOld(TStringBuf key, TString& result, TInstant reqActuality) const {
    if (Prefix) {
        return GetValueImpl(Prefix + key, result, reqActuality);
    } else {
        return GetValueImpl(key, result, reqActuality);
    }
}

bool TSettingsDB::GetValueNew(TStringBuf key, TString& result, TInstant reqActuality) const {
    if (Prefix && GetValueImpl(Prefix + key, result, reqActuality)) {
        return true;
    }
    if (GetValueImpl(key, result, reqActuality)) {
        return true;
    }
    return false;
}

bool TSettingsDB::SetValueWithRevision(const TSetting& setting, const TString& userId, NDrive::TEntitySession& session) const {
    auto key = PrefixNewBehaviour ? setting.GetKey() : (Prefix + setting.GetKey());
    TSetting temp(std::move(key), setting.GetValue());
    setting.HasRevision() && setting.GetRevision()  ? void(temp.SetRevision(setting.GetRevision())) : void();
    bool newProccessed = false;
    return !newProccessed ? SetValues({setting}, userId) : UpsertObject(temp, userId, session);
}

bool TSettingsDB::SetValues(const TVector<TSetting>& values, const TString& userId) const {
    NStorage::ITransaction::TPtr transaction = HistoryCacheDatabase->CreateTransaction(false);
    NStorage::ITableAccessor::TPtr table = HistoryCacheDatabase->GetTable(TSetting::GetTableName());
    NDrive::TEntitySession session(transaction);
    for (auto&& i : values) {
        auto key = PrefixNewBehaviour ? i.GetKey() : (Prefix + i.GetKey());
        TSetting setting(std::move(key), i.GetValue());
        const NStorage::TTableRecord record = setting.SerializeToTableRecord();
        const NStorage::TTableRecord unique = setting.SerializeUniqueToTableRecord();

        NStorage::TObjectRecordsSet<TSetting> records;
        bool isUpdate = false;
        bool newProccessed = false;
        if (!newProccessed) {
            if (!table->Upsert(record, transaction, unique, &isUpdate, &records)) {
                ERROR_LOG << transaction->GetErrors().GetStringReport() << Endl;
                return false;
            }
            if (!HistoryManager->AddHistory(records.GetObjects(), userId, isUpdate ? EObjectHistoryAction::UpdateData : EObjectHistoryAction::Add, session)) {
                ERROR_LOG << transaction->GetErrors().GetStringReport() << Endl;
                return false;
            }
        } else {
            if (!UpsertObject(setting, userId, session)) {
                return false;
            }
        }
    }
    if (!transaction->Commit()) {
        ERROR_LOG << transaction->GetErrors().GetStringReport() << Endl;
        return false;
    }
    return true;
}

bool TSettingsDB::GetHistory(const TInstant since, TVector<TAtomicSharedPtr<TObjectEvent<TSetting>>>& result, const TInstant reqActuality) const {
    Y_UNUSED(reqActuality);
    auto session = GetHistoryManager().BuildSession(true);
    auto optionalEvents = GetHistoryManager().GetEvents({}, since, session);
    if (!optionalEvents) {
        ERROR_LOG << "cannot GetEvents: " << session.GetStringReport() << Endl;
        return false;
    }

    for (auto&& i : *optionalEvents) {
        if (PrefixNewBehaviour || i.GetKey().StartsWith(Prefix)) {
            result.emplace_back(MakeAtomicShared<TObjectEvent<TSetting>>(i));
        }
    }
    return true;
}

bool TSettingsDB::HasValues(const TSet<TString>& keys, TSet<TString>& existKeys, const TInstant reqActuality) const {
    existKeys.clear();
    if (keys.empty())
        return true;

    const auto action = [&keys, &existKeys](const TSetting& setting) {
        if (keys.contains(setting.GetKey())) {
            existKeys.emplace(setting.GetKey());
        }
    };

    return ForObjectsList(action, reqActuality, &keys);
}

bool TSettingsDB::GetAllSettings(TVector<TSetting>& result, const TInstant reqActuality) const {
    const auto action = [this, &result](const TSetting& setting) {
        if (!Prefix || PrefixNewBehaviour) {
            result.emplace_back(setting);
        } else if (setting.GetKey().StartsWith(Prefix)) {
            result.emplace_back(setting);
            result.back().SetKey(setting.GetKey().substr(Prefix.size()));
        }
    };
    return ForObjectsList(action, reqActuality);
}

bool TSettingsDB::RemoveKeys(const TVector<TString>& keys, const TString& userId) const {
    if (keys.empty())
        return true;
    NStorage::ITransaction::TPtr transaction = HistoryCacheDatabase->CreateTransaction(false);
    NStorage::ITableAccessor::TPtr table = HistoryCacheDatabase->GetTable(TSetting::GetTableName());
    TVector<TString> keysPrefixed = keys;
    if (!PrefixNewBehaviour) {
        for (auto&& i : keysPrefixed) {
            i = Prefix + i;
        }
    }
    const TString condition = "setting_key IN (" + transaction->Quote(keysPrefixed) + ")";
    TRecordsSet records;
    if (!table->RemoveRow(condition, transaction, &records)) {
        ERROR_LOG << transaction->GetErrors().GetStringReport() << Endl;
        return false;
    }

    TVector<TSetting> settings;
    for (auto&& i : records) {
        TSetting setting;
        if (setting.DeserializeFromTableRecord(i, nullptr)) {
            settings.push_back(setting);
        }
    }

    NDrive::TEntitySession session(transaction);

    if (!HistoryManager->AddHistory(settings, userId, EObjectHistoryAction::Remove, session)) {
        ERROR_LOG << transaction->GetErrors().GetStringReport() << Endl;
        return false;
    }
    if (!transaction->Commit()) {
        ERROR_LOG << transaction->GetErrors().GetStringReport() << Endl;
        return false;
    }
    return true;
}
