#include "helpers.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/drivematics/signals/description.h>
#include <drive/backend/signalq/signals/tag.h>

#include <rtline/library/json/field.h>

#include <iterator>

DECLARE_FIELDS_JSON_SERIALIZER(NDrive::TTraceSignalsQuerySettings);

namespace NDrive {

namespace {
    TTagHistoryEvents GetFilteredTraceSignalsFromRawSignals(
            TJsonReport::TGuard& g,
            NDrive::TEntitySession& tx,
            const TTraceTagsManager& traceTagsManager,
            const TTagHistoryEvents& sessionTagsHistoryEvents,
            const TTraceSignalsQueryParams& params) {
        if (sessionTagsHistoryEvents.empty()) {
            return {};
        }

        TVector<TString> checkingTagIds;
        checkingTagIds.reserve(sessionTagsHistoryEvents.size());
        for (const auto& sessionTagsHistoryEvent : sessionTagsHistoryEvents) {
            if (const auto tag = sessionTagsHistoryEvent.GetTagAs<TSignalqEventTraceTag>()) {
                checkingTagIds.push_back(sessionTagsHistoryEvent.GetTagId());
            }
        }

        if (checkingTagIds.empty()) {
            return sessionTagsHistoryEvents;
        }

        TTagEventsManager::TQueryOptions checkingSessionTagsHistoryQuery;
        checkingSessionTagsHistoryQuery
            .SetTagIds(std::move(checkingTagIds))
            .SetActions({EObjectHistoryAction::TagEvolve, EObjectHistoryAction::UpdateData})
            .SetObjectIds(std::cref(params.GetVisibleSessionIds()))
            .SetDescending(true);

        TTagHistoryEvents events;
        {
            auto eg = g.BuildEventGuard("FilterTraceSignalsQuery");
            auto optionalEvents = traceTagsManager.GetEvents({ 0, Nothing() }, {params.GetPeriodRange().From, Nothing()}, tx, checkingSessionTagsHistoryQuery);
            R_ENSURE(optionalEvents, {}, "can't get events to filter", tx);
            events = std::move(*optionalEvents);
            std::reverse(events.begin(), events.end());
        }

        TMap<TString, TMaybe<TObjectEvent<TConstDBTag>>> tagIdToUpdatedSignal;
        auto signalqDeletedTagName = NSignalq::GetDeletedTagName();
        for (auto&& event : events) {
            if (!event.GetTagAs<TSignalqEventTraceTag>()) {
                continue;
            }

            const auto tagId = event.GetTagId();
            if (event->GetName() == signalqDeletedTagName) {
                tagIdToUpdatedSignal[std::move(tagId)] = Nothing();
            } else if (!tagIdToUpdatedSignal.contains(tagId)) {
                tagIdToUpdatedSignal[std::move(tagId)] = std::move(event);
            }
        }

        TTagHistoryEvents result;
        result.reserve(sessionTagsHistoryEvents.size());
        for (const auto& sessionTagsHistoryEvent : sessionTagsHistoryEvents) {
            auto it = tagIdToUpdatedSignal.find(sessionTagsHistoryEvent.GetTagId());
            if (it == tagIdToUpdatedSignal.end()) {
                result.push_back(sessionTagsHistoryEvent);
            } else if (it->second && !params.IsSupportMode()) {
                auto& ev = *it->second;
                ev.SetHistoryTimestamp(sessionTagsHistoryEvent.GetHistoryTimestamp());
                result.push_back(std::move(ev));
            }
        }

        return result;
    }
} // namespace

namespace NSignalq {
    TString GetDeletedTagName() {
        return NDrive::GetServer().GetSettings().GetValue<TString>("signalq.signalq_events.deleted_tag").GetOrElse("signalq_deleted_event_tag");
    }

    TString GetSupportPreapprovedTagName() {
        return NDrive::GetServer().GetSettings().GetValue<TString>("signalq.signalq_events.support_preapproved_tag").GetOrElse("signalq_support_preapproved_tag");
    }

    TMap<TString, TString> GetEventToTagName() {
        auto descriptions = GetSignalDescriptions();

        TMap<TString, TString> result;
        for (auto&& description : descriptions) {
            result[description.GetType()] = description.GetName();
        }
        return result;
    }

    TVector<NDrivematics::TSignalDescription> GetSignalDescriptions() {
        auto value = NDrive::GetServer().GetSettings().GetJsonValue(SignalqSignalsDescriptionSettingName);
        if (!value.IsDefined()) {
            return {};
        }
        R_ENSURE(value.IsMap(), HTTP_INTERNAL_SERVER_ERROR, "setting " << SignalqSignalsDescriptionSettingName << " is not a map");

        TVector<NDrivematics::TSignalDescription> result;
        R_ENSURE(NJson::ParseField(value["descriptions"], result), HTTP_INTERNAL_SERVER_ERROR, "Can't convert json into signalq signals descriptions");
        return result;
    }

    void ProcessTagResolution(
            TDBTag& tag,
            const NDrive::NSignalq::EEventResolution resolution,
            const NDrive::IServer* server,
            const TUserPermissions::TPtr& permissions,
            NDrive::TEntitySession& tx
    ) {
        using NDrive::NSignalq::EEventResolution;

        const auto& driveApi = Yensured(server)->GetDriveAPI();
        const auto& traceTagsManager = driveApi->GetTagsManager().GetTraceTags();
        const auto& tagsMeta = driveApi->GetTagsManager().GetTagsMeta();

        const auto eventToTagName = GetEventToTagName();
        const auto signalqDeletedTagName = GetDeletedTagName();
        const auto signalqSupportPreapprovedTag = GetSupportPreapprovedTagName();

        auto signalTag = tag.MutableTagAs<TSignalqEventTraceTag>();
        R_ENSURE(signalTag, HTTP_BAD_REQUEST, "Tag with id " + tag.GetTagId() + "is not signalqEventTraceTag");
        switch (resolution) {
            case EEventResolution::SupportApprove: {
                R_ENSURE(signalTag->GetName() == signalqSupportPreapprovedTag, HTTP_BAD_REQUEST, "can't evolve not preapproved tag name");
                const auto& eventType = signalTag->GetEvent().GetType();
                auto itEventToTagName = eventToTagName.find(eventType);
                R_ENSURE(itEventToTagName != eventToTagName.end(), HTTP_INTERNAL_SERVER_ERROR, "cannot match event_type with existing");

                const auto newTagName = itEventToTagName->second;
                auto newTag = tagsMeta.CreateTag(newTagName);
                R_ENSURE(newTag, HTTP_INTERNAL_SERVER_ERROR, "cannot create tag " + newTagName, tx);

                auto impl = std::dynamic_pointer_cast<TSignalqEventTraceTag>(newTag);
                R_ENSURE(impl, HTTP_INTERNAL_SERVER_ERROR, "cannot Cast " << newTag->GetName() << " as SignalqEventTraceTag", tx);
                impl->SetSerialNumber(std::move(signalTag->GetSerialNumber()));
                impl->SetEvent(std::move(signalTag->GetEvent()));
                impl->SetVisible(true);
                R_ENSURE(
                    traceTagsManager.EvolveTag(tag, newTag, *permissions, server, tx),
                    HTTP_INTERNAL_SERVER_ERROR,
                    "can not evolve signalq trace tag to approved"
                );
                return;
            }
            case EEventResolution::Delete: {
                if (signalTag->GetName() == signalqDeletedTagName) {
                    return;
                }

                auto newTag = tagsMeta.CreateTag(signalqDeletedTagName);
                R_ENSURE(newTag, HTTP_INTERNAL_SERVER_ERROR, "cannot create tag " + signalqDeletedTagName, tx);

                auto impl = std::dynamic_pointer_cast<TSignalqEventTraceTag>(newTag);
                R_ENSURE(impl, HTTP_INTERNAL_SERVER_ERROR, "cannot Cast " << newTag->GetName() << " as SignalqEventTraceTag", tx);
                impl->SetSerialNumber(std::move(signalTag->GetSerialNumber()));
                impl->SetEvent(std::move(signalTag->GetEvent()));
                impl->SetVisible(false);

                R_ENSURE(
                    traceTagsManager.EvolveTag(tag, newTag, *permissions, server, tx),
                    HTTP_INTERNAL_SERVER_ERROR,
                    "can not evolve signalq trace tag to deleted"
                );
                return;
            }
            default: {
                signalTag->SetResolution(ToString(resolution));
                R_ENSURE(
                    traceTagsManager.UpdateTagData(tag, permissions->GetUserId(), tx),
                    HTTP_INTERNAL_SERVER_ERROR,
                    "cannot update tag"
                );
            }
        }
    }
} // namespace NSignalq

    TTraceSignalsResultData GetTraceSignals(
            TJsonReport::TGuard& g,
            NDrive::TEntitySession& tx,
            const TTraceTagsManager& traceTagsManager,
            TTraceSignalsQueryParams&& params) {
        TTagEventsManager::TQueryOptions sessionTagsHistoryQuery;
        sessionTagsHistoryQuery
            .SetTags(std::cref(params.GetReportedSignalsNames()))
            .SetObjectIds(std::cref(params.GetVisibleSessionIds()))
            .SetActions({EObjectHistoryAction::Add, EObjectHistoryAction::TagEvolve});

        if (params.HasSignalId()) {
            sessionTagsHistoryQuery.SetTagIds({params.GetSignalIdRef()});
        } else {
            sessionTagsHistoryQuery.SetDescending(true);
        }

        if (params.HasPageSize()) {
            params.SetLimitPerQuery(Max(params.GetPageSizeRef(), params.GetLimitPerQuery()));
            sessionTagsHistoryQuery.SetLimit(params.GetLimitPerQuery() + 1);
        }

        TTraceSignalsResultData result;
        TTagHistoryEvents resultTraceSignals;
        TMaybe<ui64> currentCursor = params.OptionalSessionTagHistoryCursor();
        ui64 rawEventsSize = 0;
        for (ui32 i = 0; i < params.GetMaxQueriesCount(); ++i) {
            TTagHistoryEvents events;
            {
                auto eg = g.BuildEventGuard("SelectTraceSignalsQuery");
                auto optionalEvents = traceTagsManager.GetEvents({ 0, currentCursor }, params.GetPeriodRange(), tx, sessionTagsHistoryQuery);
                R_ENSURE(optionalEvents, {}, "can't get events", tx);
                events = std::move(*optionalEvents);
                rawEventsSize = events.size();
            }
            auto filteredTraceSignals = GetFilteredTraceSignalsFromRawSignals(
                g, tx, traceTagsManager, events, params);
            auto filteredTraceSignalsSize = filteredTraceSignals.size();

            if (filteredTraceSignalsSize > 0) {
                resultTraceSignals.insert(resultTraceSignals.end(), std::make_move_iterator(filteredTraceSignals.begin()), std::make_move_iterator(filteredTraceSignals.end()));
            }

            if (!events.empty()) {
                currentCursor = events.back().GetHistoryEventId();
            }

            if (params.HasSignalId()
                    || !params.HasPageSize()
                    || rawEventsSize < params.GetLimitPerQuery()
                    || params.GetPageSizeRef() + 1 <= filteredTraceSignalsSize) {
                break;
            }
        }

        result.SetLastEventsCursor(currentCursor);
        result.SetTraceSignals(std::move(resultTraceSignals));
        return result;
    }

} // namespace NDrive
