#include "tag.h"
#include "helpers.h"

#include <drive/library/cpp/taxi/signalq_drivematics_api/utils.h>

NDrive::ITag::TFactory::TRegistrator<TSignalqEventTraceTag> SignalqEventTraceTagRegistrator(TSignalqEventTraceTag::TypeName);
TTagDescription::TFactory::TRegistrator<TSignalqEventTraceTag::TDescription> SignalqEventTraceTagDescriptionRegistrator(TSignalqEventTraceTag::TypeName);

NJson::TJsonValue TSignalqEventTraceTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue json = TBase::DoSerializeMetaToJson();
    return json;
}

bool TSignalqEventTraceTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& json) {
    return TBase::DoDeserializeMetaFromJson(json);
}

NDrive::TScheme TSignalqEventTraceTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    return result;
}

NJson::TJsonValue TSignalqEventTraceTag::GetDetailReport() const {
    NJson::TJsonValue result;
    NJson::InsertNonNull(result, "resolution", GetResolution());
    return result;
}

TString TSignalqEventTraceTag::GetSignalNameOverride() const {
    auto supportPreapprovedTagName = NDrive::NSignalq::GetSupportPreapprovedTagName();
    if (GetName() == supportPreapprovedTagName) {
        return Event.GetType();
    }
    return {};
}

void TSignalqEventTraceTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    NJson::InsertField(json, "event", NJson::ToJson(Event));
    NJson::InsertField(json, "serial_number", SerialNumber);
}

bool TSignalqEventTraceTag::DoSpecialDataFromJson(const NJson::TJsonValue& specialJson, TMessagesCollector* errors) {
    if (!TBase::DoSpecialDataFromJson(specialJson, errors)) {
        return false;
    }
    return NJson::ParseField(specialJson["event"], Event, errors)
        && NJson::ParseField(specialJson["serial_number"], SerialNumber, errors)
    ;
}

NDrive::TScheme TSignalqEventTraceTag::GetScheme(const NDrive::IServer* /*server*/) const {
    auto result = NDrive::TScheme();
    result.Add<TFSJson>("event", "Событие");
    result.Add<TFSText>("serial_number", "Serial number");
    result.Add<TFSText>("resolution", "Resolution");
    return result;
}

TString TSignalqEventTraceTag::MakeSerialNumberEventId() const {
    return NDrive::NSignalq::MakeSerialNumberEventId(SerialNumber, Event.GetId());
}

bool TSignalqEventTraceTag::OnBeforeAdd(const TString& entityId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) {
    Y_UNUSED(userId);
    auto start = Event.GetAt();
    if (!start) {
        start = Now();
    }
    auto entityUuid = TUuid::TryParse(entityId);
    if (!entityUuid) {
        tx.SetErrorInfo("CommonSignalTag::OnBeforeAdd", TStringBuilder() << "cannot parse " << entityId << " as UUID");
        return false;
    }
    return Fill(*entityUuid, start, *server, tx);
}

bool TSignalqEventTraceTag::OnBeforeUpdate(const TDBTag& self, ITag::TPtr to, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const {
    auto impl = std::dynamic_pointer_cast<TSignalqEventTraceTag>(to);
    if (!impl) {
        return true;
    }
    return impl->OnBeforeAdd(self.GetObjectId(), userId, server, tx);
}
