#include "manager.h"

#include <rtline/util/algorithm/container.h>

#include <util/string/vector.h>


TAudioteleCallsManager::TAudioteleCallEventsManager::TAudioteleCallEventsManager(NStorage::IDatabase::TPtr database)
    : TBase(database)
{
}

TString TAudioteleCallsManager::TAudioteleCallEventsManager::GetTableName() const {
    return TAudioteleCallEvent::GetTableName();
}

TAudioteleCallEvent::TId TAudioteleCallsManager::TAudioteleCallEventsManager::GetMainId(const TAudioteleCallEvent& e) const {
    return e.GetId();
}

TMaybe<TVector<TAudioteleCallEvent>> TAudioteleCallsManager::TAudioteleCallEventsManager::GetObjects(NDrive::TEntitySession& session, const TSet<TAudioteleCallEvent::EAction>& filterActions, const TSet<TString>& callIds, const ui32 limit) const {
    NSQL::TQueryOptions options(limit, /* descending = */ true);
    options.SetOrderBy({ "id" });
    if (!filterActions.empty()) {
        TSet<TString> actions;
        Transform(filterActions.begin(), filterActions.end(), std::inserter(actions, actions.begin()), [](const auto& item) { return ToString(item); });
        options.SetGenericCondition("action", actions);
    }
    if (callIds) {
        options.SetGenericCondition("related_call_id", callIds);
    }
    return Fetch(session, options);
}


TAudioteleCallsManager::TAudioteleCallTrackManager::TAudioteleCallTrackManager(NStorage::IDatabase::TPtr database)
    : TBase(database)
{
}

TString TAudioteleCallsManager::TAudioteleCallTrackManager::GetTableName() const {
    return TAudioteleCallTrack::GetTableName();
}

TAudioteleCallTrack::TId TAudioteleCallsManager::TAudioteleCallTrackManager::GetMainId(const TAudioteleCallTrack& e) const {
    return e.GetId();
}


TAudioteleCallsManager::TAudioteleCallsManager(NStorage::IDatabase::TPtr database)
    : TDatabaseSessionConstructor(database)
    , EventManager(database)
    , TrackManager(database)
{
}

TMaybe<TVector<TAudioteleCallEvent>> TAudioteleCallsManager::GetEvents(NDrive::TEntitySession& session, const TSet<TAudioteleCallEvent::EAction>& filterActions, const TSet<TString>& callIds, const ui32 limit) const {
    return EventManager.GetObjects(session, filterActions, callIds, limit);
}

TMaybe<TVector<TAudioteleCallTrack>> TAudioteleCallsManager::GetTracksByEvents(NDrive::TEntitySession& session, const TSet<TAudioteleCallEvent::TId>& eventIds) const {
    return TrackManager.Fetch(session, NSQL::TQueryOptions().SetGenericCondition("call_entry_id", eventIds));
}

TMaybe<TVector<TAudioteleCallTrack>> TAudioteleCallsManager::GetTracksByCallId(NDrive::TEntitySession& session, const TString& callId) const {
    return TrackManager.Fetch(session, NSQL::TQueryOptions().SetCustomCondition("file_name LIKE '" + callId + "_%'"));
}

TMaybe<TMap<TString, TString>> TAudioteleCallsManager::GetCallTracksKeys(NDrive::TEntitySession& session, const TSet<TString>& callIds, const ui32 limit /* = 0 */) const {
    auto events = GetEvents(session, { TAudioteleCallEvent::EAction::Finish }, callIds, limit);
    if (!events) {
        return {};
    }
    TMap<TAudioteleCallEvent::TId, TString> eventMap;
    Transform(events->begin(), events->end(), std::inserter(eventMap, eventMap.begin()), [](const auto& event) -> std::pair<TAudioteleCallEvent::TId, TString> { return { event.GetId(), event.GetExternalCallId() }; });
    auto tracks = GetTracksByEvents(session, MakeSet(NContainer::Keys(eventMap)));
    if (!tracks) {
        return {};
    }
    TMap<TString, TString> result;
    Transform(tracks->begin(), tracks->end(), std::inserter(result, result.begin()), [&eventMap](const auto& track) -> std::pair<TString, TString> { return { eventMap[track.GetCallEventId()], track.GetMdsKey() }; });
    return result;
}

bool TAudioteleCallsManager::BindCallTrack(const TAudioteleCallEvent::TId eventId, const TAudioteleCallTrack::TId& trackId, NDrive::TEntitySession& session) const {
    auto table = TrackManager.GetDatabase().GetTable(TrackManager.GetTableName());
    NStorage::TTableRecord condition;
    condition.Set("id", trackId);
    NStorage::TTableRecord update;
    update.Set("call_entry_id", eventId);
    NStorage::TObjectRecordsSet<TAudioteleCallTrack> affected;
    auto result = table->UpdateRow(condition, update, session.GetTransaction(), &affected);
    if (!result || !result->IsSucceed()) {
        return false;
    }
    if (affected.size() != 1) {
        session.AddErrorMessage("BindCallTrack", "Wrong affected rows count: " + ToString(affected.size()));
        return false;
    }
    return true;
}
