#include "model.h"

#include <rtline/util/json_processing.h>

#include <library/cpp/json/json_reader.h>

void TSupportRequestCategorizerConfig::Init(const TYandexConfig::Section* section) {
    auto children = section->GetAllChildren();
    {
        auto it = children.find("HistoryConfig");
        if (it != children.end()) {
            HistoryConfig.Init(it->second);
        }
    }
}

void TSupportRequestCategorizerConfig::ToString(IOutputStream& os) const {
    os << "<HistoryConfig>" << Endl;
    HistoryConfig.ToString(os);
    os << "</HistoryConfig>" << Endl;
}

TSupportRequestCategorization::TSupportRequestCategorization(const TString& tagId, const TString& category, const TString& comment, const NJson::TJsonValue& metaInfo)
    : TagId(tagId)
    , Category(category)
    , Comment(comment)
    , MetaInfo(metaInfo)
{
}

NJson::TJsonValue TSupportRequestCategorization::BuildReport(const TMap<TString, TSupportCategorizerTreeNode>& categoryDescriptionMapping) const {
    NJson::TJsonValue result;
    result["id"] = GetOperatedId();
    result["comment"] = GetComment();
    result["meta_info"] = GetMetaInfo();

    NJson::TJsonValue& categoryDescriptionReport = result.InsertValue("category", NJson::JSON_MAP);
    auto categoryDescriptionPtr = categoryDescriptionMapping.FindPtr(GetCategory());
    if (categoryDescriptionPtr) {
        categoryDescriptionReport = categoryDescriptionPtr->SerializeToJson();
    } else {
        categoryDescriptionReport["id"] = GetCategory();
    }

    return result;
}

NStorage::TTableRecord TSupportRequestCategorization::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("tag_id", TagId);
    result.Set("category", Category);
    result.Set("comment", Comment);
    result.Set("operated_id", OperatedId);
    result.Set("meta_info", (MetaInfo.IsDefined()) ? MetaInfo.GetStringRobust() : "get_null()");
    return result;
}

NJson::TJsonValue TSupportRequestCategorization::SerializeToJson() const {
    NJson::TJsonValue result;
    result["tag_id"] = TagId;
    result["category"] = Category;
    result["comment"] = Comment;
    result["operated_id"] = OperatedId;
    result["meta_info"] = MetaInfo;
    return result;
}

bool TSupportRequestCategorization::DeserializeFromJson(const NJson::TJsonValue& raw) {
    JREAD_STRING(raw, "tag_id", TagId);
    JREAD_STRING_OPT(raw, "category", Category);
    JREAD_STRING_OPT(raw, "comment", Comment);
    JREAD_UINT_OPT(raw, "operated_id", OperatedId);
    MetaInfo = raw["meta_info"];
    return true;
}

bool TSupportRequestCategorization::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, TagId);
    READ_DECODER_VALUE(decoder, values, Category);
    READ_DECODER_VALUE(decoder, values, Comment);
    READ_DECODER_VALUE_DEF(decoder, values, OperatedId, 0);
    {
        TString metaInfo;
        READ_DECODER_VALUE_TEMP_OPT(decoder, values, metaInfo, MetaInfo);
        if (metaInfo && !NJson::ReadJsonTree(metaInfo, &MetaInfo)) {
            return false;
        }
    }
    return true;
}

TFullCategorization::TFullCategorization(const TString& tagId, const TVector<TAtomicSharedPtr<TObjectEvent<TSupportRequestCategorization>>>& items, const TSupportCategorizerNodesDB* nodesDB)
    : TagId(tagId)
    , NodesDB(nodesDB)
{
    for (auto&& item : items) {
        Items.emplace_back(item);
    }
}

TFullCategorization::operator bool() const noexcept {
    return !Empty();
}

bool TFullCategorization::Empty() const noexcept {
    return Items.empty();
}

NJson::TJsonValue TFullCategorization::BuildReport(const bool full) const {
    NJson::TJsonValue itemsJson = NJson::JSON_ARRAY;

    TMap<TString, TSupportCategorizerTreeNode> nodes;
    if (NodesDB) {
        TVector<TString> itemIds;
        for (auto&& item : Items) {
            itemIds.emplace_back(item->GetCategory());
        }
        nodes = NodesDB->GetCachedObjectsMap(itemIds);
    }

    for (auto&& item : Items) {
        auto oneItemJson = item->BuildReport(nodes);

        oneItemJson["operator_id"] = item->GetHistoryUserId();
        oneItemJson["timestamp"] = item->GetHistoryInstant().Seconds();
        if (full) {
            oneItemJson["action"] = ToString(item->GetHistoryAction());
        }

        itemsJson.AppendValue(std::move(oneItemJson));
    }

    NJson::TJsonValue result;
    result["tag_id"] = TagId;
    result["items"] = std::move(itemsJson);

    return result;
}

TSupportRequestCategorizationDB::TSupportRequestCategorizationDB(const THistoryContext& historyContext, const THistoryConfig& historyConfig)
    : TBase("support_request_categorization")
    , HistoryWriter(historyContext)
{
    CategorizationFetcher = MakeHolder<TCategorizationFetcher>(historyContext, historyConfig);
    CategorizationFetcher->RegisterCallback(this);
    Y_ENSURE_BT(CategorizationFetcher->Start());

    CategoryTreeNodes = MakeHolder<TSupportCategorizerNodesDB>(historyContext, historyConfig);
    Y_ENSURE_BT(CategoryTreeNodes->Start());

    CategoryTreeManager = MakeHolder<TSupportCategorizationTreeManager>(historyContext, historyConfig, CategoryTreeNodes.Get());
    Y_ENSURE_BT(CategoryTreeManager->Start());
}

TSupportRequestCategorizationDB::~TSupportRequestCategorizationDB() {
    if (!CategoryTreeManager->Stop()) {
        ERROR_LOG << "cannot stop CategoryTreeManager" << Endl;
    }
    if (!CategoryTreeNodes->Stop()) {
        ERROR_LOG << "cannot stop CategoryTreeNodes" << Endl;
    }
    CategorizationFetcher->UnregisterCallback(this);
    if (!CategorizationFetcher->Stop()) {
        ERROR_LOG << "cannot stop CategorizationFetcher" << Endl;
    }
}

bool TSupportRequestCategorizationDB::DoAcceptHistoryEventUnsafe(const THistoricalCategorization& dbEvent, const bool isNewEvent) {
    if (!isNewEvent) {
        return true;
    }

    if (dbEvent->GetHistoryAction() == EObjectHistoryAction::Add) {
        dbEvent->SetOperatedId(dbEvent->GetHistoryEventId());
    }
    auto categorizationIt = Categorizations.find(dbEvent->GetTagId());
    if (categorizationIt == Categorizations.end()) {
        categorizationIt = Categorizations.emplace(dbEvent->GetTagId(), TVector<THistoricalCategorization>()).first;
    }
    categorizationIt->second.push_back(dbEvent);
    size_t pos = categorizationIt->second.size() - 1;
    while (pos > 0 && categorizationIt->second[pos]->GetHistoryEventId() < categorizationIt->second[pos - 1]->GetHistoryEventId()) {
        std::swap(categorizationIt->second[pos], categorizationIt->second[pos - 1]);
        --pos;
    }
    return true;
}

bool TSupportRequestCategorizationDB::AddCategorization(const TSupportRequestCategorization& cat, const TString& operatorId, NDrive::TEntitySession& session) const {
    return HistoryWriter.AddHistory(cat, operatorId, EObjectHistoryAction::Add, session);
}

bool TSupportRequestCategorizationDB::RemoveCategorization(const TSupportRequestCategorization& cat, const TString& operatorId, NDrive::TEntitySession& session) const {
    return HistoryWriter.AddHistory(cat, operatorId, EObjectHistoryAction::Remove, session);
}

TFullCategorization TSupportRequestCategorizationDB::GetActualCategorization(const TString& tagId) const {
    auto result = GetActualCategorizations({tagId});
    if (!result.size()) {
        return TFullCategorization();
    }
    return result.begin()->second;
}

TMaybe<TSupportRequestCategorization> TSupportRequestCategorizationDB::GetActualCategorization(const TString& tagId, ui64 id) const {
    auto categorization = GetActualCategorization(tagId);
    for (auto&& item : categorization.GetItems()) {
        if (item && item->GetOperatedId() == id) {
            return *item;
        }
    }
    return {};
}

TMap<TString, TFullCategorization> TSupportRequestCategorizationDB::GetActualCategorizations(const TVector<TString>& tagIds) const {
    auto rg = MakeObjectReadGuard();
    TMap<TString, TFullCategorization> result;
    for (auto&& tagId : tagIds) {
        auto emplaceResult = result.emplace(tagId, TFullCategorization());
        if (!emplaceResult.second) {
            continue;
        }
        auto resultCatIt = emplaceResult.first;

        auto storedCatIt = Categorizations.find(tagId);
        if (storedCatIt == Categorizations.end()) {
            continue;
        }

        TSet<ui32> removedItemIds;
        TVector<THistoricalCategorization> preResult;
        for (auto&& hEventIt = storedCatIt->second.rbegin(); hEventIt != storedCatIt->second.rend(); ++hEventIt) {
            if ((*hEventIt)->GetHistoryAction() == EObjectHistoryAction::Remove) {
                removedItemIds.emplace((*hEventIt)->GetOperatedId());
            } else {
                if (!removedItemIds.emplace((*hEventIt)->GetOperatedId()).second) {
                    continue;
                }
                preResult.emplace_back(*hEventIt);
            }
        }
        std::reverse(preResult.begin(), preResult.end());
        resultCatIt->second = TFullCategorization(tagId, preResult, CategoryTreeNodes.Get());
    }
    return result;
}

TFullCategorization TSupportRequestCategorizationDB::GetCategorizationHistory(const TString& tagId) const {
    auto rg = MakeObjectReadGuard();

    auto storedCatIt = Categorizations.find(tagId);
    if (storedCatIt == Categorizations.end()) {
        return TFullCategorization();
    }

    TVector<THistoricalCategorization> result;
    for (auto&& hEvent : storedCatIt->second) {
        result.emplace_back(hEvent);
    }

    return TFullCategorization(tagId, result, CategoryTreeNodes.Get());
}

bool TSupportRequestCategorizationDB::RefreshCache(const TInstant actuality, const bool /*doActualizeHistory*/) const {
    if (CategorizationFetcher && !CategorizationFetcher->Update(actuality)) {
        return false;
    }
    return true;
}
