#pragma once

#include "tree.h"

#include <drive/backend/database/history/db_entities.h>

#include <library/cpp/json/writer/json_value.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/types/accessor.h>

class TSupportRequestCategorizerConfig {
public:
    R_READONLY(THistoryConfig, HistoryConfig);

public:
    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;
};

class TSupportRequestCategorization {
public:
    R_FIELD(TString, TagId);
    R_FIELD(TString, Category);
    R_FIELD(TString, Comment);
    R_FIELD(ui64, OperatedId, 0);
    R_FIELD(NJson::TJsonValue, MetaInfo);

public:
    using TId = TString;

public:
    TSupportRequestCategorization() = default;
    TSupportRequestCategorization(const TString& tagId, const TString& category, const TString& comment, const NJson::TJsonValue& metaInfo);

    class TDecoder: public TBaseDecoder {
        R_FIELD(i32, TagId, -1);
        R_FIELD(i32, Category, -1);
        R_FIELD(i32, Comment, -1);
        R_FIELD(i32, OperatedId, -1);
        R_FIELD(i32, MetaInfo, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase) {
            TagId = GetFieldDecodeIndex("tag_id", decoderBase);
            Category = GetFieldDecodeIndex("category", decoderBase);
            Comment = GetFieldDecodeIndex("comment", decoderBase);
            OperatedId = GetFieldDecodeIndex("operated_id", decoderBase);
            MetaInfo = GetFieldDecodeIndex("meta_info", decoderBase);
        }
    };

    NJson::TJsonValue BuildReport(const TMap<TString, TSupportCategorizerTreeNode>& categoryDescriptionMapping) const;

    NJson::TJsonValue SerializeToJson() const;
    bool DeserializeFromJson(const NJson::TJsonValue& raw);
    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*historyContext*/);
    NStorage::TTableRecord SerializeToTableRecord() const;
};

class TCategorizationFetcher : public TCallbackSequentialTableImpl<TObjectEvent<TSupportRequestCategorization>, TString> {
private:
    using TBase = TCallbackSequentialTableImpl<TObjectEvent<TSupportRequestCategorization>, TString>;

protected:
    virtual bool GetCallbackNeedsHistoryLoad() const override {
        return true;
    }

public:
    TCategorizationFetcher(const IHistoryContext& historyContext, const THistoryConfig& historyConfig)
        : TBase(historyContext, "support_request_categorization_history", historyConfig)
    {
    }
};

class TFullCategorization {
public:
    R_READONLY(TString, TagId);
    R_READONLY(TVector<TAtomicSharedPtr<const TObjectEvent<TSupportRequestCategorization>>>, Items);

public:
    TFullCategorization() = default;
    TFullCategorization(const TString& tagId, const TVector<TAtomicSharedPtr<TObjectEvent<TSupportRequestCategorization>>>& items, const TSupportCategorizerNodesDB* nodesDB);

    explicit operator bool() const noexcept;
    bool Empty() const noexcept;

    NJson::TJsonValue BuildReport(const bool full = false) const;

private:
    const TSupportCategorizerNodesDB* NodesDB;
};

class TCategorizationHistoryWriter: public TDatabaseHistoryManager<TSupportRequestCategorization> {
    using TBase = TDatabaseHistoryManager<TSupportRequestCategorization>;

public:
    TCategorizationHistoryWriter(const IHistoryContext& context)
        : TBase(context, "support_request_categorization_history")
    {
    }
};

class TSupportRequestCategorizationDB: public IHistoryCallback<TObjectEvent<TSupportRequestCategorization>, TString> {
private:
    using TBase = IHistoryCallback<TObjectEvent<TSupportRequestCategorization>, TString>;
    using THistoricalCategorization = TAtomicSharedPtr<TObjectEvent<TSupportRequestCategorization>>;

private:
    THolder<TCategorizationFetcher> CategorizationFetcher;
    TCategorizationHistoryWriter HistoryWriter;
    TMap<TString, TVector<THistoricalCategorization>> Categorizations;

    THolder<TSupportCategorizerNodesDB> CategoryTreeNodes;
    THolder<TSupportCategorizationTreeManager> CategoryTreeManager;

public:
    TSupportRequestCategorizationDB(const THistoryContext& historyContext, const THistoryConfig& historyConfig);
    ~TSupportRequestCategorizationDB();

    virtual TStringBuf GetEventObjectId(const TObjectEvent<TSupportRequestCategorization>& ev) const override {
        return ev.GetTagId();
    }

    TInstant GetLastUpdateTimestamp() const override {
        return CategorizationFetcher ? CategorizationFetcher->GetCurrentInstant() : TInstant::Zero();
    }

    bool AddCategorization(const TSupportRequestCategorization& cat, const TString& operatorId, NDrive::TEntitySession& session) const;
    bool RemoveCategorization(const TSupportRequestCategorization& cat, const TString& operatorId, NDrive::TEntitySession& session) const;

    TFullCategorization GetActualCategorization(const TString& tagId) const;
    TMaybe<TSupportRequestCategorization> GetActualCategorization(const TString& tagId, ui64 id) const;
    TMap<TString, TFullCategorization> GetActualCategorizations(const TVector<TString>& tagIds) const;
    TFullCategorization GetCategorizationHistory(const TString& tagId) const;
    bool RefreshCache(const TInstant actuality, const bool doActualizeHistory = true) const override;

    const TSupportCategorizationTreeManager* GetTreeManager() const {
        return CategoryTreeManager.Get();
    }

private:
    virtual bool DoAcceptHistoryEventUnsafe(const THistoricalCategorization& dbEvent, const bool isNewEvent) override;
};
