#include "tree.h"

NJson::TJsonValue TSupportCategorizerTreeNode::SerializeToJson() const {
    NJson::TJsonValue result;
    result["id"] = Id;
    result["meta"] = SerializeMeta();
    return result;
}

bool TSupportCategorizerTreeNode::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Id);
    TString metaStr;
    READ_DECODER_VALUE_TEMP(decoder, values, metaStr, Meta);
    NJson::TJsonValue meta;
    if (!NJson::ReadJsonFastTree(metaStr, &meta)) {
        return false;
    }
    return DeserializeMeta(meta);
}

NStorage::TTableRecord TSupportCategorizerTreeNode::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    if (Id) {
        result.Set("id", Id);
    }
    result.Set("meta", SerializeMeta());
    return result;
}

bool TSupportCategorizerTreeNode::DeserializeMeta(const NJson::TJsonValue& meta) {
    JREAD_UINT_OPT(meta, "order", Order);
    JREAD_BOOL_OPT(meta, "enabled", Enabled);
    if (meta.Has("label")) {
        JREAD_STRING(meta, "label", Label);
    } else {
        LabelLocalized = false;
        if (!meta.Has("labels") || !meta["labels"].IsMap()) {
            return false;
        }
        if (meta["labels"].Has("ru") && meta["labels"]["ru"].IsString() && meta["labels"]["ru"].GetString()) {
            Label = meta["labels"]["ru"].GetString();
        } else if (meta["labels"].Has("en") && meta["labels"]["en"].IsString() && meta["labels"]["en"].GetString()) {
            Label = meta["labels"]["en"].GetString();
        } else {
            return false;
        }
    }
    if (meta.Has("call_direction")) {
        if (!meta["call_direction"].IsString()) {
            ERROR_LOG << "call_direction is not string" << Endl;
            return false;
        }
        if (!TryFromString(meta["call_direction"].GetString(), Direction)) {
            ERROR_LOG << "call_direction is not enum element" << Endl;
            return false;
        }
    }
    if (meta.Has("origins")) {
        Origins.clear();
        if (!meta["origins"].IsArray()) {
            ERROR_LOG << "origins is not an array" << Endl;
            return false;
        }
        for (auto&& elem : meta["origins"].GetArray()) {
            if (!elem.IsString()) {
                ERROR_LOG << "element of origins is not string" << Endl;
                return false;
            }
            ERequestOrigin origin;
            if (!TryFromString(elem.GetString(), origin)) {
                ERROR_LOG << "element of origins is not enum element" << Endl;
                return false;
            }
            Origins.push_back(origin);
        }
    }
    if (meta.Has("keywords")) {
        Keywords.clear();
        if (!meta["keywords"].IsArray()) {
            ERROR_LOG << "keywords are not an array" << Endl;
            return false;
        }
        for (auto&& elem : meta["keywords"].GetArray()) {
            if (!elem.IsString()) {
                ERROR_LOG << "element of keywords is not string" << Endl;
                return false;
            }
            Keywords.push_back(elem.GetString());
        }
    }
    if (meta.Has("additional_data")) {
        AdditionalData = meta["additional_data"];
    }
    return true;
}

NDrive::TScheme TSupportCategorizerTreeNode::GetScheme() {
    NDrive::TScheme meta;
    meta.Add<TFSBoolean>("enabled", "Включено");
    meta.Add<TFSNumeric>("order", "Порядковый номер на своем уровне").SetDefault(0);
    meta.Add<TFSVariants>("call_direction", "Направление обращения").InitVariants<ERequestDirection>();
    meta.Add<TFSString>("label", "Имя");
    meta.Add<TFSArray>("keywords", "Ключевые слова").SetElement<TFSString>();
    meta.Add<TFSJson>("additional_data", "Дополнительная информация");
    meta.Add<TFSVariants>("origins", "Источники").InitVariants<ERequestOrigin>().SetMultiSelect(true);
    NDrive::TScheme result;
    result.Add<TFSStructure>("meta", "meta").SetStructure(meta);
    return result;
}

NJson::TJsonValue TSupportCategorizerTreeNode::SerializeMeta() const {
    NJson::TJsonValue result;
    result["enabled"] = Enabled;
    result["order"] = Order;
    result["label"] = Label;
    if (Direction != ERequestDirection::Unknown) {
        result["call_direction"] = ToString(Direction);
    }
    if (Origins.size()) {
        NJson::TJsonValue originsArray = NJson::JSON_ARRAY;
        for (auto&& origin : Origins) {
            originsArray.AppendValue(ToString(origin));
        }
        result["origins"] = std::move(originsArray);
    }
    if (Keywords.size()) {
        NJson::TJsonValue keywordsArray = NJson::JSON_ARRAY;
        for (auto&& keyword : Keywords) {
            keywordsArray.AppendValue(keyword);
        }
        result["keywords"] = std::move(keywordsArray);
    }
    if (AdditionalData.IsDefined()) {
        result["additional_data"] = AdditionalData;
    }
    return result;
}

bool TSupportCategorizerTreeNode::IsMatchingFilter(const TString& filter) const {
    if (!filter) {
        return true;
    }
    return Label.find(filter) != TString::npos;
}

NJson::TJsonValue TSupportCategorizerTreeEdge::SerializeToJson() const {
    NJson::TJsonValue result;
    result["id"] = Id;
    result["parent_id"] = ParentId;
    result["child_id"] = ChildId;
    return result;
}

bool TSupportCategorizerTreeEdge::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Id);
    READ_DECODER_VALUE(decoder, values, ParentId);
    READ_DECODER_VALUE(decoder, values, ChildId);
    return true;
}

NStorage::TTableRecord TSupportCategorizerTreeEdge::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    if (Id) {
        result.Set("id", Id);
    }
    result.Set("parent_id", ParentId);
    result.Set("child_id", ChildId);
    result.Set("meta", "{}");
    return result;
}

bool TSupportCategorizationTreeManager::UpdateEdgeInAdjacent(TMap<TString, TVector<TRecordType>>& adjacentEdges, const TString& source, const TObjectEvent<TRecordType>& ev) const {
    auto edgesIt = adjacentEdges.find(source);
    if (edgesIt == adjacentEdges.end()) {
        return false;
    }
    size_t pos;
    for (pos = 0; pos < edgesIt->second.size(); ++pos) {
        if (edgesIt->second[pos].GetId() == ev.GetId()) {
            break;
        }
    }
    if (pos == edgesIt->second.size()) {
        return false;
    }

    if (ev.GetHistoryAction() == EObjectHistoryAction::UpdateData) {
        edgesIt->second[pos] = ev;
    } else if (ev.GetHistoryAction() == EObjectHistoryAction::Remove) {
        std::swap(edgesIt->second[pos], edgesIt->second[edgesIt->second.size() - 1]);
        edgesIt->second.pop_back();
    }
    return true;
}

void TSupportCategorizationTreeManager::AcceptHistoryEventUnsafe(const TObjectEvent<TRecordType>& ev) const {
    if (ev.GetHistoryAction() == EObjectHistoryAction::Add) {
        AdjacentEdges[ev.GetParentId()].emplace_back(ev);
        AdjacentEdgesRev[ev.GetChildId()].emplace_back(ev);
    } else if (ev.GetHistoryAction() == EObjectHistoryAction::Remove || ev.GetHistoryAction() == EObjectHistoryAction::UpdateData) {
        UpdateEdgeInAdjacent(AdjacentEdges, ev.GetParentId(), ev);
        UpdateEdgeInAdjacent(AdjacentEdgesRev, ev.GetChildId(), ev);
    }
}

bool TSupportCategorizationTreeManager::DoRebuildCacheUnsafe() const {
    AdjacentEdges.clear();
    NStorage::TObjectRecordsSet<TRecordType> records;
    {
        auto table = Database->GetTable(EdgesTable->GetTableName());
        auto transaction = Database->CreateTransaction(true);
        auto result = table->GetRows("", records, transaction);
        if (!result->IsSucceed()) {
            ERROR_LOG << "Cannot refresh data for " << EdgesTable->GetTableName() << Endl;
            return false;
        }
    }
    for (auto&& record : records) {
        AdjacentEdges[record.GetParentId()].emplace_back(record);
        AdjacentEdgesRev[record.GetChildId()].emplace_back(record);
    }
    return true;
}

void TSupportCategorizationTreeManager::GetReachableNodeIds(const TString& id, TSet<TString>& visitedIds) const {
    visitedIds.insert(id);
    for (auto&& edge : AdjacentEdges[id]) {
        auto adjNode = edge.GetChildId();
        if (visitedIds.contains(adjNode)) {
            continue;
        }
        GetReachableNodeIds(adjNode, visitedIds);
    }
}

bool TSupportCategorizationTreeManager::RemoveEdge(const TString& parentId, const TString& childId, const TString& userId, NDrive::TEntitySession& session) const {
    auto rg = MakeObjectReadGuard();
    for (auto&& edge : AdjacentEdges[parentId]) {
        if (edge.GetChildId() == childId) {
            if (!EdgesTable->Remove(edge.GetId(), session)) {
                session.AddErrorMessage("support_categorizer_tree", "could not remove edge");
                return false;
            }
            if (!HistoryManager->AddHistory(edge, userId, EObjectHistoryAction::Remove, session)) {
                session.AddErrorMessage("support_categorizer_tree", "could not remove edge in history");
                return false;
            }
        }
    }
    return true;
}

bool TSupportCategorizationTreeManager::AddEdge(const TString& parentId, const TString& childId, const TString& userId, NDrive::TEntitySession& session, const bool force) const {
    auto rg = MakeObjectReadGuard();
    if (!force && (!GetNode(parentId).Defined() || !GetNode(childId).Defined())) {
        session.AddErrorMessage("support_categorizer_tree", "one of edge endpoints does not exist");
        return false;
    }
    TSet<TString> reachableNodeIds;
    GetReachableNodeIds(childId, reachableNodeIds);
    if (reachableNodeIds.contains(parentId)) {
        session.AddErrorMessage("support_categorizer_tree", "adding an edge would cause a cycle");
        return false;
    }
    {
        NSQL::TQueryOptions options;
        options.AddGenericCondition("parent_id", parentId);
        options.AddGenericCondition("child_id", childId);
        auto edges = EdgesTable->FetchInfo(session, options);
        if (!edges) {
            return false;
        }
        if (!edges.empty()) {
            return true;
        }
    }
    TSupportCategorizerTreeEdge e(parentId, childId);
    NStorage::TObjectRecordsSet<TSupportCategorizerTreeEdge> insertedEdges;
    if (!EdgesTable->Insert(e, session, &insertedEdges) || insertedEdges.size() != 1) {
        session.AddErrorMessage("support_categorizer_tree", "could not insert new edge");
        return false;
    }
    if (!HistoryManager->AddHistory(*insertedEdges.begin(), userId, EObjectHistoryAction::Add, session)) {
        session.AddErrorMessage("support_categorizer_tree", "could not write history");
        return false;
    }
    return true;
}

bool TSupportCategorizationTreeManager::RemoveNode(const TString& id, const TString& userId, NDrive::TEntitySession& session) const {
    auto rg = MakeObjectReadGuard();
    TSet<TString> reachableNodeIds;
    GetReachableNodeIds(id, reachableNodeIds);
    if (reachableNodeIds.size() >= 16) {
        session.AddErrorMessage("support_categorizer_tree", "too many nodes to delete");
        return false;
    }
    TSet<TSupportCategorizerTreeEdge> edgesForRemoval;
    for (auto&& id : reachableNodeIds) {
        for (auto&& adjEdge : AdjacentEdges[id]) {
            edgesForRemoval.insert(adjEdge);
        }
        for (auto&& adjEdge : AdjacentEdgesRev[id]) {
            edgesForRemoval.insert(adjEdge);
        }
    }
    if (!NodesDB->RemoveObject(reachableNodeIds, userId, session)) {
        session.AddErrorMessage("support_categorizer_tree", "could not delete nodes");
        return false;
    }
    if (edgesForRemoval) {
        TVector<TString> edgeIds;
        TVector<TObjectEvent<TSupportCategorizerTreeEdge>> hEntries;
        auto actuality = Now();
        for (auto&& edge : edgesForRemoval) {
            TObjectEvent<TSupportCategorizerTreeEdge> hEdge(edge, EObjectHistoryAction::Remove, actuality, userId, session.GetOriginatorId(), session.GetComment());
            hEntries.push_back(hEdge);
            edgeIds.push_back(edge.GetId());
        }
        if (!EdgesTable->Remove(edgeIds, session)) {
            session.AddErrorMessage("support_categorizer_tree", "could not delete edge");
            return false;
        }
        if (!HistoryManager->AddHistory(hEntries, session)) {
            session.AddErrorMessage("support_categorizer_tree", "could write history about deleted edges");
            return false;
        }
    }
    return true;
}

TStringBuf TSupportCategorizationTreeManager::GetEventObjectId(const TObjectEvent<TRecordType>& ev) const {
    return ev.GetId();
}

NJson::TJsonValue TSupportCategorizationTreeManager::GetTreeReportImpl(const TString& node, const TMap<TString, TSupportCategorizerTreeNode>& nodesInfo, const TString& filter, const bool isAlreadyMatched) const {
    TVector<TNodeTraversalReportData> childrenReportElements;
    auto edgesIt = AdjacentEdges.find(node);
    if (edgesIt != AdjacentEdges.end()) {
        for (auto&& edge : AdjacentEdges[node]) {
            auto nextNodeIt = nodesInfo.find(edge.GetChildId());
            if (nextNodeIt == nodesInfo.end()) {
                continue;
            }

            bool nextAlreadyMatched = isAlreadyMatched ? true : nextNodeIt->second.IsMatchingFilter(filter);
            auto report = GetTreeReportImpl(edge.GetChildId(), nodesInfo, filter, nextAlreadyMatched);
            if (report == NJson::JSON_NULL) {
                continue;
            }
            childrenReportElements.emplace_back(TNodeTraversalReportData(nextNodeIt->second.GetOrder(), std::move(report)));
        }
    }

    if (isAlreadyMatched || childrenReportElements.size()) {
        auto currentNodeIt = nodesInfo.find(node);
        if (currentNodeIt == nodesInfo.end()) {
            return NJson::JSON_NULL;
        }
        NJson::TJsonValue report = currentNodeIt->second.SerializeToJson();
        if (childrenReportElements.size()) {
            Sort(childrenReportElements.begin(), childrenReportElements.end());
            NJson::TJsonValue childrenReport = NJson::JSON_ARRAY;
            for (auto&& element : childrenReportElements) {
                childrenReport.AppendValue(std::move(element.Report));
            }
            report["children"] = std::move(childrenReport);
        }
        return report;
    }

    return NJson::JSON_NULL;
}

NJson::TJsonValue TSupportCategorizationTreeManager::GetTreeReport(const TString& filter) const {
    auto rg = MakeObjectReadGuard();
    auto cachedNodes = NodesDB->GetCachedObjectsMap();

    TVector<TNodeTraversalReportData> rootReportElements;
    for (auto&& nodeIt : cachedNodes) {
        if (!AdjacentEdgesRev[nodeIt.first].empty()) {
            continue;
        }

        auto report = GetTreeReportImpl(nodeIt.first, cachedNodes, filter, nodeIt.second.IsMatchingFilter(filter));
        if (report == NJson::JSON_NULL) {
            continue;
        }

        rootReportElements.emplace_back(TNodeTraversalReportData(nodeIt.second.GetOrder(), std::move(report)));
    }
    std::sort(rootReportElements.begin(), rootReportElements.end());

    NJson::TJsonValue childrenList = NJson::JSON_ARRAY;
    for (auto&& elem : rootReportElements) {
        childrenList.AppendValue(elem.Report);
    }
    NJson::TJsonValue fullReport;
    fullReport["children"] = std::move(childrenList);

    return fullReport;
}
