#pragma once

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/db_entities.h>

#include <library/cpp/json/writer/json_value.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/types/accessor.h>

class TSupportCategorizerTreeNode {
public:
    enum ERequestDirection {
        Unknown /* "unknown" */,
        Incoming /* "incoming" */,
        Outgoing /* "outgoing" */,
    };

    enum ERequestOrigin {
        Autopark     /* "autopark" */,
        CallIncoming /* "call_incoming" */,
        CallOutgoing /* "call_outgoing" */,
        ChatIncoming /* "chat_incoming" */,
        ChatOutgoing /* "chat_outgoing" */,
        ClientApp    /* "client_app" */,
    };

    R_FIELD(TString, Id);
    R_FIELD(TString, Label);
    R_FIELD(ui32, Order, Max<ui32>() / 3);
    R_FIELD(bool, Enabled, true);
    R_FIELD(bool, LabelLocalized, true);
    R_FIELD(ERequestDirection, Direction, ERequestDirection::Unknown);
    R_FIELD(TVector<ERequestOrigin>, Origins);
    R_FIELD(TVector<TString>, Keywords);
    R_FIELD(NJson::TJsonValue, AdditionalData, NJson::JSON_NULL);

public:
    bool operator !() const {
        return !Label;
    }

    TMaybe<ui32> OptionalRevision() const {
        return {};
    }

    using TId = TString;

    static TString GetTableName() {
        return "support_categorization_tree_nodes";
    }

    static TString GetHistoryTableName() {
        return "support_categorization_tree_nodes_history";
    }

    const TString& GetInternalId() const {
        return GetId();
    }

    class TDecoder: public TBaseDecoder {
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, Meta, -1);
    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase) {
            Id = GetFieldDecodeIndex("id", decoderBase);
            Meta = GetFieldDecodeIndex("meta", decoderBase);
        }
    };

    static NDrive::TScheme GetScheme();
    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
    NStorage::TTableRecord SerializeToTableRecord() const;

    NJson::TJsonValue SerializeToJson() const;
    bool DeserializeMeta(const NJson::TJsonValue& meta);
    NJson::TJsonValue SerializeMeta() const;

    bool IsMatchingFilter(const TString& filter) const;
};

class TCategorizerNodeConditionConstructor {
public:
    static TString BuildCondition(const TSet<TString>& ids, NDrive::TEntitySession& session) {
        return "id IN (" + session->Quote(ids) + ")";
    }

    static NStorage::TTableRecord BuildCondition(const TString& id) {
        NStorage::TTableRecord trCondition;
        if (id) {
            trCondition.Set("id", id);
        }
        return trCondition;
    }

    template <class T>
    static NStorage::TTableRecord BuildCondition(const T& object) {
        return BuildCondition(object.GetId());
    }
};

class TSupportCategorizerNodesDB : public TDBEntitiesManager<TSupportCategorizerTreeNode, TCategorizerNodeConditionConstructor> {
private:
    using TBase = TDBEntitiesManager<TSupportCategorizerTreeNode, TCategorizerNodeConditionConstructor>;

public:
    using TBase::TBase;
};

class TSupportCategorizerTreeEdge {
    R_READONLY(TString, Id);
    R_READONLY(TString, ParentId);
    R_READONLY(TString, ChildId);

public:
    TSupportCategorizerTreeEdge() = default;
    TSupportCategorizerTreeEdge(const TString& parentId, const TString& childId)
        : ParentId(parentId)
        , ChildId(childId)
    {
    }

    bool operator !() const {
        return !ChildId;
    }

    bool operator < (const TSupportCategorizerTreeEdge& other) const {
        return Id < other.GetId();
    }

    TMaybe<ui32> OptionalRevision() const {
        return {};
    }

    using TId = TString;

    static TString GetTableName() {
        return "support_categorization_tree_edges";
    }

    static TString GetHistoryTableName() {
        return "support_categorization_tree_edges_history";
    }

    const TString& GetInternalId() const {
        return GetId();
    }

    class TDecoder: public TBaseDecoder {
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, ParentId, -1);
        R_FIELD(i32, ChildId, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase) {
            Id = GetFieldDecodeIndex("id", decoderBase);
            ParentId = GetFieldDecodeIndex("parent_id", decoderBase);
            ChildId = GetFieldDecodeIndex("child_id", decoderBase);
        }
    };

    bool Parse(const NStorage::TTableRecord& row) {
        return TBaseDecoder::DeserializeFromTableRecord(*this, row);
    }

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
    NStorage::TTableRecord SerializeToTableRecord() const;
    NJson::TJsonValue SerializeToJson() const;
};

class TCategorizerEdgesTable : public TDBEntities<TSupportCategorizerTreeEdge> {
private:
    using TBase = TDBEntities<TSupportCategorizerTreeEdge>;
    using TEntity = TSupportCategorizerTreeEdge;

public:
    using TBase::TBase;

    virtual TString GetTableName() const override {
        return "support_categorization_tree_edges";
    }

    virtual TString GetColumnName() const override {
        return "id";
    }

    virtual TString GetMainId(const TSupportCategorizerTreeEdge& e) const override {
        return ToString(e.GetId());
    }
};

class TSupportCategorizerTreeEdgeHistoryManager : public TIndexedAbstractHistoryManager<TSupportCategorizerTreeEdge> {
private:
    using TBase = TIndexedAbstractHistoryManager<TSupportCategorizerTreeEdge>;

public:
    TSupportCategorizerTreeEdgeHistoryManager(const IHistoryContext& context, const THistoryConfig& hConfig)
        : TBase(context, "support_categorization_tree_edges_history", hConfig)
    {
    }
};

class TSupportCategorizationTreeManager : public TDBCacheWithHistoryOwner<TSupportCategorizerTreeEdgeHistoryManager, TSupportCategorizerTreeEdge> {
private:
    using TRecordType = TSupportCategorizerTreeEdge;
    using TCacheBase = TDBCacheWithHistoryOwner<TSupportCategorizerTreeEdgeHistoryManager, TSupportCategorizerTreeEdge>;
    using TDBTable = TCategorizerEdgesTable;

private:
    const TSupportCategorizerNodesDB* NodesDB;
    mutable TMap<TString, TVector<TRecordType>> AdjacentEdges;
    mutable TMap<TString, TVector<TRecordType>> AdjacentEdgesRev;
    THolder<TDBTable> EdgesTable;
    NStorage::IDatabase::TPtr Database;

private:
    virtual void AcceptHistoryEventUnsafe(const TObjectEvent<TRecordType>& ev) const override;
    virtual bool DoRebuildCacheUnsafe() const override;
    virtual TStringBuf GetEventObjectId(const TObjectEvent<TRecordType>& ev) const override;

    bool UpdateEdgeInAdjacent(TMap<TString, TVector<TRecordType>>& adjacentEdges, const TString& source, const TObjectEvent<TRecordType>& ev) const;

    NJson::TJsonValue GetTreeReportImpl(const TString& node, const TMap<TString, TSupportCategorizerTreeNode>& nodesInfo, const TString& filter, const bool isAlreadyMatched) const;

    struct TNodeTraversalReportData {
        ui32 Order;
        NJson::TJsonValue Report;

        TNodeTraversalReportData() = default;

        TNodeTraversalReportData(const ui32 order, const NJson::TJsonValue& report)
            : Order(order)
            , Report(report)
        {
        }

        bool operator < (const TNodeTraversalReportData& other) const {
            return Order < other.Order;
        }
    };

    void GetReachableNodeIds(const TString& id, TSet<TString>& visitedIds) const;

public:
    TSupportCategorizationTreeManager(const THistoryContext& hContext, const THistoryConfig& hConfig, const TSupportCategorizerNodesDB* nodesDB)
        : TCacheBase("support_categorization_tree_edges", hContext, hConfig)
        , NodesDB(nodesDB)
        , Database(hContext.GetDatabase())
    {
        EdgesTable.Reset(new TDBTable(hContext.GetDatabase()));
    }

    NJson::TJsonValue GetTreeReport(const TString& filter) const;

    bool UpsertNode(const TSupportCategorizerTreeNode& node, const TString& userId, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TSupportCategorizerTreeNode>* upserted) const {
        return NodesDB->ForceUpsertObject(node, userId, session, upserted);
    }

    TMaybe<TSupportCategorizerTreeNode> GetNode(const TString& nodeId) const {
        return NodesDB->GetObject(nodeId);
    }

    bool RemoveEdge(const TString& parentId, const TString& childId, const TString& userId, NDrive::TEntitySession& session) const;
    bool AddEdge(const TString& parentId, const TString& childId, const TString& userId, NDrive::TEntitySession& session, const bool force = false) const;
    bool RemoveNode(const TString& id, const TString& userId, NDrive::TEntitySession& session) const;

    bool RefreshAll(const TInstant actuality) const {
        if (!NodesDB->RefreshCache(actuality)) {
            return false;
        }
        return RefreshCache(actuality);
    }
};
