#include "manager.h"

#include <drive/backend/roles/manager.h>

#include <drive/library/cpp/auth/tvm.h>
#include <drive/library/cpp/mds/client.h>
#include <drive/library/cpp/raw_text/datetime.h>

TSupportRequestsDistributonManager::TSupportRequestsDistributonManager(const NDrive::IServer* server, const TString& notifierName)
    : Server(server)
    , NotifierName(notifierName)
{
}

bool TSupportRequestsDistributonManager::ResolveSupportRequests(const TVector<TString>& tagNames) const {
    auto actuality = Now();
    bool isSucceeded = true;
    for (auto&& tagName : tagNames) {
        if (!ResolveSupportRequests(tagName, actuality)) {
            isSucceeded = false;
        }
    }
    return isSucceeded;
}

bool TSupportRequestsDistributonManager::ResolveSupportRequests(const TString& tagName, const TInstant actuality) const {
    TVector<TDBTag> awaitingRequests;
    auto probablePerformers = MakeVector(Server->GetDriveAPI()->GetRolesManager()->GetPotentialTagAssignees(tagName, actuality));
    if (probablePerformers.empty()) {
        WARNING_LOG << "potential tag assignees for " << tagName << " are empty" << Endl;
    }

    TMap<TString, TVector<TString>> performedTagNamesByUser;

    {
        auto roSession = Server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTagsWithoutPerformer({tagName}, TVector<TString>(), awaitingRequests, roSession)) {
            MaybeNotify("Не могу восстановить теги с именем: " + tagName + " и без performer-a");
            return false;
        }

        TVector<TDBTag> performedTags;
        if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestorePerformerTags(probablePerformers, performedTags, roSession)) {
            MaybeNotify("Не могу восстановить по performer-ам для оценки лимитов");
            return false;
        }
        for (auto&& tag : performedTags) {
            performedTagNamesByUser[tag->GetPerformer()].emplace_back(tag->GetName());
        }
    }

    if (probablePerformers.empty() && !awaitingRequests.empty()) {
        MaybeNotify("Проблема: по тегу " + tagName + " есть обращения, но нет активных исполнителей, способных взять их в работу");
        return false;
    }

    TVector<ui32> remainingQuota(probablePerformers.size());
    TVector<TUserPermissions::TPtr> permissions(probablePerformers.size());
    for (size_t i = 0; i < probablePerformers.size(); ++i) {
        const auto& userId = probablePerformers[i];
        permissions[i] = Server->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());
        remainingQuota[i] = permissions[i]->GetRemainingPerformLoad(tagName, performedTagNamesByUser[userId]);
    }

    while (!awaitingRequests.empty()) {
        TVector<size_t> assigneeIndices;
        ui32 maxRemainingQuota = 0;
        ui32 secondMaxRemainingQuota = 0;
        for (size_t i = 0; i < remainingQuota.size(); ++i) {
            if (remainingQuota[i] > maxRemainingQuota) {
                secondMaxRemainingQuota = maxRemainingQuota;
                maxRemainingQuota = remainingQuota[i];
                assigneeIndices.clear();
                assigneeIndices.push_back(i);
            } else if (remainingQuota[i] == maxRemainingQuota) {
                assigneeIndices.push_back(i);
            } else if (remainingQuota[i] > secondMaxRemainingQuota) {
                secondMaxRemainingQuota = remainingQuota[i];
            }
        }

        if (!maxRemainingQuota) {
            break;
        }

        auto margin = maxRemainingQuota - secondMaxRemainingQuota;
        for (size_t iter = 0; iter < margin; ++iter) {
            std::random_shuffle(assigneeIndices.begin(), assigneeIndices.end());
            for (i32 i = 0; i < (i32)assigneeIndices.size() && !awaitingRequests.empty(); ++i) {
                auto assigneeId = assigneeIndices[i];
                auto performerPermissions = permissions[assigneeId];

                const auto& request = awaitingRequests.back();
                if (!AssignSupportOperator(request, performerPermissions)) {
                    MaybeNotify("Оператор " + probablePerformers[assigneeId] + " имеет достаточно квоты, но не может взять в работу пользователя " + request.GetObjectId());

                    remainingQuota[assigneeId] = 0;
                    assigneeIndices[i] = assigneeIndices[assigneeIndices.size() - 1];
                    assigneeIndices.pop_back();
                    --i;
                } else {
                    awaitingRequests.pop_back();
                }
            }
        }
    }

    return true;
}

bool TSupportRequestsDistributonManager::AssignSupportOperator(const TDBTag& supportRequest, const TUserPermissions::TPtr permissions) const {
    auto session = Server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
    return Server->GetDriveAPI()->GetTagsManager().GetUserTags().InitPerformer({ supportRequest }, *permissions.Get(), Server, session) && session.Commit();
}

void TSupportRequestsDistributonManager::MaybeNotify(const TString& message) const {
    DEBUG_LOG << "Notification: " << message << Endl;
    if (NotifierName) {
        NDrive::INotifier::Notify(Server->GetNotifier(NotifierName), message);
    }
}

TLoadBalanceCalendarHelper::TLoadBalanceCalendarHelper(const TString& applicationName)
    : ApplicationName(applicationName)
{
}

TMaybe<NCallCenterYandex::TAppDistribution> TLoadBalanceCalendarHelper::GetCurrentValue() const {
    auto [weekDay, hour] = GetCurrentWeekDayHourPair();

    if (auto distributionByHourPtr = CalendarData.FindPtr(weekDay)) {
        if (auto distributionPtr = distributionByHourPtr->FindPtr(hour)) {
            return *distributionPtr;
        }
    }

    return {};
}

bool TLoadBalanceCalendarHelper::SetCurrentValue(const NCallCenterYandex::TAppDistribution& update) {
    auto [weekDay, hour] = GetCurrentWeekDayHourPair();

    if (auto distributionByHourPtr = CalendarData.FindPtr(weekDay)) {
        if (auto distributionPtr = distributionByHourPtr->FindPtr(hour)) {
            *distributionPtr = update;
            return true;
        }
    }

    return false;
}

NJson::TJsonValue TLoadBalanceCalendarHelper::SerializeToJson() const {
    NJson::TJsonValue result;

    for (auto&& [weekDay, distributionByHour]: CalendarData) {
        NJson::TJsonValue& distributionByHourData = result.InsertValue(weekDay, NJson::JSON_MAP);

        for (auto&& [hour, distribution]: distributionByHour) {
            distributionByHourData.InsertValue(hour, distribution.SerializeLoadBalanceToJson());
        }
    }

    return result;
}

bool TLoadBalanceCalendarHelper::DeserializeFromJson(const NJson::TJsonValue& data) {
    if (!data.IsMap()) {
        return false;
    }

    for (auto&& [weekDay, distributionByHourData]: data.GetMap()) {
        if (!distributionByHourData.IsMap()) {
            return false;
        }

        auto& weekDayDistribution = CalendarData[weekDay];
        for (auto&& [hour, distributionData]: distributionByHourData.GetMap()) {
            NCallCenterYandex::TAppDistribution distribution;
            distribution.SetApplicationName(ApplicationName);
            if (!distribution.DeserializeLoadBalanceFromJson(distributionData)) {
                return false;
            }
            weekDayDistribution.emplace(hour, std::move(distribution));
        }
    }

    return true;
}

std::pair<TString, TString> TLoadBalanceCalendarHelper::GetCurrentWeekDayHourPair() const {
    auto localNow = NUtil::ConvertTimeZone(Now(), NUtil::GetUTCTimeZone(), NUtil::GetTimeZone("Europe/Moscow"));

    TString weekDay = ::ToString(NUtil::GetWeekDay(localNow));
    TString hour = NUtil::FormatDatetime(localNow, "%H:00");

    return { weekDay, hour };
}

TSupportCenterManager::TSupportCenterManager(const NDrive::IServer* server, const TSupportCenterManagerConfig& config)
    : TDatabaseSessionConstructor(server->GetDatabase(config.GetDBName()))
    , Server(server)
    , WebPhoneCallManager(THistoryContext(GetDatabasePtr()), config.GetWebphoneClientConfig())
    , AudioteleCallsManager(server->GetDatabase(config.GetAudioteleDBName()))
    , CiptCallEventsReader(server->GetDatabase(config.GetInternalCallCenterDBName()))
    , InternalCallCenterClient(config.GetInternalCallCenterConfig())
    , SupportAICallManager(THistoryContext(server->GetDatabase(config.GetSupportAIDBName())))
{
    Categorizer.Reset(new TSupportRequestCategorizationDB(THistoryContext(GetDatabasePtr()), config.GetCategorizerConfig().GetHistoryConfig()));
    LegacyPriorityManager.Reset(new TLegacyPriorityManager(server->GetDriveAPI()->GetDatabasePtr()));
    WebphoneClient.Reset(new TWebphoneClient(config.GetWebphoneClientConfig(), Server->GetTvmClient(config.GetWebphoneClientConfig().GetSelfClientId())));
    if (config.HasCallCenterYandexConfig()) {
        CallCenterYandexClient = MakeHolder<NCallCenterYandex::TCallCenterYandexClient>(config.GetCallCenterYandexConfigRef());
    }
    if (config.HasMDSClientConfig()) {
        MDSClient = MakeHolder<TS3Client>(config.GetMDSClientConfigRef());
        if (!MDSClient->Init(true)) {
            ERROR_LOG << "Can't initialize mds client" << Endl;
        }
    }
}

void TSupportCenterManager::Start() {
}

void TSupportCenterManager::Stop() {
}

NDrive::TEntitySession TSupportCenterManager::BuildSession(const bool readOnly) const {
    return TDatabaseSessionConstructor::BuildSession(readOnly);
}

TSupportRequestCategorizationDB* TSupportCenterManager::GetSupportRequestCategorizer() const {
    return Categorizer.Get();
}

TLegacyPriorityManager* TSupportCenterManager::GetLegacyPriorityManager() const {
    return LegacyPriorityManager.Get();
}

TWebphoneClient* TSupportCenterManager::GetWebphoneClient() const {
    return WebphoneClient.Get();
}

const NCallCenterYandex::TInternalCallCenterClient& TSupportCenterManager::GetInternalCallCenterClient() const {
    return InternalCallCenterClient;
}

const TCiptCallEventsReader& TSupportCenterManager::GetCiptCallEventsReader() const {
    return CiptCallEventsReader;
}

TS3Client* TSupportCenterManager::GetMDSClient() const {
    return MDSClient.Get();
}

const TWebPhoneCallManager& TSupportCenterManager::GetWebPhoneCallManager() const {
    return WebPhoneCallManager;
}

const TAudioteleCallsManager& TSupportCenterManager::GetAudioteleCallsManager() const {
    return AudioteleCallsManager;
}

const TSupportAICallManager& TSupportCenterManager::GetSupportAICallManager() const {
    return SupportAICallManager;
}

NCallCenterYandex::TCallCenterYandexClient* TSupportCenterManager::GetCallCenterYandexClient() const {
    return CallCenterYandexClient.Get();
}

TMaybe<TLoadBalanceCalendarHelper> TSupportCenterManager::GetLoadBalanceCalendar(const TString& appName) const {
    auto calendarHelper = TLoadBalanceCalendarHelper(appName);
    if (!calendarHelper.DeserializeFromJson(Server->GetSettings().GetJsonValue(LoadBalanceSettingKey + "." + appName))) {
        return {};
    }
    return calendarHelper;
}

bool TSupportCenterManager::UpdateLoadBalanceCalendar(const TLoadBalanceCalendarHelper& calendar, const TString& performerId) const {
    return Server->GetSettings().SetValue(LoadBalanceSettingKey + "." + calendar.GetApplicationName(), calendar.SerializeToJson().GetStringRobust(), performerId);
}

TMaybe<NCallCenterYandex::TAppDistribution> TSupportCenterManager::GetCurrentLoadBalanceCalendarValue(const TString& appName) const {
    auto calendarHelper = GetLoadBalanceCalendar(appName);
    return (calendarHelper) ? calendarHelper->GetCurrentValue() : Nothing();
}

bool TSupportCenterManager::UpdateCurrentLoadBalance(const NCallCenterYandex::TAppDistribution& value, const TString& performerId, TMessagesCollector& errors, bool updateCalendar) const {
    if (!CallCenterYandexClient) {
        errors.AddMessage(__LOCATION__, "Yandex callcenter client does not configured");
        return false;
    }

    if (!value.GetApplicationName()) {
        errors.AddMessage(__LOCATION__, "Application name is not set");
        return false;
    }

    auto applicationName = value.GetApplicationName();

    if (updateCalendar) {
        auto calendarHelper = GetLoadBalanceCalendar(applicationName);
        if (!calendarHelper) {
            errors.AddMessage(__LOCATION__, "Cannot load current calendar");
            return false;
        }
        calendarHelper->SetCurrentValue(value);
        if (!UpdateLoadBalanceCalendar(*calendarHelper, performerId)) {
            errors.AddMessage(__LOCATION__, "Cannot store calendar value");
            return false;
        }
    }

    if (!CallCenterYandexClient->UpdateAppLoadBalance(value, errors)) {
        return false;
    }

    return true;
}

NThreading::TFuture<std::pair<TSupportCenterManager::EProcessCallResult, TString>> TSupportCenterManager::ProcessInternalCall(const NS3::TBucket& bucket, const TString& callId, const bool needCheck /* = false */) const {
    auto loadTrackApplier = [callId, &bucket = bucket](const NThreading::TFuture<TString>& content) -> NThreading::TFuture<std::pair<EProcessCallResult, TString>> {
        if (!content.HasValue()) {
            return NThreading::MakeFuture(std::make_pair<EProcessCallResult, TString>(EProcessCallResult::CallCenterError, NThreading::GetExceptionMessage(content)));
        }
        return bucket.PutKey(callId, content.GetValue()).Apply([callId](const NThreading::TFuture<NUtil::THttpReply>& reply) mutable -> NThreading::TFuture<std::pair<EProcessCallResult, TString>> {
            if (!reply.HasValue()) {
                return NThreading::MakeFuture(std::make_pair<EProcessCallResult, TString>(EProcessCallResult::MdsError, "Failed upload file(" + callId + ") " + NThreading::GetExceptionMessage(reply)));
            }
            const auto& report = reply.GetValue();
            if (!report.IsSuccessReply()) {
                return NThreading::MakeFuture(std::make_pair<EProcessCallResult, TString>(EProcessCallResult::MdsError, "Got failed reply" + report.ErrorMessage()));
            }
            return NThreading::MakeFuture<std::pair<EProcessCallResult, TString>>({EProcessCallResult::Ok, std::move(callId)});
        });
    };
    if (needCheck) {
        return bucket.HasKey(callId).Apply([this, callId, loadTrackApplier](const NThreading::TFuture<bool>& reply) -> NThreading::TFuture<std::pair<EProcessCallResult, TString>> {
            if (!reply.HasValue()) {
                return NThreading::MakeFuture(std::make_pair<EProcessCallResult, TString>(EProcessCallResult::MdsError, "Failed check file(" + callId + ") " + NThreading::GetExceptionMessage(reply)));
            }
            if (reply.GetValue()) {
                return NThreading::MakeFuture<std::pair<EProcessCallResult, TString>>({EProcessCallResult::Ok, std::move(callId)});
            }
            return GetInternalCallCenterClient().LoadTrack(callId).Apply(loadTrackApplier);
        });
    }
    return GetInternalCallCenterClient().LoadTrack(callId).Apply(loadTrackApplier);
}


TWebPhoneCallManager::ECallDataError TWebPhoneCallManager::GetCallData(const NDrive::IServer& server, const TWebPhoneCall::TId callId, TWebPhoneCall& call, TMessagesCollector& errors) const {
    return GetCallData(server, ConvertOriginCallId(callId), call, errors);
}

TWebPhoneCallManager::ECallDataError TWebPhoneCallManager::GetCallData(const NDrive::IServer& server, const TString& callId, TWebPhoneCall& call, TMessagesCollector& errors) const {
    if (!server.GetDriveAPI()) {
        errors.AddMessage("GetCallData", "DriveApi problem");
        return ECallDataError::InternalError;
    }
    if (!server.GetDriveAPI()->HasMDSClient()) {
        errors.AddMessage("GetCallData", "Undefined MDS client");
        return ECallDataError::InternalError;
    }
    auto bucket = server.GetDriveAPI()->GetMDSClient().GetBucket(BucketName);
    if (!bucket) {
        errors.AddMessage("GetCallData", "MDS bucket is not found");
        return ECallDataError::InternalError;
    }
    const TString fileKey = callId + ".json";
    TString data;
    const ui32 code = bucket->GetFile(fileKey, data, errors);
    if (code / 100 != 2) {
        return code == HTTP_NOT_FOUND ? ECallDataError::NotFoundData : ECallDataError::RequestError;
    }
    NJson::TJsonValue json;
    if (!ReadJsonTree(data, &json) || !call.DeserializeFromJson(json)) {
        errors.AddMessage("GetCallData", "Fail to parse json file '" + data + "'");
        return ECallDataError::WrongData;
    }
    call.SetStatus(call.GetAnswerTS() ? TWebPhoneCall::EStatus::Servised : TWebPhoneCall::EStatus::NotServised);
    return ECallDataError::Ok;
}

TString TWebPhoneCallManager::ConvertOriginCallId(const TWebPhoneCall::TId id) const {
    return CallIdPrefix + ToString(id);
}

bool TWebPhoneCallManager::TryParseOriginCallId(const TString& callId, TWebPhoneCall::TId& result) const {
    return callId.StartsWith(CallIdPrefix) && TryFromString(callId.substr(CallIdPrefix.size()), result);
}
