#pragma once

#include "config.h"
#include "ifaces.h"

#include <drive/backend/support_center/audiotele/manager.h>
#include <drive/backend/support_center/support_ai/manager.h>
#include <drive/backend/support_center/telephony/entities.h>
#include <drive/backend/support_center/telephony/manager.h>
#include <drive/backend/support_center/yandex/client.h>
#include <drive/backend/support_center/yandex/entities.h>
#include <drive/backend/support_center/yandex/manager.h>

#include <drive/backend/database/drive_api.h>
#include <drive/backend/user_devices/manager.h>
#include <drive/backend/users/blacklist_external.h>

#include <drive/library/cpp/mds/client.h>
#include <drive/library/cpp/webphone/client.h>

namespace NDrive {
    class IServer;
}

class TSupportRequestsDistributonManager {
public:
    TSupportRequestsDistributonManager(const NDrive::IServer* server, const TString& notifierName);

    bool ResolveSupportRequests(const TVector<TString>& tagNames) const;
    bool ResolveSupportRequests(const TString& tagName, const TInstant actuality) const;

private:
    bool AssignSupportOperator(const TDBTag& supportRequest, const TUserPermissions::TPtr permissions) const;
    void MaybeNotify(const TString& message) const;

private:
    const NDrive::IServer* Server;
    const TString NotifierName;
};

class TLoadBalanceCalendarHelper {
public:
    using TCalendar = TMap< TString, TMap<TString, NCallCenterYandex::TAppDistribution> >;  // (day of week, { (hour, distribution), ... }), ...

    explicit TLoadBalanceCalendarHelper(const TString& applicationName);

    TMaybe<NCallCenterYandex::TAppDistribution> GetCurrentValue() const;
    bool SetCurrentValue(const NCallCenterYandex::TAppDistribution& update);

    NJson::TJsonValue SerializeToJson() const;
    bool DeserializeFromJson(const NJson::TJsonValue& data);

private:
    std::pair<TString, TString> GetCurrentWeekDayHourPair() const;

private:
    R_READONLY(TString, ApplicationName);
    R_READONLY(TCalendar, CalendarData);
};

class TWebPhoneCallConditionConstructor {
public:
    static TString BuildCondition(const TSet<TWebPhoneCall::TId>& ids, NDrive::TEntitySession& session) {
        return "id IN (" + session->Quote(ids) + ")";
    }

    static NStorage::TTableRecord BuildCondition(const TWebPhoneCall::TId& id) {
        NStorage::TTableRecord trCondition;
        trCondition.Set("id", id);
        return trCondition;
    }

    static NStorage::TTableRecord BuildCondition(const TWebPhoneCall& object) {
        return BuildCondition(object.GetId());
    }
};

class TWebPhoneCallManager: public TDatabaseEntitiesManager<TWebPhoneCall, TWebPhoneCallConditionConstructor> {
public:
    enum ECallDataError {
        InternalError /* "internal_error" */,
        NotFoundData /* "not_found_data" */,
        RequestError /* "request_error" */,
        WrongData /* "wrong_data" */,
        Ok /* "ok" */,
    };

private:
    using TBase = TDatabaseEntitiesManager<TWebPhoneCall, TWebPhoneCallConditionConstructor>;
    R_FIELD(TString, CallIdPrefix);
    R_FIELD(TString, BucketName);

public:
    TWebPhoneCallManager(const IHistoryContext& context, const TWebphoneClientConfig& config)
        : TBase(context)
        , CallIdPrefix(config.GetCallIdPrefix())
        , BucketName(config.GetBucketName())
    {
    }

    ECallDataError GetCallData(const NDrive::IServer& server, const TWebPhoneCall::TId callId, TWebPhoneCall& call, TMessagesCollector& errors) const;
    TString ConvertOriginCallId(const TWebPhoneCall::TId id) const;
    bool TryParseOriginCallId(const TString& callId, TWebPhoneCall::TId& result) const;

private:
    ECallDataError GetCallData(const NDrive::IServer& server, const TString& callId, TWebPhoneCall& call, TMessagesCollector& errors) const;
};


class TSupportCenterManager
    : public ISupportCenterManager
    , public TDatabaseSessionConstructor
{
public:
    using EProcessCallResult = NCallCenterYandex::EInternalCallProcessResult;

public:
    TSupportCenterManager(const NDrive::IServer* server, const TSupportCenterManagerConfig& config);

    virtual void Start() override;
    virtual void Stop() override;

    virtual NDrive::TEntitySession BuildSession(const bool readOnly = false) const override;

    virtual TSupportRequestCategorizationDB* GetSupportRequestCategorizer() const override;
    virtual TLegacyPriorityManager* GetLegacyPriorityManager() const override;
    virtual TWebphoneClient* GetWebphoneClient() const override;
    virtual NCallCenterYandex::TCallCenterYandexClient* GetCallCenterYandexClient() const override;
    virtual const NCallCenterYandex::TInternalCallCenterClient& GetInternalCallCenterClient() const override;
    virtual const TAudioteleCallsManager& GetAudioteleCallsManager() const override;
    virtual const TWebPhoneCallManager& GetWebPhoneCallManager() const override;
    virtual const TCiptCallEventsReader& GetCiptCallEventsReader() const override;
    virtual const TSupportAICallManager& GetSupportAICallManager() const override;
    virtual TS3Client* GetMDSClient() const override;

    virtual TMaybe<TLoadBalanceCalendarHelper> GetLoadBalanceCalendar(const TString& appName) const override;
    virtual bool UpdateLoadBalanceCalendar(const TLoadBalanceCalendarHelper& calendar, const TString& performerId) const override;

    virtual TMaybe<NCallCenterYandex::TAppDistribution> GetCurrentLoadBalanceCalendarValue(const TString& appName) const override;
    virtual bool UpdateCurrentLoadBalance(const NCallCenterYandex::TAppDistribution& value, const TString& performerId, TMessagesCollector& errors, bool updateCalendar = true) const override;

    virtual NThreading::TFuture<std::pair<EProcessCallResult, TString>> ProcessInternalCall(const NS3::TBucket& bucket, const TString& callId, const bool needCheck = false) const override;

private:
    const NDrive::IServer* Server;

    TWebPhoneCallManager WebPhoneCallManager;
    TAudioteleCallsManager AudioteleCallsManager;
    TCiptCallEventsReader CiptCallEventsReader;
    NCallCenterYandex::TInternalCallCenterClient InternalCallCenterClient;
    TSupportAICallManager SupportAICallManager;
    THolder<TSupportRequestCategorizationDB> Categorizer;
    THolder<TLegacyPriorityManager> LegacyPriorityManager;
    THolder<TWebphoneClient> WebphoneClient;
    THolder<NCallCenterYandex::TCallCenterYandexClient> CallCenterYandexClient;
    THolder<TS3Client> MDSClient;
};
