#include "request_filter.h"

#include <drive/backend/chat_robots/configuration/chat_script.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/tags/tags_search.h>

void TBaseSupportRequestsFilter::InitFromCgi(const NDrive::IServer* server, const TCgiParameters& cgi, TUserPermissionsConstPtr permissions) {
    Permissions = permissions;
    if (cgi.Has("tag_names")) {
        auto preTagNames = MakeSet(GetStrings(cgi, "tag_names"));
        auto tagDescrs = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(GetProcessedTagType());
        for (auto&& descr : tagDescrs) {
            if (preTagNames.contains(descr->GetName())) {
                TagNames.emplace(descr->GetName());
            }
        }
    }
    {
        auto performerId = GetString(cgi, "performer_id");
        if (performerId) {
            if (performerId == "0") {
                PerformStatus = EPerformStatus::NotPerformed;
            } else {
                PerformStatus = TChatSupportRequestsFilter::EPerformStatus::Performed;
                PerformerIds.emplace(performerId);
            }
        }
    }
    {
        auto userIds = GetStrings(cgi, "user_id");
        if (userIds) {
            UserIds = MakeSet(userIds);
        }
    }
    DoInitFromCgi(cgi);
}

TMaybe<TVector<TConstDBTag>> TBaseSupportRequestsFilter::GetTagsForProcessing(const NDrive::IServer* server, TMessagesCollector& errors) const {
    TVector<TConstDBTag> requestTags;
    TSet<TString> queryTagNames = GetQueryTagNames(server);
    if (queryTagNames.empty()) {
        return requestTags;
    }
    TVector<TDBTag> dbTags;
    if (PerformerIds.empty()) {
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        auto optionalTags = server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TVector<TString>(), MakeVector(queryTagNames), session);
        if (!optionalTags) {
            errors.AddMessage("SupportRequestFilter", session.GetStringReport());
            return {};
        }
        dbTags = std::move(optionalTags.GetRef());
    } else {
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RestorePerformerTags(MakeVector(queryTagNames), MakeVector(PerformerIds), dbTags, session)) {
            errors.AddMessage("SupportRequestFilter", session.GetStringReport());
            return {};
        }
    }
    for (auto&& tag : dbTags) {
        requestTags.emplace_back(std::move(tag));
    }

    return requestTags;
}

TVector<TString> TBaseSupportRequestsFilter::GetStrings(const TCgiParameters& cgi, TStringBuf name) const {
    TVector<TString> result;
    auto range = cgi.equal_range(name);
    for (auto i = range.first; i != range.second; ++i) {
        StringSplitter(i->second).SplitBySet(",").SkipEmpty().AddTo(&result);
    }
    return result;
}

TString TBaseSupportRequestsFilter::GetString(const TCgiParameters& cgi, TStringBuf name) const {
    auto result = GetStrings(cgi, name);
    if (result.empty()) {
        return "";
    }
    return result.front();
}

TSet<TString> TBaseSupportRequestsFilter::GetQueryTagNames(const NDrive::IServer* server) const {
    auto permissions = GetPermissions();
    if (!permissions) {
        return {};
    }
    TVector<TString> queryTagNames;
    if (TagNames.contains("@" + GetProcessedTagType())) {
        auto tagDescrs = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(GetProcessedTagType());
        for (auto&& descr : tagDescrs) {
            queryTagNames.emplace_back(descr->GetName());
        }
    } else if (TagNames.empty()) {
        auto tagDescrs = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(GetProcessedTagType());
        for (auto&& descr : tagDescrs) {
            if (!GetExcludedTagsByDefault().contains(descr->GetName())) {
                queryTagNames.emplace_back(descr->GetName());
            }
        }
    } else {
        queryTagNames = MakeVector(TagNames);
    }
    return MakeSet(permissions->FilterTagNames(queryTagNames, TTagAction::ETagAction::Observe));
}

TSet<TString> TChatSupportRequestsFilter::GetExcludedTagsByDefault() const {
    return TSet<TString>(
        {
            TSupportChatTag::DeferredName,
            TSupportChatTag::FeedbackName
        }
    );
}

NDrive::TScheme TChatMetricFilter::GetScheme() {
    NDrive::TScheme scheme;
    scheme.Add<TFSVariants>("metric_type", "Тип метрики").InitVariants<EMetricType>();
    scheme.Add<TFSNumeric>("min_value", "Минимальное допустимое значение метрики").SetDefault(Min<TMetricValueType>()).SetRequired(true);
    scheme.Add<TFSNumeric>("max_value", "Максимальное допустимое значение метрики").SetDefault(Max<TMetricValueType>()).SetRequired(true);
    scheme.Add<TFSBoolean>("split_by_operator", "Добавить разбивку по операторам").SetDefault(false);
    return scheme;
}

template <>
NJson::TJsonValue NJson::ToJson(const TChatMetricFilter& filter) {
    NJson::TJsonValue result;
    NJson::InsertField(result, "metric_type", NJson::Stringify(filter.GetType()));
    NJson::InsertField(result, "min_value", filter.GetMinValue());
    NJson::InsertField(result, "max_value", filter.GetMaxValue());
    NJson::InsertField(result, "split_by_operator", filter.GetSplitByOperator());
    if (filter.GetMessageTTL() != TDuration::Max()) {
        NJson::InsertField(result, "message_ttl", filter.GetMessageTTL().Seconds());
    }
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TChatMetricFilter& result) {
    return
        NJson::ParseField(value, "metric_type", NJson::Stringify(result.MutableType()), true) &&
        NJson::ParseField(value, "min_value", result.MutableMinValue(), true) &&
        NJson::ParseField(value, "max_value", result.MutableMaxValue(), true) &&
        NJson::ParseField(value, "split_by_operator", result.MutableSplitByOperator(), false) &&
        NJson::ParseField(value, "message_ttl", NJson::Hr(result.MutableMessageTTL()));
}

NJson::TJsonValue TChatMetricFilter::SerializeToJson() const {
    NJson::TJsonValue result;
    result["metric_type"] = ToString(Type);
    result["min_value"] = MinValue;
    result["max_value"] = MaxValue;
    result["split_by_operator"] = SplitByOperator;
    TJsonProcessor::WriteDurationInt(result, "message_ttl", MessageTTL, TDuration::Max());
    return result;
}

bool TChatMetricFilter::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_FROM_STRING(jsonInfo, "metric_type", Type);
    JREAD_INT(jsonInfo, "min_value", MinValue);
    JREAD_INT(jsonInfo, "max_value", MaxValue);
    JREAD_BOOL_OPT(jsonInfo, "split_by_operator", SplitByOperator);
    JREAD_DURATION_OPT(jsonInfo, "message_ttl", MessageTTL);
    return true;
}

TString TFilteredSupportChat::GetYasmBucketName() const {
    if (!Performer) {
        return Tag->GetName();
    } else {
        return Tag->GetName() + "-" + Performer;
    }
}

TString TFilteredSupportChat::GetOverallBucketName() const {
    if (!Performer) {
        return "overall";
    } else {
        return "overall-" + Performer;
    }
}

NJson::TJsonValue TFilteredSupportChat::BuildReport() const {
    NJson::TJsonValue result;
    result["metric_value"] = MetricValue;
    result["chat_tag"] = Tag.BuildJsonReport();
    result["last_message"] = DisplayMessage.SerializeToJson();
    return result;
}

NDrive::TScheme TChatSupportRequestsFilter::GetScheme() {
    NDrive::TScheme scheme;
    scheme.Add<TFSString>("chat_robot_id", "id робота").SetRequired(true);
    scheme.Add<TFSArray>("node_names", "Имена вершин для фильтра").SetElement<TFSString>();
    scheme.Add<TFSArray>("banned_node_names", "Имена вершин для исключения из фильтра").SetElement<TFSString>();
    scheme.Add<TFSArray>("tag_names", "Имена тегов-линий").SetElement<TFSString>();
    scheme.Add<TFSArray>("user_ids", "Пользователи").SetElement<TFSString>();
    scheme.Add<TFSArray>("performer_ids", "Операторы").SetElement<TFSString>();
    scheme.Add<TFSArray>("original_support_lines", "Изначальная линия поддержки").SetElement<TFSString>();
    scheme.Add<TFSVariants>("chat_status", "Статус чата").InitVariants<EChatStatus>();
    scheme.Add<TFSVariants>("perform_status", "Статус исполнения").InitVariants<EPerformStatus>();
    scheme.Add<TFSVariants>("muted_status", "Замьюченность").InitVariants<EMutedStatus>();
    scheme.Add<TFSNumeric>("visible_traits", "Видимые трейты").SetDefault(0);
    scheme.Add<TFSStructure>("metric_filter", "Основной фильтр по метрике").SetStructure(TChatMetricFilter::GetScheme());
    scheme.Add<TFSDuration>("message_ttl", "Исключать все сообщения старше чем");
    scheme.Add<TFSArray>("additional_metric_filters", "Дополнительные фильтры по метрике").SetElement<TFSStructure>().SetStructure(TChatMetricFilter::GetScheme());
    return scheme;
}

NJson::TJsonValue TChatSupportRequestsFilter::SerializeToJson() const {
    NJson::TJsonValue result;
    result["chat_robot_id"] = ChatRobotId;
    TJsonProcessor::WriteContainerArray(result, "node_names", NodeNames);
    TJsonProcessor::WriteContainerArray(result, "banned_node_names", BannedNodeNames);
    TJsonProcessor::WriteContainerArray(result, "tag_names", GetTagNames());
    TJsonProcessor::WriteContainerArray(result, "user_ids", GetUserIds());
    TJsonProcessor::WriteContainerArray(result, "performer_ids", GetPerformerIds());
    TJsonProcessor::WriteContainerArray(result, "original_support_lines", OriginalSupportLines);
    result["chat_status"] = ToString(ChatStatus);
    result["perform_status"] = ToString(GetPerformStatus());
    result["muted_status"] = ToString(MutedStatus);
    result["visible_traits"] = VisibleTraits;
    NJson::InsertField(result, "message_ttl", NJson::Hr(MessageTTL));
    NJson::InsertField(result, "metric_filter", MetricFilter);
    NJson::InsertField(result, "additional_metric_filters", AdditionalMetricFilters);
    if (HasChatTopic()) {
        result["chat_topic"] = GetChatTopicUnsafe();
    }
    return result;
}

bool TChatSupportRequestsFilter::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    TJsonProcessor::ReadContainer(jsonInfo, "node_names", NodeNames);
    TJsonProcessor::ReadContainer(jsonInfo, "banned_node_names", BannedNodeNames);
    TJsonProcessor::ReadContainer(jsonInfo, "tag_names", MutableTagNames());
    TJsonProcessor::ReadContainer(jsonInfo, "user_ids", MutableUserIds());
    TJsonProcessor::ReadContainer(jsonInfo, "performer_ids", MutablePerformerIds());
    TJsonProcessor::ReadContainer(jsonInfo, "original_support_lines", OriginalSupportLines);
    if (!NJson::ParseField(jsonInfo, "metric_filter", MetricFilter)) {
        return false;
    }
    if (MetricFilter.GetMessageTTL()) {
        MessageTTL = MetricFilter.GetMessageTTL();
    }
    return
        NJson::ParseField(jsonInfo, "additional_metric_filters", AdditionalMetricFilters, false) &&
        NJson::ParseField(jsonInfo, "chat_robot_id", ChatRobotId) &&
        NJson::ParseField(jsonInfo, "perform_status", NJson::Stringify(MutablePerformStatus())) &&
        NJson::ParseField(jsonInfo, "muted_status", NJson::Stringify(MutedStatus)) &&
        NJson::ParseField(jsonInfo, "visible_traits", VisibleTraits, false) &&
        NJson::ParseField(jsonInfo, "chat_status", NJson::Stringify(ChatStatus)) &&
        NJson::ParseField(jsonInfo, "chat_topic", ChatTopic, false) &&
        NJson::ParseField(jsonInfo, "message_ttl", NJson::Hr(MessageTTL), false);
}

TString TChatSupportRequestsFilter::BuildUserReport(const NDrive::IServer* server, const TString& userId) const {
    if (server && server->GetDriveAPI() && server->GetDriveAPI()->GetUsersData()) {
        auto userData = server->GetDriveAPI()->GetUsersData()->GetCachedObject(userId);
        if (userData.Defined()) {
            return userData->GetHRReport();
        }
    }
    return "[error, user id: " + userId + "]";
}

TString TChatSupportRequestsFilter::GetLogin(const NDrive::IServer* server, const TString& userId) const {
    if (server && server->GetDriveAPI() && server->GetDriveAPI()->GetUsersData()) {
        auto userData = server->GetDriveAPI()->GetUsersData()->GetCachedObject(userId);
        if (userData.Defined()) {
            return userData->GetLogin();
        }
    }
    return userId;
}

TString TChatSupportRequestsFilter::BuildHRReport(const NDrive::IServer* server, const TFilteredSupportChat& chat) const {
    TString chatReport;
    {
        auto chatTag = chat.GetTag().GetTagAs<TSupportChatTag>();
        if (!chatTag) {
            chatReport = "[error, tag id: " + chat.GetTag().GetTagId() + "]";
        } else {
            chatReport = chatTag->BuildChatLink(chat.GetTag().GetObjectId());
        }
    }

    TString userReport = BuildUserReport(server, chat.GetTag().GetObjectId());

    TString report = "Чат " + chatReport + " от пользователя " + userReport;
    if (!!chat.GetTag()->GetPerformer()) {
        auto performerReport = BuildUserReport(server, chat.GetTag()->GetPerformer());
        report += " назначенный на оператора " + performerReport;
    } else {
        report += " без исполнителя";
    }

    if (MetricFilter.GetType() == TChatMetricFilter::EMetricType::AbsoluteWait) {
        report += " ожидает ответа уже ";
        report += ToString(chat.GetMetricValue() / 60) + " м. ";
        report += ToString(chat.GetMetricValue() % 60) + " с.";
    }

    return report;
}

TFilteredSupportChat::TMessagesCount CountMessages(const NDrive::NChat::TMessageEvents& messages, const ui64 lastViewedId) {
    TFilteredSupportChat::TMessagesCount result;
    result.Total = messages.size();
    result.Unread = 0;
    for (auto& message : messages) {
        if (message.GetType() != NDrive::NChat::TMessage::EMessageType::Separator) {
            result.Total += 1;
            if (message.GetHistoryEventId() > lastViewedId) {
                result.Unread += 1;
            }
        }
    }
    return result;
}

bool TChatSupportRequestsFilter::GetSupportRequests(const NDrive::IServer* server, TVector<TFilteredSupportChat>& result, const TInstant actuality, const TString& requestUserId, TMessagesCollector& errors) const {
    if (!server || !server->GetDriveAPI()) {
        errors.AddMessage("ChatSupportRequestsFilter", "server not configured");
        return false;
    }

    auto queryTagNames = GetQueryTagNames(server);
    auto requestTags = GetTagsForProcessing(server, errors);
    if (!requestTags) {
        return false;
    }
    auto currentInstant = Now();

    auto chatRobots = server->GetChatRobots();
    TSet<TString> chatsToUpdate;
    TVector<TRequestChatInfo> requestChats;
    for (auto&& tag : *requestTags) {
        auto userId = tag.GetObjectId();
        auto permissions = GetPermissions();
        if (permissions && !permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, "", userId)) {
            continue;
        }

        auto impl = tag.GetTagAs<TSupportChatTag>();
        if (!impl) {
            continue;
        }

        if (GetPerformStatus() != EPerformStatus::Any) {
            bool hasPerformer = !!impl->GetPerformer();
            bool needPerformer = GetPerformStatus() == EPerformStatus::Performed;
            if (hasPerformer != needPerformer) {
                continue;
            }
        }
        if (!GetPerformerIds().empty() && !GetPerformerIds().contains(impl->GetPerformer())) {
            continue;
        }

        if (OriginalSupportLines.size() && !OriginalSupportLines.contains(impl->GetOriginalSupportLine())) {
            continue;
        }

        if (MutedStatus != EMutedStatus::Any) {
            bool isMuted = impl->IsMuted();
            bool needMuted = MutedStatus == EMutedStatus::Muted;
            if (isMuted != needMuted) {
                continue;
            }
        }

        if (!queryTagNames.contains(impl->GetName())) {
            continue;
        }

        TString chatRobotId;
        TString chatRobotTopic;
        IChatRobot::ParseTopicLink(impl->GetTopicLink(), chatRobotId, chatRobotTopic);

        if (HasChatTopic() && GetChatTopicUnsafe() != chatRobotTopic) {
            continue;
        }

        if (ChatRobotId && chatRobotId != ChatRobotId) {
            continue;
        }

        if (permissions && !permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ChatRobot, "", chatRobotId)) {
            continue;
        }
        auto chatRobotIt = chatRobots.find(chatRobotId);
        if (chatRobotIt == chatRobots.end() || !chatRobotIt->second) {
            continue;
        }
        auto chatRobot = chatRobotIt->second;
        if (!NodeNames.empty() || !BannedNodeNames.empty()) {
            TChatRobotScriptItem currentScriptItem;
            if (!chatRobot->GetCurrentScriptItem(userId, chatRobotTopic, currentScriptItem, actuality)) {
                continue;
            }
            if (!NodeNames.empty() && !NodeNames.contains(currentScriptItem.GetId())) {
                continue;
            }
            if (!BannedNodeNames.empty() && BannedNodeNames.contains(currentScriptItem.GetId())) {
                continue;
            }
        }
        requestChats.emplace_back(tag, chatRobotId, chatRobotTopic, chatRobot, chatRobot->GetLastViewedMessageId(userId, chatRobotTopic, requestUserId));
        chatsToUpdate.emplace(chatRobot->GetChatSearchId(userId, chatRobotTopic));
    }

    {
        auto session = server->GetChatEngine()->BuildSession(true);
        if (!server->GetChatEngine()->UpdateCachedMessages(chatsToUpdate, session, Now())) {
            errors.AddMessage("ChatSupportRequestsFilter", session.GetStringReport());
            return false;
        }
    }

    for (auto&& requestChat : requestChats) {
        auto& tag = requestChat.Tag;
        auto chatRobot = requestChat.ChatRobot;
        auto userId = tag.GetObjectId();

        TChatMetricFilter::TMetricValueType metricValue;
        NDrive::NChat::TMessageEvents messages;
        TMaybe<NDrive::NChat::TMessageEvent> lastMessage;
        TFilteredSupportChat::TMessagesCount messagesCount;
        {
            auto expectedMessages = chatRobot->GetCachedMessages(chatRobot->GetChatSearchId(userId, requestChat.Topic));
            if (!expectedMessages) {
                errors.AddMessage("ChatSupportRequestsFilter", expectedMessages.GetError().GetReport().GetStringRobust());
                return false;
            }
            messages = std::move(*expectedMessages);
            if (messages.size()) {
                lastMessage = messages.back();
                messagesCount = CountMessages(messages, requestChat.LastViewedMessageId);
            }
            if (!IsMatchingMetricAllMessages(messages, tag, MetricFilter, metricValue)) {
                continue;
            }
            ui32 additionalValue;
            bool isMatching = IsMatchingMetricAllMessages(messages, tag, MetricFilter, metricValue);
            for (auto&& it = AdditionalMetricFilters.begin(); it != AdditionalMetricFilters.end() && isMatching; ++it) {
                isMatching &= IsMatchingMetricAllMessages(messages, tag, *it, additionalValue);
            }
            if (!isMatching) {
                continue;
            }

            RemoveInsignificantMessages(messages, tag, server);
            if (!messages.size()) {
                continue;
            }
        }

        if (GetMessageTTL() != TDuration::Max() && messages.front().GetHistoryInstant() + GetMessageTTL() < currentInstant) {
            size_t removalOffset = 0;
            while (removalOffset < messages.size() && messages[removalOffset].GetHistoryInstant() + GetMessageTTL() < currentInstant) {
                ++removalOffset;
            }
            if (removalOffset == messages.size()) {
                continue;
            }
            if (removalOffset) {
                NDrive::NChat::TMessageEvents truncatedView;
                truncatedView.reserve(messages.size() - removalOffset);
                for (size_t i = removalOffset; i < messages.size(); ++i) {
                    truncatedView.emplace_back(messages[i]);
                }
                messages = std::move(truncatedView);
            }
        }

        if (!IsChatStatusMatching(messages, userId)) {
            continue;
        }

        ui32 additionalValue;
        bool isMatching = IsMatchingMetric(messages, tag, MetricFilter, metricValue);
        for (auto&& it = AdditionalMetricFilters.begin(); it != AdditionalMetricFilters.end() && isMatching; ++it) {
            isMatching &= IsMatchingMetric(messages, tag, *it, additionalValue);
        }
        if (!isMatching) {
            continue;
        }

        if (!MetricFilter.GetSplitByOperator()) {
            result.emplace_back(TFilteredSupportChat(tag, metricValue, GetDisplayMessage(messages), lastMessage, messagesCount));
        } else {
            AddSplittedChatStats(tag, messages, metricValue, result, server, lastMessage, messagesCount);
        }
    }

    return true;
}

NDrive::NChat::TMessageEvent TChatSupportRequestsFilter::GetDisplayMessage(const NDrive::NChat::TMessageEvents& messages) const {
    auto displayMessage = messages.back();
    for (auto it = messages.rbegin(); it != messages.rend(); ++it) {
        if (!it->GetHistoryUserId().StartsWith("robot") && it->IsVisible(VisibleTraits) && it->GetType() != NDrive::NChat::TMessage::EMessageType::Separator) {
            displayMessage = *it;
            break;
        }
    }
    return displayMessage;
}

void TChatSupportRequestsFilter::RemoveInsignificantMessages(NDrive::NChat::TMessageEvents& messages, const TConstDBTag& tag, const NDrive::IServer* server) const {
    bool deleteSeparator = true;
    bool deleteFromRobot = true;
    TSet<TSupportChatTag::EInsignificantMessageType> includeMessages;
    auto tagDescription = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(tag->GetName());
    if (tagDescription) {
        auto description = tagDescription->GetAs<TSupportChatTag::TDescription>();
        if (description) {
            includeMessages = description->GetIncludedFilterMessageTypes();
            deleteSeparator = !includeMessages.contains(TSupportChatTag::EInsignificantMessageType::Separator);
            deleteFromRobot = !includeMessages.contains(TSupportChatTag::EInsignificantMessageType::Robot);
        }
    }
    while (messages.size() && ((!messages.back().IsVisible(VisibleTraits))
                || (deleteSeparator && messages.back().GetType() == NDrive::NChat::TMessage::EMessageType::Separator)
                || (deleteFromRobot && messages.back().GetHistoryUserId().StartsWith("robot-")))) {
        messages.pop_back();
    }
}

void TChatSupportRequestsFilter::AddSplittedChatStats(const TConstDBTag& tag, const NDrive::NChat::TMessageEvents& messages, const TChatMetricFilter::TMetricValueType metricValue, TVector<TFilteredSupportChat>& result, const NDrive::IServer* server, const TMaybe<NDrive::NChat::TMessageEvent>& lastMessage, const TFilteredSupportChat::TMessagesCount count) const {
    if (MetricFilter.GetType() != TChatMetricFilter::EMetricType::OutgoingMessages) {
        TString performer = tag->GetPerformer();
        if (!performer) {
            performer = "null";
        }
        result.emplace_back(TFilteredSupportChat(tag, metricValue, GetDisplayMessage(messages), lastMessage, count, performer));
    } else {
        TMap<TString, size_t> messagesCount;
        for (auto&& message : messages) {
            if (message.GetHistoryUserId() == tag.GetObjectId() || message.GetHistoryUserId().StartsWith("robot")) {
                continue;
            }
            if (!message.IsVisible(VisibleTraits)) {
                continue;
            }
            messagesCount[message.GetHistoryUserId()] += 1;
        }
        for (auto&& it : messagesCount) {
            result.emplace_back(TFilteredSupportChat(tag, it.second, GetDisplayMessage(messages), lastMessage, count, GetLogin(server, it.first)));
        }
    }
}

bool TChatSupportRequestsFilter::IsChatStatusMatching(const NDrive::NChat::TMessageEvents& messages, const TString& chatOwnerId) const {
    if (ChatStatus == EChatStatus::Any) {
        return true;
    }

    if (messages.back().GetHistoryUserId() != chatOwnerId) {
        if (ChatStatus != EChatStatus::Any) {
            return false;
        }
    }

    if (ChatStatus == EChatStatus::WaitingForInitialAnswer) {
        for (auto&& message : messages) {
            if (message.GetHistoryUserId() != chatOwnerId && !message.GetHistoryUserId().StartsWith("robot")) {
                return false;
            }
        }
    }

    return true;
}

bool TChatSupportRequestsFilter::IsMatchingMetric(const NDrive::NChat::TMessageEvents& messages, const TConstDBTag& tag, const TChatMetricFilter& metricFilter, ui32& metricValue) const {
    auto chatOwnerId = tag.GetObjectId();

    bool result;
    switch (metricFilter.GetType()) {
        case TChatMetricFilter::EMetricType::One: {
            if (metricFilter.GetMinValue() <= 1 && 1 <= MetricFilter.GetMaxValue()) {
                metricValue = 1;
                result = true;
            } else {
                result = false;
            }
            break;
        }
        case TChatMetricFilter::EMetricType::OutgoingMessages: {
            TChatMetricFilter::TMetricValueType preMetricValue = 0;
            for (auto&& message : messages) {
                if (message.GetHistoryUserId() == chatOwnerId || message.GetHistoryUserId().StartsWith("robot")) {
                    continue;
                }
                if (!message.IsVisible(VisibleTraits)) {
                    continue;
                }
                preMetricValue += 1;
            }
            if (metricFilter.GetMinValue() <= preMetricValue && preMetricValue <= metricFilter.GetMaxValue()) {
                metricValue = preMetricValue;
                result = true;
            } else {
                result = false;
            }
            break;
        }
        case TChatMetricFilter::EMetricType::SinceLastMessage: {
            if (messages.empty()) {
                result = false;
                break;
            }
            auto preMetricValue = (Now() - messages.back().GetHistoryInstant()).Seconds();
            if (metricFilter.GetMinValue() <= preMetricValue && preMetricValue <= metricFilter.GetMaxValue()) {
                metricValue = preMetricValue;
                result = true;
            } else {
                result = false;
            }
            break;
        }
        case TChatMetricFilter::EMetricType::SinceLastPerformerMessage: {
            result = true;
            break;
        }
        case TChatMetricFilter::EMetricType::AbsoluteWait: {
            if (messages.empty()) {
                result = false;
                break;
            }
            if (messages.back().GetHistoryUserId() != chatOwnerId) {
                if (metricFilter.GetMinValue() <= 0 && 0 <= metricFilter.GetMaxValue()) {
                    metricValue = 0;
                    result = true;
                } else {
                    result = false;
                }
                break;
            }
            // no break;
        }
        default: {
            if (messages.empty()) {
                result = false;
                break;
            }

            size_t groupStart = messages.size() - 1;
            if (metricFilter.GetType() == TChatMetricFilter::EMetricType::SinceLastUserGroupStart) {
                while (groupStart > 0 && messages[groupStart].GetHistoryUserId() != chatOwnerId) {
                    --groupStart;
                }
            }

            while (groupStart > 0 && messages[groupStart - 1].GetHistoryUserId() == chatOwnerId) {
                --groupStart;
            }

            auto metricValueDuration = Now() - messages[groupStart].GetHistoryInstant();
            if (metricFilter.GetMinValue() <= metricValueDuration.Seconds() && metricValueDuration.Seconds() <= metricFilter.GetMaxValue()) {
                metricValue = metricValueDuration.Seconds();
                result = true;
            } else {
                result = false;
            }
        }
    }

    return result;
}

bool TChatSupportRequestsFilter::IsMatchingMetricAllMessages(const NDrive::NChat::TMessageEvents& messages, const TConstDBTag& tag, const TChatMetricFilter& metricFilter, ui32& metricValue) const {
    auto supportChatTag = tag.GetTagAs<TSupportChatTag>();
    if (!supportChatTag) {
        return true;
    }
    if (metricFilter.GetType() == TChatMetricFilter::EMetricType::SinceLastPerformerMessage) {
        for (auto it = messages.rbegin(); it != messages.rend(); ++it) {
            if (supportChatTag->GetPerformer() == it->GetHistoryUserId() && it->IsVisible(VisibleTraits)) {
                auto messageInstant = (ModelingNow() - it->GetHistoryInstant()).Seconds();
                if (metricFilter.GetMinValue() <= messageInstant && messageInstant <= metricFilter.GetMaxValue()) {
                    metricValue = messageInstant;
                    return true;
                } else {
                    return false;
                }
            }
        }
        return false;
    }
    return true;
}

bool TSimpleSupportRequestsFilter::GetSupportRequests(const NDrive::IServer* server, TVector<TBasicFilteredSupportRequest>& result, TMessagesCollector& errors) const {
    if (!server || !server->GetDriveAPI()) {
        errors.AddMessage("SimpleSupportRequestsFilter", "server not configured");
        return false;
    }

    auto queryTagNames = GetQueryTagNames(server);
    auto requestTags = GetTagsForProcessing(server, errors);
    if (!requestTags) {
        return false;
    }

    result.clear();
    for (auto&& tag : *requestTags) {
        auto userId = tag.GetObjectId();
        auto permissions = GetPermissions();
        if (permissions && !permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, "", userId)) {
            continue;
        }
        if (GetPerformStatus() != EPerformStatus::Any) {
            bool hasPerformer = !!tag->GetPerformer();
            bool needPerformer = GetPerformStatus() == EPerformStatus::Performed;
            if (hasPerformer != needPerformer) {
                continue;
            }
        }
        if (!GetPerformerIds().empty() && !GetPerformerIds().contains(tag->GetPerformer())) {
            continue;
        }
        if (!queryTagNames.contains(tag->GetName())) {
            continue;
        }
        result.push_back(TBasicFilteredSupportRequest(tag));
    }

    return true;
}
