#pragma once

#include <drive/backend/chat/engine.h>
#include <drive/backend/chat_robots/ifaces.h>
#include <drive/backend/data/container_tag.h>
#include <drive/backend/data/support_tags.h>
#include <drive/backend/data/dedicated_fleet.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/roles/permissions.h>

#include <rtline/util/types/accessor.h>

class TBasicFilteredSupportRequest {
    R_READONLY(TConstDBTag, Tag);

public:
    TBasicFilteredSupportRequest(const TConstDBTag& tag)
        : Tag(tag)
    {
    }
};

class TBaseSupportRequestsFilter {
public:
    enum class EPerformStatus: ui32 {
        Any = 0 /* "any" */,
        Performed = 1 /* "performed" */,
        NotPerformed = 2 /* "not_performed" */,
    };

private:
    R_FIELD(TSet<TString>, TagNames);
    R_FIELD(TSet<TString>, UserIds);
    R_FIELD(TSet<TString>, PerformerIds);
    R_FIELD(EPerformStatus, PerformStatus, EPerformStatus::Any);

protected:
    virtual TString GetProcessedTagType() const = 0;
    virtual TSet<TString> GetExcludedTagsByDefault() const = 0;

    virtual void DoInitFromCgi(const TCgiParameters& /*cgi*/) const {
    }

    TVector<TString> GetStrings(const TCgiParameters& cgi, TStringBuf name) const;
    TString GetString(const TCgiParameters& cgi, TStringBuf name) const;
    TMaybe<TVector<TConstDBTag>> GetTagsForProcessing(const NDrive::IServer* server, TMessagesCollector& errors) const;

public:
    virtual void InitFromCgi(const NDrive::IServer* server, const TCgiParameters& cgi, TUserPermissionsConstPtr permissions = nullptr) final;
    TSet<TString> GetQueryTagNames(const NDrive::IServer* server) const;

    void SetPermissions(TUserPermissionsConstPtr permissions) {
        Permissions = permissions;
    }

    TUserPermissionsConstPtr GetPermissions() const {
        return Permissions;
    }

    virtual ~TBaseSupportRequestsFilter() = default;

private:
    TUserPermissionsConstPtr Permissions = nullptr;
};

class TSimpleSupportRequestsFilter : public TBaseSupportRequestsFilter {
public:
    bool GetSupportRequests(const NDrive::IServer* server, TVector<TBasicFilteredSupportRequest>& result, TMessagesCollector& errors) const;
};

class TChatMetricFilter {
public:
    using TMetricValueType = ui32;

    enum class EMetricType: ui32 {
        One = 0 /* "one" */,
        AbsoluteWait = 1 /* "absolute_wait" */,
        OutgoingMessages = 2 /* "outgoing_messages" */,
        SinceLastMessage = 3 /* "since_last_message" */,
        SinceLastPerformerMessage = 4 /* "since_last_performer_message" */,
        SinceLastUserGroupStart = 5 /* "since_last_user_group_start" */
    };

private:
    R_FIELD(EMetricType, Type, EMetricType::One);
    R_FIELD(TMetricValueType, MinValue, Min<TMetricValueType>());
    R_FIELD(TMetricValueType, MaxValue, Max<TMetricValueType>());
    R_FIELD(bool, SplitByOperator, false);
    R_FIELD(TDuration, MessageTTL, TDuration::Max());

public:
    static NDrive::TScheme GetScheme();
    NJson::TJsonValue SerializeToJson() const;
    bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo);
};

struct TRequestChatInfo {
    TConstDBTag& Tag;
    TString ChatId;
    TString Topic;
    TAtomicSharedPtr<IChatRobot> ChatRobot;
    ui64 LastViewedMessageId;

    TRequestChatInfo(TConstDBTag& tag, const TString& chatId, const TString& topic, TAtomicSharedPtr<IChatRobot> chatRobot, const ui64 lastViewedMessageId)
        : Tag(tag)
        , ChatId(chatId)
        , Topic(topic)
        , ChatRobot(chatRobot)
        , LastViewedMessageId(lastViewedMessageId)
    {
    }
};

class TFilteredSupportChat {
public:
    struct TMessagesCount {
        ui32 Total = 0;
        ui32 Unread = 0;
    };

private:
    R_READONLY(TConstDBTag, Tag);
    R_READONLY(TChatMetricFilter::TMetricValueType, MetricValue, 0);
    R_CONST(NDrive::NChat::TMessageEvent, DisplayMessage);
    R_OPTIONAL(NDrive::NChat::TMessageEvent, LastMessage);
    R_READONLY(TString, Performer);
    R_READONLY(TMessagesCount, MessagesCount);

public:
    TFilteredSupportChat(const TConstDBTag& chatTag, const TChatMetricFilter::TMetricValueType& metricValue, const NDrive::NChat::TMessageEvent& displayMessage, const TMaybe<NDrive::NChat::TMessageEvent>& lastMessage, const TMessagesCount count, const TString& performer = "")
        : Tag(chatTag)
        , MetricValue(metricValue)
        , DisplayMessage(displayMessage)
        , LastMessage(lastMessage)
        , Performer(performer)
        , MessagesCount(count)
    {
    }

    NJson::TJsonValue BuildReport() const;
    TString GetYasmBucketName() const;
    TString GetOverallBucketName() const;
};

class TChatSupportRequestsFilter : public TBaseSupportRequestsFilter {
public:
    enum class EChatStatus: ui32 {
        Any = 0 /* "any" */,
        WaitingForSupport = 1 /* "waiting_for_support" */,
        WaitingForInitialAnswer = 2 /* "waiting_for_initial_answer" */,
    };

    enum class EMutedStatus: ui32 {
        Any = 0 /* "any" */,
        Muted = 1 /* "muted" */,
        NotMuted = 2 /* "not_muted" */,
    };

private:
    R_FIELD(TString, ChatRobotId);
    R_FIELD(TSet<TString>, NodeNames);
    R_FIELD(TSet<TString>, BannedNodeNames);
    R_FIELD(EChatStatus, ChatStatus, EChatStatus::Any);
    R_FIELD(EMutedStatus, MutedStatus, EMutedStatus::Any);
    R_FIELD(ui32, VisibleTraits, 0);
    R_FIELD(TChatMetricFilter, MetricFilter);
    R_FIELD(TVector<TChatMetricFilter>, AdditionalMetricFilters);
    R_OPTIONAL(TString, ChatTopic, {}, mutable);
    R_FIELD(TSet<TString>, OriginalSupportLines);
    R_FIELD(TDuration, MessageTTL, TDuration::Max());

    virtual TSet<TString> GetExcludedTagsByDefault() const override;
    virtual TString GetProcessedTagType() const override {
        return TSupportChatTag::TypeName;
    }

public:
    bool GetSupportRequests(const NDrive::IServer* server, TVector<TFilteredSupportChat>& result, const TInstant actuality, const TString& requestUserId, TMessagesCollector& errors) const;

    TString BuildHRReport(const NDrive::IServer* server, const TFilteredSupportChat& chat) const;

    static NDrive::TScheme GetScheme();
    NJson::TJsonValue SerializeToJson() const;
    bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo);

private:
    bool IsChatStatusMatching(const NDrive::NChat::TMessageEvents& messages, const TString& chatOwnerId) const;
    bool IsMatchingMetric(const NDrive::NChat::TMessageEvents& messages, const TConstDBTag& tag, const TChatMetricFilter& metricFilter, ui32& metricValue) const;
    bool IsMatchingMetricAllMessages(const NDrive::NChat::TMessageEvents& messages, const TConstDBTag& tag, const TChatMetricFilter& metricFilter, TChatMetricFilter::TMetricValueType& metricValue) const;
    void AddSplittedChatStats(const TConstDBTag& tag, const NDrive::NChat::TMessageEvents& messages, const TChatMetricFilter::TMetricValueType metricValue, TVector<TFilteredSupportChat>& result, const NDrive::IServer* server, const TMaybe<NDrive::NChat::TMessageEvent>& lastMessage, const TFilteredSupportChat::TMessagesCount count) const;
    void RemoveInsignificantMessages(NDrive::NChat::TMessageEvents& messages, const TConstDBTag& tag, const NDrive::IServer* server) const;
    NDrive::NChat::TMessageEvent GetDisplayMessage(const NDrive::NChat::TMessageEvents& messages) const;

    TString BuildUserReport(const NDrive::IServer* server, const TString& userId) const;
    TString GetLogin(const NDrive::IServer* server, const TString& userId) const;
};

class TCallSupportRequestsFilter : public TSimpleSupportRequestsFilter {
private:
    virtual TString GetProcessedTagType() const override {
        return TSupportPhoneCallTag::TypeName;
    }

    virtual TSet<TString> GetExcludedTagsByDefault() const override {
        return {};
    }
};

class TOutgoingSupportRequestsFilter : public TSimpleSupportRequestsFilter {
private:
    virtual TString GetProcessedTagType() const override {
        return TSupportOutgoingCommunicationTag::TypeName;
    }

    virtual TSet<TString> GetExcludedTagsByDefault() const override {
        return {};
    }
};

class TUserContainerSupportRequestsFilter: public TSimpleSupportRequestsFilter {
public:
    virtual TString GetProcessedTagType() const override {
        return TUserContainerTag::TypeName;
    }

    virtual TSet<TString> GetExcludedTagsByDefault() const override {
        return {};
    }
};

class TOutgoingFleetSupportRequestsFilter : public TSimpleSupportRequestsFilter {
public:
    virtual TString GetProcessedTagType() const override {
        return TFleetSupportOutgoingCommunicationTag::TypeName;
    }

    virtual TSet<TString> GetExcludedTagsByDefault() const override {
        return {};
    }
};
