#include "manager.h"

#include <drive/backend/abstract/settings.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/parse.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/util/algorithm/ptr.h>

#include <util/string/cast.h>

namespace {
    void AddSignal(const ESupportTelephonyIncomingEventType eventType, const ESupportTelephonySignalType signalType, const double value = 1) {
        TUnistatSignalsCache::SignalAdd("support_center-telephony", ::ToString(eventType) + "-" + ::ToString(signalType), value);
    }
}

bool ISupportTelephonyIncomingEvent::ProcessPerformerCallTags(const NDrive::IServer* server, NDrive::TEntitySession& tagsSession) const {
    if (!Performer) {
        return false;
    }

    TVector<TDBTag> currentPerformingCalls;
    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RestorePerformerTags(GetTelephonyTagNames(server), {Performer}, currentPerformingCalls, tagsSession)) {
        ERROR_LOG << "cannot restore user tags" << Endl;
        return false;
    }

    TVector<TDBTag> callTagsToRemove;
    for (auto&& tag : currentPerformingCalls) {
        if (tag.GetObjectId() != UserId) {
            callTagsToRemove.push_back(std::move(tag));
        }
    }

    if (callTagsToRemove) {
        return server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTags(callTagsToRemove, Performer, server, tagsSession, true);
    }

    return true;
}

TMaybe<TDBTag> ISupportTelephonyIncomingEvent::HandleActualUserCallTag(const NDrive::IServer* server, NDrive::TEntitySession& tagsSession, bool create) const {
    bool result;
    TMaybe<TDBTag> actualUserCallTag;
    {
        std::tie(result, actualUserCallTag) = ProcessUserCallTags(server, tagsSession);
        if (!result) {
            return {};
        }
    }
    if (create && !actualUserCallTag) {
        std::tie(result, actualUserCallTag) = AddUserCallTag(server, tagsSession);
        if (!result) {
            return {};
        }
    }
    return actualUserCallTag;
}

std::tuple<bool, TMaybe<TDBTag>> ISupportTelephonyIncomingEvent::ProcessUserCallTags(const NDrive::IServer* server, NDrive::TEntitySession& tagsSession) const {
    bool result;
    TMaybe<TDBTag> actualCallTag;
    TVector<TDBTag> otherCallTags;

    std::tie(result, actualCallTag, otherCallTags) = GetUserCallTags(server, tagsSession);
    if (!result) {
        return {false, {}};
    }

    if (otherCallTags) {
        result = server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTags(otherCallTags, GetUserId(), server, tagsSession, true);
    }

    return {result, std::move(actualCallTag)};
}

std::tuple<bool, TMaybe<TDBTag>> ISupportTelephonyIncomingEvent::AddUserCallTag(const NDrive::IServer* server, NDrive::TEntitySession& tagsSession) const {
    TAtomicSharedPtr<TSupportPhoneCallTag> tag = ConstructTag(server);
    if (!tag) {
        ERROR_LOG << "cannot construct tag" << Endl;
        return {false, {}};
    }

    const TDuration tagSLA = server->GetSettings().GetValueDef("support.cc.call_tags.default_sla", TDuration::Minutes(11));
    tag->SetSLAInstant(Now() + tagSLA);
    tag->SetInternalCallId(InternalCallId);

    auto addedTags = server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, UserId, UserId, server, tagsSession, EUniquePolicy::SkipIfExists);
    if (!addedTags || addedTags->size() != 1) {
        ERROR_LOG << "cannot add tag" << Endl;
        return {false, {}};
    }

    return {true, std::move(addedTags->front())};
}

TAtomicSharedPtr<TSupportPhoneCallTag> ISupportTelephonyIncomingEvent::ConstructTag(const NDrive::IServer* server) const {
    TAtomicSharedPtr<TSupportPhoneCallTag> tag = ConstructTagByQueue(server);

    if (!tag && server->GetSettings().GetValueDef("support.cc.call_tags.use_tag_name_by_cc_fallback", false)) {
        tag = ConstructTagByCallCenter(server);
    }

    if (!tag) {
        auto defaultTagName = server->GetSettings().GetValueDef<TString>("support.cc.call_tags.default_name", "cc_internal_incoming");
        tag = MakeAtomicShared<TSupportPhoneCallTag>(defaultTagName);
    }

    return tag;
}

TAtomicSharedPtr<TSupportPhoneCallTag> ISupportTelephonyIncomingEvent::ConstructTagByQueue(const NDrive::IServer* server) const {
    TAtomicSharedPtr<TSupportPhoneCallTag> tag;
    if (!Queue) {
        return tag;
    }

    auto tagDescriptions = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(TSupportPhoneCallTag::TypeName);
    for (auto&& td : tagDescriptions) {
        auto callTagDescription = std::dynamic_pointer_cast<const TSupportPhoneCallTag::TDescription>(td);
        if (callTagDescription && Queue == callTagDescription->GetCallQueue()) {
            if (!tag) {
                tag = MakeAtomicShared<TSupportPhoneCallTag>(callTagDescription->GetName());
            } else {
                ERROR_LOG << "multiple tags with same call center queue assigned: " << Queue << Endl;
                tag = nullptr;
                break;
            }
        }
    }

    return tag;
}

TAtomicSharedPtr<TSupportPhoneCallTag> ISupportTelephonyIncomingEvent::ConstructTagByCallCenter(const NDrive::IServer* /* server */) const {
    TAtomicSharedPtr<TSupportPhoneCallTag> tag;
    if (CCType == ECallCenter::Audiotele) {
        tag = MakeAtomicShared<TSupportPhoneCallTag>("cc_audiotele_incoming");
    } else if (CCType == ECallCenter::NextContact) {
        tag = MakeAtomicShared<TSupportPhoneCallTag>("cc_nextcontact_incoming");
    } else if (CCType == ECallCenter::YandexInternal) {
        tag = MakeAtomicShared<TSupportPhoneCallTag>("cc_internal_carsharing");
    } else {
        ERROR_LOG << "unknown callcenter: " << ToString(CCType) << Endl;
    }
    return tag;
}

TVector<TString> ISupportTelephonyIncomingEvent::GetTelephonyTagNames(const NDrive::IServer* server) const {
    TVector<TString> result;
    auto tagDescriptions = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(TSupportPhoneCallTag::TypeName);
    for (auto&& td : tagDescriptions) {
        result.emplace_back(td->GetName());
    }
    return result;
}

std::tuple<bool, TMaybe<TDBTag>, TVector<TDBTag>> ISupportTelephonyIncomingEvent::GetUserCallTags(const NDrive::IServer* server, NDrive::TEntitySession& tagsSession) const {
    TMaybe<TDBTag> actualCallTag;
    TVector<TDBTag> otherCallTags;

    TVector<TDBTag> currentCallTags;
    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(GetUserId(), GetTelephonyTagNames(server), currentCallTags, tagsSession)) {
        ERROR_LOG << "cannot restore user tags" << Endl;
        return {false, {}, {}};
    }

    for (auto&& tag : currentCallTags) {
        if (auto callTagPtr = tag.GetTagAs<TSupportPhoneCallTag>()) {
            if (callTagPtr->GetInternalCallId() != GetInternalCallId()) {
                otherCallTags.push_back(std::move(tag));
            } else {
                actualCallTag = std::move(tag);
            }
        }
    }

    return {true, std::move(actualCallTag), std::move(otherCallTags)};
}

bool ISupportTelephonyIncomingEvent::Construct(const TString& userId, const NJson::TJsonValue& json) {
    UserId = userId;
    return NJson::ParseField(json["cc"], NJson::Stringify(CCType)) &&
           NJson::ParseField(json["type"], NJson::Stringify(Type)) &&
           NJson::ParseField(json["performer"], Performer) &&
           NJson::ParseField(json["call_id"], InternalCallId) &&
           NJson::ParseField(json["queue"], Queue);
}

bool TSupportTelephonyEvent::Apply(const NDrive::IServer* server, NDrive::TEntitySession& tagsSession) const {
    switch (GetType()) {
    case EType::Start:
        return StartCall(server, tagsSession);
    case EType::SetPerformer:
    case EType::ForcePerformer:
        return SetCallPerformer(server, tagsSession);
    case EType::Finish:
        return FinishCall(server, tagsSession);
    case EType::DropPerformer:
        return DropCallPerformer(server, tagsSession);
    }
}

bool TSupportTelephonyEvent::StartCall(const NDrive::IServer* server, NDrive::TEntitySession& tagsSession) const {
    auto actualUserCallTag = HandleActualUserCallTag(server, tagsSession, /* create = */ true);
    if (!actualUserCallTag) {
        ERROR_LOG << "could not add call tag" << Endl;
        AddSignal(ESupportTelephonyIncomingEventType::Start, ESupportTelephonySignalType::UpsertCallTagError);
        return false;
    }
    return true;
}

bool TSupportTelephonyEvent::SetCallPerformer(const NDrive::IServer* server, NDrive::TEntitySession& tagsSession) const {
    if (!ProcessPerformerCallTags(server, tagsSession)) {
        ERROR_LOG << "could not process performer tags" << Endl;
        AddSignal(ESupportTelephonyIncomingEventType::SetPerformer, ESupportTelephonySignalType::OperatorTagsProcessingError);
        return false;
    }

    auto actualUserCallTag = HandleActualUserCallTag(server, tagsSession, /* create = */ true);
    if (!actualUserCallTag) {
        ERROR_LOG << "could get or create actual call tag" << Endl;
        AddSignal(ESupportTelephonyIncomingEventType::SetPerformer, ESupportTelephonySignalType::UpsertCallTagError);
        return false;
    }

    if ((*actualUserCallTag)->GetPerformer() == GetPerformer()) {
        return true;
    }

    TUserPermissions::TPtr performerPermissionsPtr = server->GetDriveAPI()->GetUserPermissions(GetPerformer(), TUserPermissionsFeatures());
    if (!performerPermissionsPtr) {
        ERROR_LOG << "could not restore permissions" << Endl;
        AddSignal(ESupportTelephonyIncomingEventType::SetPerformer, ESupportTelephonySignalType::RestorePermissionsError);
        return false;
    }

    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().InitPerformer({*actualUserCallTag}, *performerPermissionsPtr, server, tagsSession)) {
        ERROR_LOG << "could not init performer" << Endl;
        AddSignal(ESupportTelephonyIncomingEventType::SetPerformer, ESupportTelephonySignalType::TagPerformError);
        return false;
    }

    return true;
}

bool TSupportTelephonyEvent::DropCallPerformer(const NDrive::IServer* server, NDrive::TEntitySession& tagsSession) const {
    auto actualUserCallTag = HandleActualUserCallTag(server, tagsSession, /* create = */ false);

    if (!actualUserCallTag) {
        ERROR_LOG << "no tag to drop performer" << Endl;
        AddSignal(ESupportTelephonyIncomingEventType::DropPerformer, ESupportTelephonySignalType::AbsentTagError);
        return false;
    }

    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().DropTagsPerformer({*actualUserCallTag}, GetUserId(), tagsSession, true)) {
        ERROR_LOG << "could not drop performer" << Endl;
        AddSignal(ESupportTelephonyIncomingEventType::DropPerformer, ESupportTelephonySignalType::TagReleaseError);
        return false;
    }

    return true;
}

bool TSupportTelephonyEvent::FinishCall(const NDrive::IServer* server, NDrive::TEntitySession& tagsSession) const {
    auto actualUserCallTag = HandleActualUserCallTag(server, tagsSession, /* create = */ false);

    if (!actualUserCallTag) {
        ERROR_LOG << "no tag to finish" << Endl;
        AddSignal(ESupportTelephonyIncomingEventType::Finish, ESupportTelephonySignalType::AbsentTagError);
        return false;
    }

    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTag(*actualUserCallTag, GetUserId(), server, tagsSession, true)) {
        ERROR_LOG << "error removing tags" << Endl;
        AddSignal(ESupportTelephonyIncomingEventType::Finish, ESupportTelephonySignalType::RemoveTagError);
        return false;
    }

    return true;
}
