#include "client.h"

#include <drive/backend/logging/events.h>

#include <util/string/strip.h>

namespace NCallCenterYandex {
    void TResponseCallback::ProcessResponseData() {
        if (Result.Has("STATUSCODE")) {
            Code = Result["STATUSCODE"].GetInteger();
        }

        if (Result.Has("STATUSMSG")) {
            ErrorMessage = "Status message: " + Result["STATUSMSG"].GetString() + " ";
        }
        if (Result.Has("STATUSDESC")) {
            ErrorMessage += "Status description: " + Result["STATUSDESC"].GetString();
        }
        if (!Result.Has("DATA")) {
            ErrorMessage = "Incorrect json format: no DATA section presents";
        }
    }

    NNeh::THttpRequest& TCallCenterYandexClient::PrepareRequest(NNeh::THttpRequest& req, const bool useRobot /* = true */) const {
        if (useRobot) {
            req.SetUri("robot.php/");
        } else if(Config.GetUseDevEnv()) {
            req.SetUri("/dev/");
        }
        Config.Authorize(req);
        return req;
    }

    TString TCallCenterYandexClient::PrepareCallerNumber(const TString& number) const {
        return StripStringLeft(number, EqualsStripAdapter('+'));
    }

    bool TCallCenterYandexClient::GetStat(const TInstant& since, const TInstant& until, const TVector<TString>& queues, TStatResponse& result, TMessagesCollector& errors) const {
        TStatRequest request(since, until, queues);
        NJson::TJsonValue reply;
        return SendRequest(ECallCenterYandexOperationType::GetStat, PrepareRequest(request), reply, errors)
            && result.DeserializeFromJson(reply, errors);
    }

    bool TCallCenterYandexClient::GetAppDistributionInfo(const TString& applicationName, NJson::TJsonValue& result, TMessagesCollector& errors) const {
        TAppDistribRequest request(applicationName);
        return SendRequest(ECallCenterYandexOperationType::GetAppDistributionInfo, PrepareRequest(request), result, errors);
    }

    bool TCallCenterYandexClient::GetAppDistributionInfo(const TString& applicationName, TAppDistribResponse& result, TMessagesCollector& errors) const {
        NJson::TJsonValue reply;
        return GetAppDistributionInfo(applicationName, reply, errors)
            && result.DeserializeFromJson(reply, errors);
    }

    bool TCallCenterYandexClient::UpdateAppDistribution(const TAppDistribution& distribution, TMessagesCollector& errors) const {
        TAppDistribUpdateRequest request(distribution);
        NJson::TJsonValue reply;
        return SendRequest(ECallCenterYandexOperationType::UpdateAppDistribution, PrepareRequest(request), reply, errors);
    }

    bool TCallCenterYandexClient::UpdateAppLoadBalance(const TAppDistribution& distribution, TMessagesCollector& errors) const {
        TAppLoadBalanceUpdateRequest request(distribution);
        NJson::TJsonValue reply;
        return SendRequest(ECallCenterYandexOperationType::UpdateAppLoadBalance, PrepareRequest(request), reply, errors);
    }

    bool TCallCenterYandexClient::GetAppPlaybackInfo(const TString& applicationName, TAppPlayback& result, TMessagesCollector& errors) const {
        TAppPlaybackRequest request(applicationName);
        NJson::TJsonValue reply;
        return SendRequest(ECallCenterYandexOperationType::GetAppPlaybackInfo, PrepareRequest(request), reply, errors)
            && result.DeserializeFromJson(reply, errors);
    }

    bool TCallCenterYandexClient::UpdateAppPlayback(const TAppPlayback& appPlayback, TMessagesCollector& errors, NJson::TJsonValue* reply) const {
        TAppPlaybackUpdateRequest request(appPlayback);
        NJson::TJsonValue initResp;
        NJson::TJsonValue& resp = reply ? *reply : initResp;
        return SendRequest(ECallCenterYandexOperationType::UpdateAppPlayback, PrepareRequest(request, false), resp, errors);
    }

    bool TCallCenterYandexClient::GetCallerRoutingInfo(const TString& queueName, const TMaybe<TString>& number, TStandaloneSourceRoutingGetResponse& result, TMessagesCollector& errors) const {
        TStandaloneSourceRoutingGetRequest request(queueName, (number) ? PrepareCallerNumber(*number) : number);
        NJson::TJsonValue reply;
        return SendRequest(ECallCenterYandexOperationType::GetRountingInfo, PrepareRequest(request), reply, errors)
            && result.DeserializeFromJson(reply, errors);
    }

    bool TCallCenterYandexClient::EnableCallerQueueRouting(const TString& queueName, const TString& number, const TString& targetQueueName, TMessagesCollector& errors) const {
        TStandaloneRedirectToQueueRequest request(queueName, PrepareCallerNumber(number), targetQueueName);
        NJson::TJsonValue reply;
        return SendRequest(ECallCenterYandexOperationType::EnableQueueRouting, PrepareRequest(request), reply, errors);
    }

    bool TCallCenterYandexClient::EnableCallerNumberRouting(const TString& queueName, const TString& number, const TString& targetNumber, TMessagesCollector& errors) const {
        TStandaloneRedirectToNumberRequest request(queueName, PrepareCallerNumber(number), PrepareCallerNumber(targetNumber));
        NJson::TJsonValue reply;
        return SendRequest(ECallCenterYandexOperationType::EnableNumberRouting, PrepareRequest(request), reply, errors);
    }

    bool TCallCenterYandexClient::EnableCallerDrop(const TString& queueName, const TString& number, TMessagesCollector& errors) const {
        TStandaloneDropCallRequest request(queueName, PrepareCallerNumber(number));
        NJson::TJsonValue reply;
        return SendRequest(ECallCenterYandexOperationType::EnableDrop, PrepareRequest(request), reply, errors);
    }

    bool TCallCenterYandexClient::DisableCallerRouting(const TString& queueName, const TString& number, TMessagesCollector& errors) const {
        TStandaloneSourceRoutingRemoveRequest request(queueName, PrepareCallerNumber(number));
        NJson::TJsonValue reply;
        return SendRequest(ECallCenterYandexOperationType::DisableRouting, PrepareRequest(request), reply, errors);
    }

    bool TCallCenterYandexClient::GetAgentState(const TString& agentId, TAgentStateGetResponse& result, TMessagesCollector& errors) const {
        TAgentStateGetRequest request(agentId);
        NJson::TJsonValue reply;
        return SendRequest(ECallCenterYandexOperationType::GetAgentState, PrepareRequest(request), reply, errors)
            && result.DeserializeFromJson(reply, errors);
    }

    bool TCallCenterYandexClient::ConnectAgent(const TString& agentId, const TVector<TString>& queues, TMessagesCollector& errors, bool paused) const {
        TAgentConnectRequest request(agentId, queues, TAgentConnectRequest::DefaultPriority, paused);
        auto requestWithAuth = PrepareRequest(request);
        NJson::TJsonValue reply;
        auto result = SendRequest(ECallCenterYandexOperationType::ConnectAgent, requestWithAuth, reply, errors);
        NDrive::TEventLog::Log("CallCenterYandexClient", NJson::TMapBuilder
            ("action", "connect")
            ("agent_id", agentId)
            ("request", requestWithAuth.GetDebugRequest())
            ("reply", reply)
            ("errors", errors.GetReport())
        );
        return result;
    }

    bool TCallCenterYandexClient::DisconnectAgent(const TString& agentId, const TVector<TString>& queues, TMessagesCollector& errors) const {
        TAgentDisconnectRequest request(agentId, queues);
        auto requestWithAuth = PrepareRequest(request);
        NJson::TJsonValue reply;
        auto result = SendRequest(ECallCenterYandexOperationType::DisconnectAgent, requestWithAuth, reply, errors);
        NDrive::TEventLog::Log("CallCenterYandexClient", NJson::TMapBuilder
            ("action", "disconnect")
            ("agent_id", agentId)
            ("request", requestWithAuth.GetDebugRequest())
            ("reply", reply)
            ("errors", errors.GetReport())
        );
        return result;
    }

    bool TCallCenterYandexClient::ChangeStatus(const TString& agentId, const TVector<TString>& queues, TMessagesCollector& errors, const bool paused, const ui32 priority) const {
        TAgentChangeStatusRequest request(agentId, queues, paused, priority);
        auto requestWithAuth = PrepareRequest(request);
        NJson::TJsonValue reply;
        auto result = SendRequest(ECallCenterYandexOperationType::ChangeStatus, requestWithAuth, reply, errors);
        NDrive::TEventLog::Log("CallCenterYandexClient", NJson::TMapBuilder
            ("action", "change_status")
            ("agent_id", agentId)
            ("request", requestWithAuth.GetDebugRequest())
            ("reply", reply)
            ("errors", errors.GetReport())
        );
        return result;
    }

    TInternalCallCenterClient::TInternalCallCenterClient(const TInternalCallCenterConfig& config)
        : Config(config)
        , Client(MakeAtomicShared<NNeh::THttpClient>(Config.GetUri()))
    {
    }

    NNeh::THttpRequest TInternalCallCenterClient::MakeRawTrackRequest(const TString& content, const TString& mediaUri, const TString& callId) {
        auto uriPos = content.find(mediaUri);
        if (uriPos == TString::npos) {
            ythrow yexception() << "Fail to parse content by uri(" << mediaUri << ") with call_id(" << callId << "): " << content;
        }
        auto cgiEnd = content.find('\"', uriPos);
        if (cgiEnd == TString::npos) {
            ythrow yexception() << "Fail to parse content with call_id(" << callId << "): " << content;
        }
        auto requestStr = content.substr(uriPos, cgiEnd - uriPos);
        if (!requestStr) { // uri and cgi prams are expected
            ythrow yexception() << "Fail to find cgi params from content with call_id(" << callId << "): " << content;
        }
        NNeh::THttpRequest request;
        request.SetUri(requestStr);
        return request;
    }

    NThreading::TFuture<NNeh::THttpRequest> TInternalCallCenterClient::GetTrackRequest(const TString& callId) const {
        NNeh::THttpRequest request;
        request.SetUri(Config.GetCallRecUri());
        request.SetCgiData("callid=" + callId);
        return Client->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply([callId, mediaUri = Config.GetCallMediaUri()](const NThreading::TFuture<NUtil::THttpReply>& request) {
            const auto& report = request.GetValue();
            TUnistatSignalsCache::SignalAdd("cipt-api-data-codes", ToString(report.Code()), 1);
            if (!report.HasReply()) {
                ythrow yexception() << "No track data reply with call_id(" << callId << "): " << report.ErrorMessage();
            }
            if (!report.IsSuccessReply()) {
                ythrow yexception() << "Error getting data by call_id(" << callId << ")): " << report.GetDebugReply();
            }
            return NThreading::MakeFuture(MakeRawTrackRequest(report.Content(), mediaUri, callId));
        });
    }

    NThreading::TFuture<TString> TInternalCallCenterClient::LoadTrack(const TString& callId) const {
        const auto deadline = Now() + Config.GetRequestTimeout();
        return GetTrackRequest(callId).Apply([client = Client, deadline, callId](const NThreading::TFuture<NNeh::THttpRequest>& request) {
            return client->SendAsync(request.GetValue(), deadline).Apply([callId](const NThreading::TFuture<NUtil::THttpReply>& reply) {
                const auto& report = reply.GetValue();
                TUnistatSignalsCache::SignalAdd("cipt-api-media-codes", ToString(report.Code()), 1);
                if (!report.HasReply()) {
                    ythrow yexception() << "No media reply with call_id(" << callId << "): " << report.ErrorMessage();
                }
                if (!report.IsSuccessReply()) {
                    ythrow yexception() << "Error getting data by call_id(" << callId << "): " << report.GetDebugReply();
                }
                return NThreading::MakeFuture(report.Content());
            });
        });
    }

    TDuration TInternalCallCenterClient::GetRequestTimeout() const {
        return Config.GetRequestTimeout();
    }
};
