#include "config.h"

#include <drive/backend/abstract/base.h>

#include <drive/library/cpp/openssl/rsa.h>
#include <drive/library/cpp/scheme/scheme.h>

#include <library/cpp/json/json_value.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/merge.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/types/messages_collector.h>

#include <util/stream/file.h>
#include <util/string/builder.h>
#include <util/string/hex.h>
#include <util/string/join.h>
#include <util/string/split.h>
#include <util/system/hostname.h>

namespace NCallCenterYandex {
    const TString TCallCenterYandexQueueConfig::SettingPrefix = "support.cc.internal";
    const TString TCallCenterYandexClientConfig::DefaultUsername = "robot-carsharing";

    TMaybe<TCallCenterYandexQueueConfig> TCallCenterYandexQueueConfig::Construct(const NJson::TJsonValue& data) {
        TCallCenterYandexQueueConfig config;
        if (!config.DeserializeFromJson(data)) {
            return {};
        }
        return config;
    }

    NDrive::TScheme TCallCenterYandexQueueConfig::GetScheme(const IServerBase& /* server */) {
        NDrive::TScheme scheme;
        scheme.Add<TFSArray>("ordinary_queues").SetElement<TFSString>();
        scheme.Add<TFSArray>("extra_queues").SetElement<TFSString>();
        return scheme;
    }

    NJson::TJsonValue TCallCenterYandexQueueConfig::GetSettingDefaults() {
        if (!NDrive::HasServer()) {
            return {};
        }

        const auto& settings = NDrive::GetServer().GetSettings();
        const TString settingPrefix = SettingPrefix + ".queues";

        NJson::TJsonValue result;

        auto ordinaryQueues = settings.GetJsonValue(settingPrefix + ".ordinary_queues");
        if (ordinaryQueues.IsDefined()) {
            NJson::InsertField(result, "ordinary_queues", ordinaryQueues);
        }

        auto extraQueues = settings.GetJsonValue(settingPrefix + ".extra_queues");
        if (extraQueues.IsDefined()) {
            NJson::InsertField(result, "extra_queues", extraQueues);
        }

        return result;
    }

    bool TCallCenterYandexQueueConfig::DeserializeFromJson(const NJson::TJsonValue& data) {
        auto defaults = GetSettingDefaults();
        auto& fullData = NJson::MergeJson(data, defaults);
        return NJson::ParseField(fullData["ordinary_queues"], OrdinaryQueues, /* required = */ true) &&
               NJson::ParseField(fullData["extra_queues"], ExtraQueues);
    }

    NJson::TJsonValue TCallCenterYandexQueueConfig::SerializeToJson() const {
        return NJson::TMapBuilder
            ("ordinary_queues", NJson::ToJson(OrdinaryQueues))
            ("extra_queues", NJson::ToJson(ExtraQueues));
    }

    TSet<TString> TCallCenterYandexQueueConfig::GetAllQueues() const {
        TSet<TString> allQueues;
        allQueues.insert(OrdinaryQueues.begin(), OrdinaryQueues.end());
        allQueues.insert(ExtraQueues.begin(), ExtraQueues.end());
        return allQueues;
    }

    bool TCallCenterYandexQueueConfig::AreQueuesValid(const TSet<TString>& queues, TMessagesCollector& errors) const {
        bool hasUnknownQueues = false;

        auto allQueues = GetAllQueues();
        for (const auto& q: queues) {
            if (!allQueues.contains(q)) {
                errors.AddMessage(__LOCATION__, "Unknown queue: " + q);
                hasUnknownQueues = true;
            }
        }

        return !hasUnknownQueues;
    }

    void TCallCenterYandexClientConfig::Init(const TYandexConfig::Section* section, const TMap<TString, NSimpleMeta::TConfig>* requestPolicy) {
        TBase::Init(section, requestPolicy);

        Username = section->GetDirectives().Value<TString>("Username", Username);
        UseDevEnv = section->GetDirectives().Value<bool>("UseDevEnv", UseDevEnv);
        AssertCorrectConfig(!!Username, "No username provided");
    }

    void TCallCenterYandexClientConfig::ToString(IOutputStream& os) const {
        TBase::ToString(os);
        os << "Username: " << Username << Endl;
        os << "UseDevEnv: " << UseDevEnv << Endl;
    }

    void TCallCenterYandexClientConfig::Authorize(NNeh::THttpRequest& request) const {
        // auth reference:
        // - https://wiki.yandex-team.ru/noc/office/tel/auth/

        auto uri = request.GetUri();

        auto cgiData = request.GetCgiData();
        request.SetCgiData(
            TStringBuilder()
            << cgiData
            << ((cgiData) ? "&" : "")
            << "dauth=" << GetUsername()
            << "&dhost=" << GetDefaultHost()
            << "&dtime=" << TInstant::Now().Seconds()
        );

        TString digest = to_lower(HexEncode(NOpenssl::CalcSHA1(request.GetCgiData() + GetToken())));
        request.AddCgiData("&dsign=" + digest);
    }

    TString TCallCenterYandexClientConfig::GetDefaultHost() const {
        return GetFQDNHostName();
    }

    void TInternalCallCenterConfig::Init(const TYandexConfig::Section* section) {
        Uri = section->GetDirectives().Value("Uri", Uri);
        CallRecUri = section->GetDirectives().Value("CallRecUri", CallRecUri);
        CallMediaUri = section->GetDirectives().Value("CallMediaUri", CallMediaUri);
        RequestTimeout  = section->GetDirectives().Value("RequestTimeout", RequestTimeout);
    }

    void TInternalCallCenterConfig::ToString(IOutputStream& os) const {
        os << "Uri: " << Uri << Endl;
        os << "CallRecUri: " << CallRecUri << Endl;
        os << "CallMediaUri: " << CallMediaUri << Endl;
        os << "RequestTimeout: " << RequestTimeout << Endl;
    }
};
