#include "request.h"

#include "logger.h"

#include <library/cpp/json/json_reader.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/json_processing.h>
#include <rtline/util/algorithm/container.h>

#include <library/cpp/string_utils/quote/quote.h>

#include <util/generic/algorithm.h>
#include <util/string/builder.h>
#include <util/string/cast.h>
#include <util/string/join.h>

namespace {
    bool ParseInstantFromString(const NJson::TJsonValue& data, TInstant& result) {
        TString strValue;
        ui64 value;
        if (!NJson::ParseField(data, strValue) || !TryFromString(strValue, value)) {
            return false;
        }
        result = TInstant::Seconds(value);
        return true;
    }
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NCallCenterYandex::TAppPlayback::TActionPlayback::EActionType& result) {
    return NJson::TryFromJson(value, NJson::Stringify(result));
}

template <>
bool NJson::TryFromJson(const TJsonValue& json, TVector<NCallCenterYandex::TAppPlayback::TActionPlayback>& result) {
    if (!json.IsMap()) {
        return false;
    }
    TMessagesCollector errors;
    for (const auto& [key, val] : json.GetMap()) {
        NCallCenterYandex::TAppPlayback::TActionPlayback action;
        if (!action.DeserializeFromJson(val, errors)) {
            ERROR_LOG << errors.GetStringReport() << Endl;
            return false;
        }
        action.SetActionName(key);
        result.emplace_back(std::move(action));
    }
    return true;
}

namespace NCallCenterYandex {
    TStatRequest::TStatRequest(const TInstant& since, const TInstant& until, const TVector<TString>& queues)
        : NNeh::THttpRequest()
    {
        NJson::TJsonValue requestData;
        NJson::InsertField(requestData, "SECTION", TStringBuf{"QM"});
        NJson::InsertField(requestData, "TYPE", TStringBuf{"DBLIST_NOTI"});
        NJson::InsertField(requestData, "QUEUEDB", TStringBuf{"direct"});
        NJson::InsertField(requestData, "QUEUELIST", NJson::ToJson(queues));

        NJson::TJsonValue timeRangeData = NJson::TMapBuilder("START", since.Seconds())("END", until.Seconds());
        NJson::InsertField(requestData, "TIMERANGE", timeRangeData);

        TString cgiData = "/mod.cipt-call-center/api";
        cgiData += "&request=" + CGIEscapeRet(requestData.GetStringRobust());
        SetCgiData(cgiData);
    }

    bool TStatResponse::TStatEntry::DeserializeFromJson(const NJson::TJsonValue& data, TMessagesCollector& errors) {
        return
            NJson::ParseField(data["time_id"], NJson::Seconds(TimeId), /* required = */ true, errors) &&
            NJson::ParseField(data["queue"], Queue, /* required = */ true, errors) &&
            NJson::ParseField(data["call_id"], CallId, /* required = */ true, errors) &&
            NJson::ParseField(data["agent"], Agent, /* required = */ true, errors) &&
            NJson::ParseField(data["verb"], Verb, /* required = */ true, errors) &&
            NJson::ParseField(data["data2"], Data, /* required = */ true, errors);
    }

    bool TStatResponse::DeserializeFromJson(const NJson::TJsonValue& reply, TMessagesCollector& errors) {
        if (!reply["DATA"].IsMap()) {
            errors.AddMessage("DeserializeFromJson::DeserializeFromJson", "No data field");
            return false;
        }

        for (const auto& [unusedId, rawEntry]: reply["DATA"].GetMap()) {
            TStatEntry entry;
            if (!entry.DeserializeFromJson(rawEntry, errors)) {
                return false;
            }
            Entries.push_back(std::move(entry));
        }

        return true;
    }

    NJson::TJsonValue TAppDistribution::TAppAction::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, SpecificActionFieldName, Action);
        NJson::InsertField(result, "VALUE", Value);
        if (Parameters.IsDefined()) {
            result[SpecificParametersFieldName] = Parameters;
        }
        return result;
    }

    bool TAppDistribution::TAppAction::DeserializeFromJson(const NJson::TJsonValue& data, TMessagesCollector& errors) {
        /*
        Some examples:
        - agent redirect-- "ACTION": "DIAL", "VALUE": "GLOBAL\/7288"
        - external queue redirect [, params are used if multiple actions provided] -- "ACTION": "DIAL", "VALUE": "GLOBAL\/80078" [, "PARAM": { "TIMEOUT": 60 } ]
        - application playback -- "ACT": "APP", "VALUE": "APP_PLAYBACK", "PARAM": "CARSH_OVERFLOW"
        - queue redirect by name -- "ACTION": "QUEUE", "VALUE": "carsharing-test", "PARAMS": { "ENDPOINT": "QPROC-Q-V4", "TIMEOUT": 300, "ENDPOINTCOUNT": 2 }
        */

        if (data.Has("ACTION")) {
            if (!NJson::ParseField(data["ACTION"], Action, /* required = */ true, errors)) {
                return false;
            }
            SpecificActionFieldName = "ACTION";
        } else if (data.Has("ACT")) {
            if (!NJson::ParseField(data["ACT"], Action, /* required = */ true, errors)) {
                return false;
            }
            SpecificActionFieldName = "ACT";
        } else {
            return false;
        }

        if (!NJson::ParseField(data["VALUE"], Value, /* required = */ true, errors)) {
            return false;
        }

        if (data.Has("PARAMS")) {
            Parameters = data["PARAMS"];
            SpecificParametersFieldName = "PARAMS";
        } else if (data.Has("PARAM")) {
            Parameters = data["PARAM"];
            SpecificParametersFieldName = "PARAM";
        } else {
            // okay, no parameters allowed
        }

        return true;
    }

    bool TAppDistribution::TAppDistributionEntry::IsFallback() const {
        return Name == "_FALLBACK";
    }

    bool TAppDistribution::TAppDistributionEntry::IsOverflowAnnounce() const {
        return Name == "OVERFLOW_ANNONCE";
    }

    NJson::TJsonValue TAppDistribution::TAppDistributionEntry::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "NAME", Name);
        NJson::InsertNonNull(result, "VALUE", Value);

        auto& actionValues = result.InsertValue("ACTIONVALUE", NJson::JSON_MAP);
        for (auto&& [priority, actionValue]: PrioritizedActions) {
            actionValues[::ToString(priority)] = actionValue.SerializeToJson();
        }

        return result;
    }

    bool TAppDistribution::TAppDistributionEntry::DeserializeFromJson(const NJson::TJsonValue& data, TMessagesCollector& errors) {
        if (!NJson::ParseField(data["NAME"], Name, /* required = */ true, errors)) {
            return false;
        }
        if (!IsFallback() && !NJson::ParseField(data["VALUE"], Value, /* required = */ true, errors)) {
            return false;
        }
        for (auto&& [rawPriority, rawAction]: data["ACTIONVALUE"].GetMap()) {
            ui32 priority;
            TAppAction action;
            if (!NJson::TryFromJson(rawPriority, priority) || !action.DeserializeFromJson(rawAction, errors)) {
                return false;
            }
            PrioritizedActions.emplace(priority, std::move(action));
        }
        if (!PrioritizedActions) {
            errors.AddMessage("TAppDistributionEntry::DeserializeFromJson", "Non prioritized action");
            return false;
        }
        return true;
    }

    TString TAppDistribution::GetApplicationType() const {
        return "APP_DISTRIB";
    }

    TSet<TString> TAppDistribution::GetNames() const {
        TSet<TString> names;
        for (const auto& entry: Entries) {
            names.emplace(entry.GetName());
        }
        return names;
    }

    TMaybe<ui32> TAppDistribution::GetValue(const TString& name) const {
        auto it = Find(name);
        return (it != Entries.end()) ? it->OptionalValue() : Nothing();
    }

    bool TAppDistribution::SetValue(const TString& name, const TMaybe<ui32>& value) {
        auto it = Find(name);
        if (it != Entries.end()) {
            it->SetValue(value);
            return true;
        }
        return false;
    }

    bool TAppDistribution::Has(const TString& name) const {
        return Find(name) != Entries.end();
    }

    TAppDistribution::TDistributionEntries::iterator TAppDistribution::Find(const TString& name) {
        return FindIf(Entries, [&name](const auto& e){ return e.GetName() == name; });
    }

    TAppDistribution::TDistributionEntries::const_iterator TAppDistribution::Find(const TString& name) const {
        return FindIf(Entries, [&name](const auto& e){ return e.GetName() == name; });
    }

    bool TAppDistribution::Validate() const {
        return ValidateLoadBalance() && AllOf(Entries, [](const auto& entry){ return !entry.GetPrioritizedActions().empty(); });
    }

    bool TAppDistribution::ValidateLoadBalance() const {
        ui32 total = 0;
        for (auto&& entry: Entries) {
            total += entry.OptionalValue().GetOrElse(0);
        }
        return (total == 100);
    }

    NJson::TJsonValue TAppDistribution::SerializeToJson() const {
        if (!Validate()) {
            return {};
        }

        NJson::TJsonValue result;

        NJson::InsertField(result, "NAME", ApplicationName);
        NJson::InsertField(result, "TYPE", GetApplicationType());

        auto& content = result.InsertValue("CONTENT", NJson::JSON_MAP);
        for (const auto& entry: Entries) {
            content[entry.GetName()] = entry.SerializeToJson();
        }

        return result;
    }

    NJson::TJsonValue TAppDistribution::SerializeLoadBalanceToJson() const {
        if (!Validate()) {
            return {};
        }

        NJson::TJsonValue result(NJson::JSON_MAP);

        for (const auto& entry: Entries) {
            if (entry.HasValue()) {
                result[entry.GetName()] = ::ToString(entry.GetValueRef());
            }
        }

        return result;
    }

    bool TAppDistribution::DeserializeFromJson(const NJson::TJsonValue& reply, TMessagesCollector& errors) {
        TString applicationType;
        if (!NJson::ParseField(reply["NAME"], ApplicationName, /* required = */ true, errors) ||
            !NJson::ParseField(reply["TYPE"], applicationType, /* required = */ true, errors)
        ) {
            return false;
        }

        if (applicationType != GetApplicationType()) {
            errors.AddMessage("TAppDistribution::DeserializeFromJson", "Incorrect application type");
            return false;
        }

        if (!reply["CONTENT"].IsMap()) {
            errors.AddMessage("TAppDistribution::DeserializeFromJson", "Incorrect content field");
            return false;
        }

        for (auto&& [unusedName, rawEntry]: reply["CONTENT"].GetMap()) {
            TAppDistributionEntry entry;
            if (!entry.DeserializeFromJson(rawEntry, errors)) {
                return false;
            }
            Entries.push_back(std::move(entry));
        }

        return true;
    }

    bool TAppDistribution::DeserializeLoadBalanceFromJson(const NJson::TJsonValue& data) {
        if (!data.IsMap()) {
            return false;
        }
        for (auto&& [name, rawLoadValue]: data.GetMap()) {
            TAppDistributionEntry entry;
            entry.SetName(name).SetValue(NJson::TryFromJson<ui32>(rawLoadValue));
            Entries.push_back(std::move(entry));
        }
        return ValidateLoadBalance();
    }

    TAppDistribRequest::TAppDistribRequest(const TString& applicationName)
        : NNeh::THttpRequest()
    {
        AddCgiData("/mod.cipt-admin/api/routing/applications/get/" + applicationName + "/content");
    }

    bool TAppDistribResponse::DeserializeFromJson(const NJson::TJsonValue& reply, TMessagesCollector& errors) {
        const auto& data = reply["DATA"];
        return Distribution.DeserializeFromJson(data, errors);
    }

    TAppDistribUpdateRequest::TAppDistribUpdateRequest(const TAppDistribution& distribution)
        : NNeh::THttpRequest()
    {
        TString cgiData = TStringBuilder() << "/mod.cipt-admin/api/routing/applications/update/" << distribution.GetApplicationName() << "/content/";
        if (distribution.Validate()) {
            auto data = distribution.SerializeToJson();
            cgiData += "&_DATA=" + CGIEscapeRet(distribution.SerializeToJson().GetStringRobust());  // actually update nothing if invalid
        }
        SetCgiData(cgiData);
    }

    TAppLoadBalanceUpdateRequest::TAppLoadBalanceUpdateRequest(const TAppDistribution& distribution)
        : NNeh::THttpRequest()
    {
        TString cgiData = TStringBuilder() << "/mod.cipt-admin/api/routing/applications/update/" << distribution.GetApplicationName() << "/content/distrib/";
        if (distribution.ValidateLoadBalance()) {
            cgiData += "&_DATA=" + CGIEscapeRet(distribution.SerializeLoadBalanceToJson().GetStringRobust());  // actually update nothing if invalid
        }
        SetCgiData(cgiData);
    }

    NJson::TJsonValue TAppPlayback::GetReport() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "application_name", GetApplicationName());
        auto& actions = result.InsertValue("actions", NJson::JSON_ARRAY);
        for (const auto& action : Actions) {
            NJson::TJsonValue subResult;
            NJson::InsertField(subResult, "action_name", action.GetActionName());
            NJson::InsertField(subResult, "records", action.GetRecords());
            actions.AppendValue(subResult);
        }
        return result;
    }

    NJson::TJsonValue TAppPlayback::SerializeToJson() const {
        NJson::TJsonValue result;
        auto& content = result.InsertValue("CONTENT", NJson::JSON_MAP);
        for (const auto& action : Actions) {
            NJson::InsertField(content, action.GetActionName(), action.SerializeToJson());
        }
        return result;
    }

    NJson::TJsonValue TAppPlayback::TActionPlayback::SerializeToJson() const {
        NJson::TJsonValue result;
        auto& value = result.InsertValue("ACTIONVALUE", NJson::JSON_MAP);
        result["TYPE"] = ToString(Type);
        for (const auto& [key, val] : GetRecords()) {
            NJson::TJsonValue json = NJson::TMapBuilder("CONTENT", val)("SRC", "FILE");
            NJson::InsertField(value, key, json);
        }
        return result;
    }

    bool TAppPlayback::DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector& errors) {
        if (json.Has("DATA")) {
            return NJson::ParseField(json["DATA"]["NAME"], ApplicationName, /* required = */ true, errors)
                && NJson::ParseField(json["DATA"]["CONTENT"], Actions, /* required = */ true, errors);
        }
        return NJson::ParseField(json["NAME"], ApplicationName, /* required = */ false, errors)
            && NJson::ParseField(json["CONTENT"], Actions, /* required = */ true, errors);
    }

    bool TAppPlayback::TActionPlayback::DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector& errors) {
        if (json.Has("ACTIONVALUE")) {
            for (const auto& [key, val] : json["ACTIONVALUE"].GetMap()) {
                TString subRes;
                if (NJson::ParseField(val["CONTENT"], subRes, /* required = */ true, errors)) {
                    Records[key] = subRes;
                }
            }
        } else {
            if (json.IsArray() || json["RECORDS"].IsArray()) {
                auto& jsonArray = json.IsArray() ? json.GetArray() : json["RECORDS"].GetArray();
                for (ui32 i = 0; i < jsonArray.size(); ++i) {
                    TString subRes;
                    if (NJson::ParseField(jsonArray[i], subRes, /* required = */ true, errors)) {
                        Records[ToString(i)] = subRes;
                    }
                }
            }
        }
        return NJson::ParseField(json["TYPE"], Type, /* required = */ false, errors)
            && NJson::ParseField(json["NAME"], ActionName, /* required = */ false, errors);
    }

    TAppPlaybackRequest::TAppPlaybackRequest(const TString& applicationName)
        : NNeh::THttpRequest()
    {
        AddCgiData("/mod.cipt-admin/api/routing/applications/get/" + applicationName + "/content");
    }

    TAppPlaybackUpdateRequest::TAppPlaybackUpdateRequest(const TAppPlayback& app)
        : NNeh::THttpRequest()
    {
        SetCgiData(TStringBuilder() << "/mod.cipt-admin/API/ROUTING/APPLICATIONS/UPDATE/" << app.GetApplicationName() << "/CONTENT/SET");
        SetPostData(NJson::TMapBuilder("_DATA", app.SerializeToJson()));
        SetContentType("application/json");
    }

    TStandaloneSourceRoutingGetRequest::TStandaloneSourceRoutingGetRequest(const TString& queueName, const TMaybe<TString>& number)
        : NNeh::THttpRequest()
    {
        AddCgiData(
            TStringBuilder()
            << "/mod.cipt-call-center/API/CALLCENTER/SRCROUTING"
            << "/" << queueName
            << "/list"
            << ((number) ? ("/" + *number) : "")
            << "/"
        );
    }

    bool TStandaloneSourceRoutingGetResponse::TSourceRouting::DeserializeFromJson(const NJson::TJsonValue& data, TMessagesCollector& errors) {
        {
            ui32 actionId;
            if (!NJson::ParseField(data["ACTIONID"], actionId, /* required = */ true, errors)) {
                return false;
            }
            ActionId = static_cast<ERoutingAction>(actionId);
        }

        if (data.Has("EXPIREDATE") && !ParseInstantFromString(data["EXPIREDATE"], ExpireDate)) {
            errors.AddMessage("TSourceRouting::DeserializeFromJson", "Incorrect expiredate field");
            return false;
        }

        TString idStr;
        return
            NJson::ParseField(data["ID"], idStr, /* required = */ true, errors) && TryFromString(idStr, Id) &&
            NJson::ParseField(data["QUEUE"], QueueName, /* required = */ true, errors) &&
            NJson::ParseField(data["SRCNUMB"], SourceNumber, /* required = */ true, errors) &&
            NJson::ParseField(data["ACTIONDATA"], ActionData, /* required = */ false, errors) &&
            NJson::ParseField(data["STREAMSRC"], StreamSource, /* required = */ false, errors);
    }

    bool TStandaloneSourceRoutingGetResponse::DeserializeFromJson(const NJson::TJsonValue& reply, TMessagesCollector& errors) {
        if (!reply["DATA"].IsMap()) {
            errors.AddMessage("TStandaloneSourceRoutingGetResponse::DeserializeFromJson", "Incorrect data field");
            return false;
        }
        for (const auto& [_, sourceRoutingData] : reply["DATA"].GetMap()) {
            if (!sourceRoutingData.IsMap()) {
                continue;
            }
            TSourceRouting r;
            if (!r.DeserializeFromJson(sourceRoutingData, errors)) {
                return false;
            }
            SourceRoutings.push_back(std::move(r));
        }
        return true;
    }

    TStandaloneSourceRoutingUpdateRequest::TStandaloneSourceRoutingUpdateRequest(const TString& queueName, const TString& number, ERoutingAction actionId, const TString& actionData, const TMaybe<TInstant>& expireDate) {
        TString cgiData = TStringBuilder() << "/mod.cipt-call-center/API/CALLCENTER/SRCROUTING/" << queueName << "/add/";

        NJson::TJsonValue params;
        NJson::InsertField(params, "SRCNUMB", number);
        NJson::InsertField(params, "ACTIONID", ::ToString(static_cast<ui32>(actionId)));
        NJson::InsertNonNull(params, "ACTIONDATA", actionData);
        if (expireDate && *expireDate) {
            NJson::InsertField(params, "EXPIREDATE", ::ToString(expireDate->Seconds()));
        }
        cgiData += "&_DATA=" + CGIEscapeRet(params.GetStringRobust());

        SetCgiData(cgiData);
    }

    TStandaloneRedirectToQueueRequest::TStandaloneRedirectToQueueRequest(const TString& queueName, const TString& number, const TString& targetQueueName, const TMaybe<TInstant>& expireDate)
        : TStandaloneSourceRoutingUpdateRequest(queueName, number, ERoutingAction::REDIRECT_TO_QUEUE, targetQueueName, expireDate)
    {
    }

    TStandaloneRedirectToNumberRequest::TStandaloneRedirectToNumberRequest(const TString& queueName, const TString& number, const TString& targetNumber, const TMaybe<TInstant>& expireDate)
        : TStandaloneSourceRoutingUpdateRequest(queueName, number, ERoutingAction::REDIRECT_TO_NUMBER, targetNumber, expireDate)
    {
    }

    TStandaloneDropCallRequest::TStandaloneDropCallRequest(const TString& queueName, const TString& number, const TMaybe<TInstant>& expireDate)
        : TStandaloneSourceRoutingUpdateRequest(queueName, number, ERoutingAction::DROP_CALL, queueName, expireDate)
    {
    }

    TStandaloneSourceRoutingRemoveRequest::TStandaloneSourceRoutingRemoveRequest(const TString& queueName, const TString& number) {
        AddCgiData(
            TStringBuilder()
            << "/mod.cipt-call-center/API/CALLCENTER/SRCROUTING"
            << "/" << queueName
            << "/delete/"
            << number
            << "/"
        );
    }

    TFurtherSourceRoutingGetRequest::TFurtherSourceRoutingGetRequest(const TString& applicationName, const TString& number, const TMaybe<ui32>& priority)
        : NNeh::THttpRequest()
    {
        AddCgiData(
            TStringBuilder()
            << "/mod.cipt-admin/api/routing/applications/get"
            << "/" << applicationName
            << "/content"
            << "/" << number
            << ((!priority.Empty()) ? "/" + ::ToString(priority.GetRef()) : "")
            << "/"
        );
    }

    bool TFurtherSourceRoutingGetResponse::TSourceRouting::DeserializeFromJson(const NJson::TJsonValue& data) {
        {
            ui32 actionId;
            if (!NJson::ParseField(data["ACTIONID"], actionId, /* required = */ true)) {
                return false;
            }
            ActionId = static_cast<ERoutingAction>(actionId);
        }

        if (data.Has("CHANGEDATE") && !ParseInstantFromString(data["CHANGEDATE"], ChangeDate)) {
            return false;
        }
        if (data.Has("EXPIREDATE") && !ParseInstantFromString(data["EXPIREDATE"], ExpireDate)) {
            return false;
        }

        TString idStr, priorityStr;
        return
            NJson::ParseField(data["ID"], idStr, /* required = */ true) && TryFromString(idStr, Id) &&
            NJson::ParseField(data["KEYNAME"], KeyName, /* required = */ true) &&
            NJson::ParseField(data["SRCNUMB"], SourceNumber, /* required = */ true) &&
            NJson::ParseField(data["PRIOR"], priorityStr, /* required = */ true) && TryFromString(priorityStr, Priority) &&
            NJson::ParseField(data["ACTIONDATA"], ActionData);
    }

    bool TFurtherSourceRoutingGetResponse::DeserializeFromJson(const NJson::TJsonValue& reply) {
        const auto& data = reply["DATA"];
        if (!data.Has("CONTENT") || !data["CONTENT"].IsMap()) {
            return false;
        }
        for (const auto& [_, sourceRoutingData] : data["CONTENT"].GetMap()) {
            TSourceRouting r;
            if (!r.DeserializeFromJson(sourceRoutingData)) {
                return false;
            }
            SourceRoutings.push_back(std::move(r));
        }
        return true;
    }

    TFurtherSourceRoutingUpdateRequest::TFurtherSourceRoutingUpdateRequest(const TString& applicationName, const TString& number, ERoutingAction actionId, ui32 priority, const TString& actionData, const TMaybe<TInstant>& expireDate)
        : NNeh::THttpRequest()
    {
        AddCgiData(
            TStringBuilder()
            << "/mod.cipt-admin/api/routing/applications/update"
            << "/" << applicationName
            << "/content/add"
            << "/" << number
            << "/" << priority
            << "/" << static_cast<ui32>(actionId)
            << ((actionData) ? "/" + actionData : "")
            << ((!expireDate.Empty()) ? "/" + ::ToString(expireDate.GetRef()) : "")
            << "/"
        );
    }

    TFurtherRedirectToQueueRequest::TFurtherRedirectToQueueRequest(const TString& applicationName, const TString& number, const TString& queueName, ui32 priority, const TMaybe<TInstant>& expireDate)
        : TFurtherSourceRoutingUpdateRequest(applicationName, number, ERoutingAction::REDIRECT_TO_QUEUE, priority, queueName, expireDate)
    {
    }

    TFurtherRedirectToNumberRequest::TFurtherRedirectToNumberRequest(const TString& applicationName, const TString& number, const TString& targetNumber, ui32 priority, const TMaybe<TInstant>& expireDate)
        : TFurtherSourceRoutingUpdateRequest(applicationName, number, ERoutingAction::REDIRECT_TO_NUMBER, priority, targetNumber, expireDate)
    {
    }

    TFurtherDropCallRequest::TFurtherDropCallRequest(const TString& applicationName, const TString& number, ui32 priority)
        : TFurtherSourceRoutingUpdateRequest(applicationName, number, ERoutingAction::DROP_CALL, priority)
    {
    }

    TFurtherSourceRoutingRemoveRequest::TFurtherSourceRoutingRemoveRequest(const TString& applicationName, const TString& number, ui32 priority)
        : NNeh::THttpRequest()
    {
        AddCgiData(
            TStringBuilder()
            << "/mod.cipt-admin/api/routing/applications/update"
            << "/" << applicationName
            << "/content/delete"
            << "/" << number
            << "/" << priority
            << "/"
        );
    }

    const int TAgentStateGetRequest::CC_ID = 2;

    TAgentStateGetRequest::TAgentStateGetRequest(const TString& agentId)
        : NNeh::THttpRequest()
    {
        AddCgiData(
            TStringBuilder()
            << "/mod.cipt-call-center/api/callcenter/agent"
            << "/" << CC_ID
            << "/" << agentId
            << "/show/"
        );
    }

    bool TAgentStateGetResponse::TAgentState::DeserializeFromJson(const NJson::TJsonValue& data, TMessagesCollector& errors) {
        return
            NJson::ParseField(data["PRIOR"], Priority, /* required = */ true, errors) &&
            NJson::ParseField(data["STATUS"], Status, /* required = */ true, errors) &&
            NJson::ParseField(data["CURPRIOR"], CurrentPriority, /* required = */ true, errors) &&
            NJson::ParseField(data["VENDUNIQUEID"], UniqueId, /* required = */ true, errors) &&
            NJson::ParseField(data["VENDMEMBNAME"], MemberName, /* required = */ true, errors) &&
            NJson::ParseField(data["VENDINTERFACE"], Interface, /* required = */ true, errors) &&
            NJson::ParseField(data["VENDPAUSED"], Paused, /* required = */ true, errors);
    }

    bool TAgentStateGetResponse::DeserializeFromJson(const NJson::TJsonValue& reply, TMessagesCollector& errors) {
        const auto& data = reply["DATA"];

        TString memberName;
        ui32 callCenterId;

        if (!NJson::ParseField(data["MEMBNAME"], memberName, /* required = */ true, errors) ||
            !NJson::ParseField(data["CALLCENTERID"], callCenterId, /* required = */ true, errors)
        ) {
            return false;
        }

        for (const auto& [queue, queueData] : data["QUEUES"].GetMap()) {
            TAgentState s;
            s.SetMemberName(memberName).SetCallCenterId(callCenterId).SetQueue(queue);
            if (!s.DeserializeFromJson(queueData, errors)) {
                return false;
            }
            AgentStates.push_back(std::move(s));
        }

        return true;
    }

    const ui32 TAgentConnectRequest::DefaultPriority = 1;

    TAgentConnectRequest::TAgentConnectRequest(const TString& agentId, const TVector<TString>& queues, ui32 priority, bool paused)
        : NNeh::THttpRequest()
    {
        AddCgiData(
            TStringBuilder()
            << "/mod.cipt-call-center/api/callcenter/agent"
            << "/" << TAgentStateGetRequest::CC_ID
            << "/" << agentId
            << "/connect"
            << "/" << JoinSeq(",", queues)
            << "/" << priority
            << "/" << int(paused)
        );
    }

    TAgentDisconnectRequest::TAgentDisconnectRequest(const TString& agentId, const TVector<TString>& queues)
        : NNeh::THttpRequest()
    {
        AddCgiData(
            TStringBuilder()
            << "/mod.cipt-call-center/api/callcenter/agent"
            << "/" << TAgentStateGetRequest::CC_ID
            << "/" << agentId
            << "/disconnect"
            << "/" << JoinSeq(",", queues)
        );
    }

    const ui32 TAgentChangeStatusRequest::DefaultPriority = 1;

    TAgentChangeStatusRequest::TAgentChangeStatusRequest(const TString& agentId, const TVector<TString>& queues, bool paused, ui32 priority)
        : NNeh::THttpRequest()
    {
        AddCgiData(
            TStringBuilder()
            << "/mod.cipt-call-center/api/callcenter/agent"
            << "/" << TAgentStateGetRequest::CC_ID
            << "/" << agentId
            << "/changestatus"
            << "/" << JoinSeq(",", queues)
            << "/" << priority
            << "/" << int(paused)
        );
    }
}
