#pragma once

#include "config.h"

#include <library/cpp/json/writer/json_value.h>

#include <rtline/util/network/neh_request.h>
#include <rtline/util/types/accessor.h>

#include <util/generic/map.h>
#include <util/generic/vector.h>

namespace NCallCenterYandex {
    // GetStat

    class TStatRequest : public NNeh::THttpRequest {
    public:
        TStatRequest(const TInstant& since, const TInstant& until, const TVector<TString>& queues);
    };

    class TStatResponse {
    public:
        // format is described here: https://wiki.yandex-team.ru/noc/office/tel/api/callcenterqevents/
        class TStatEntry {
        public:
            bool DeserializeFromJson(const NJson::TJsonValue& data, TMessagesCollector& errors);

        private:
            R_FIELD(TInstant, TimeId);
            R_FIELD(TString, Queue);
            R_FIELD(TString, CallId);
            R_FIELD(TString, Agent);
            R_FIELD(TString, Verb);
            R_FIELD(TString, Data);
        };

    public:
        bool DeserializeFromJson(const NJson::TJsonValue& reply, TMessagesCollector& errors);

    private:
        R_READONLY(TVector<TStatEntry>, Entries);
    };

    // GetAppDistribution, UpdateAppDistribution

    // Routing reference:
    // - https://wiki.yandex-team.ru/noc/office/tel/api/routing/
    // - https://wiki.yandex-team.ru/noc/office/tel/api/routing/apps/

    // Internal numbers reference (used in app distribution and routing):
    // - https://wiki.yandex-team.ru/noc/office/iptel/enumb/
    // - https://wiki.yandex-team.ru/noc/office/callcenter/numplan/ya/

    class TAppDistribution {
    public:
        class TAppAction {
        public:
            NJson::TJsonValue SerializeToJson() const;
            bool DeserializeFromJson(const NJson::TJsonValue& data, TMessagesCollector& errors);

        private:
            R_FIELD(TString, Action);
            R_FIELD(TString, SpecificActionFieldName);

            R_FIELD(TString, Value);

            R_FIELD(NJson::TJsonValue, Parameters);
            R_FIELD(TString, SpecificParametersFieldName);
        };

        class TAppDistributionEntry {
        public:
            using TPrioritizedActions = TMap<ui32, TAppAction>;

            bool IsFallback() const;
            bool IsOverflowAnnounce() const;

            NJson::TJsonValue SerializeToJson() const;
            bool DeserializeFromJson(const NJson::TJsonValue& data, TMessagesCollector& errors);

        private:
            R_FIELD(TString, Name);
            R_OPTIONAL(ui32, Value);
            R_FIELD(TPrioritizedActions, PrioritizedActions);
        };

    public:
        using TDistributionEntries = TVector<TAppDistributionEntry>;

        TString GetApplicationType() const;

        TSet<TString> GetNames() const;
        TMaybe<ui32> GetValue(const TString& name) const;
        bool SetValue(const TString& name, const TMaybe<ui32>& value);

        bool Has(const TString& name) const;

        TDistributionEntries::iterator Find(const TString& name);
        TDistributionEntries::const_iterator Find(const TString& name) const;

        bool Validate() const;
        bool ValidateLoadBalance() const;

        NJson::TJsonValue SerializeToJson() const;
        NJson::TJsonValue SerializeLoadBalanceToJson() const;

        bool DeserializeFromJson(const NJson::TJsonValue& reply, TMessagesCollector& errors);
        bool DeserializeLoadBalanceFromJson(const NJson::TJsonValue& data);

    private:
        R_FIELD(TString, ApplicationName);
        R_FIELD(TDistributionEntries, Entries);
    };

    class TAppDistribRequest : public NNeh::THttpRequest {
    public:
        TAppDistribRequest(const TString& applicationName);
    };

    class TAppDistribResponse {
    public:
        bool DeserializeFromJson(const NJson::TJsonValue& reply, TMessagesCollector& errors);

    private:
        R_FIELD(TAppDistribution, Distribution);
    };

    class TAppDistribUpdateRequest : public NNeh::THttpRequest {
    public:
        TAppDistribUpdateRequest(const TAppDistribution& distribution);
    };

    class TAppLoadBalanceUpdateRequest : public NNeh::THttpRequest {
    // update load balancing values only
    public:
        TAppLoadBalanceUpdateRequest(const TAppDistribution& distribution);
    };

    // Drive IVRs: IPTEL-543; https://wiki.yandex-team.ru/users/kuzmiak/IVR-Drive/

    class TAppPlayback {
    public:
        class TActionPlayback {
        public:
            enum class EActionType {
                Random /* "RANDOM" */,
                TopDown /* "TOPDOWN" */,
            };

        private:
            R_FIELD(EActionType, Type, EActionType::Random);
            R_FIELD(TString, ActionName);
            using TRecordsMap = TMap<TString, TString>;
            R_FIELD(TRecordsMap, Records);

        public:
            bool DeserializeActionsFromSettingJson(const NJson::TJsonValue& json);
            bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector& errors);
            NJson::TJsonValue SerializeToJson() const;
        };

    private:
        R_FIELD(TString, ApplicationName);
        R_FIELD(TVector<TActionPlayback>, Actions);

    public:
        TAppPlayback() = default;
        TAppPlayback(const TString& name)
            : ApplicationName(name)
        {
        }

        NJson::TJsonValue GetReport() const;
        NJson::TJsonValue SerializeToJson() const;
        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector& errors);
    };

    class TAppPlaybackRequest : public NNeh::THttpRequest {
    public:
        TAppPlaybackRequest(const TString& applicationName);
    };

    class TAppPlaybackUpdateRequest : public NNeh::THttpRequest {
    public:
        TAppPlaybackUpdateRequest(const TAppPlayback& app);
    };

    // GetRountingInfo, EnableQueueRouting, EnableNumberRouting, EnableDrop, DisableRouting

    // Source routing reference:
    // - currently used: https://wiki.yandex-team.ru/noc/office/tel/api/#sourceroutingdljakollcentra
    // - further implementation (still pending): https://wiki.yandex-team.ru/noc/office/tel/api/routing/#prilozhenieappsrcrouting

    // Routing implementations differs in a way that
    //   currently used one is a stand-alone detached solution as while as
    //   the further one is build on routing apps engine (still not finally implemented)

    enum class ERoutingAction : ui32 {
        IDLE = 0,
        OLD_REDIRECT_TO_QUEUE = 1,
        REDIRECT_TO_QUEUE = 10,
        REDIRECT_TO_NUMBER = 100,
        DROP_CALL = 400,
        SET_PRIORITY = 501,  /* (0 by default, max - 15) */
    };

    class TStandaloneSourceRoutingGetRequest : public NNeh::THttpRequest {
    public:
        TStandaloneSourceRoutingGetRequest(const TString& queueName, const TMaybe<TString>& number);
    };

    class TStandaloneSourceRoutingGetResponse {
    public:
        class TSourceRouting {
            R_FIELD(ui32, Id, 0);
            R_FIELD(TString, QueueName);
            R_FIELD(TString, SourceNumber);
            R_FIELD(ERoutingAction, ActionId, ERoutingAction::IDLE);
            R_FIELD(TString, ActionData);
            R_FIELD(TString, StreamSource);
            R_FIELD(TInstant, ExpireDate);

        public:
            bool DeserializeFromJson(const NJson::TJsonValue& data, TMessagesCollector& errors);
        };

        using TSourceRoutings = TVector<TSourceRouting>;

        R_READONLY(TSourceRoutings, SourceRoutings);

    public:
        bool DeserializeFromJson(const NJson::TJsonValue& reply, TMessagesCollector& errors);
    };

    class TStandaloneSourceRoutingUpdateRequest : public NNeh::THttpRequest {
    public:
        TStandaloneSourceRoutingUpdateRequest(const TString& queueName, const TString& number, ERoutingAction actionId, const TString& actionData = "", const TMaybe<TInstant>& expireDate = {});
    };

    class TStandaloneRedirectToQueueRequest : public TStandaloneSourceRoutingUpdateRequest {
    public:
        TStandaloneRedirectToQueueRequest(const TString& queueName, const TString& number, const TString& targetQueueName, const TMaybe<TInstant>& expireDate = {});
    };

    class TStandaloneRedirectToNumberRequest : public TStandaloneSourceRoutingUpdateRequest {
    public:
        TStandaloneRedirectToNumberRequest(const TString& queueName, const TString& number, const TString& targetNumber, const TMaybe<TInstant>& expireDate = {});
    };

    class TStandaloneDropCallRequest : public TStandaloneSourceRoutingUpdateRequest {
    public:
        TStandaloneDropCallRequest(const TString& queueName, const TString& number, const TMaybe<TInstant>& expireDate = {});
    };

    class TStandaloneSourceRoutingRemoveRequest : public NNeh::THttpRequest {
    public:
        TStandaloneSourceRoutingRemoveRequest(const TString& queueName, const TString& number);
    };

    // Further routing implementations (not invoked from client as they're not finally implemented)

    class TFurtherSourceRoutingGetRequest : public NNeh::THttpRequest {
    public:
        TFurtherSourceRoutingGetRequest(const TString& applicationName, const TString& number, const TMaybe<ui32>& priority = {});
    };

    class TFurtherSourceRoutingGetResponse {
    public:
        class TSourceRouting {
            R_READONLY(ui32, Id, 0);
            R_READONLY(TString, KeyName);
            R_READONLY(TString, SourceNumber);
            R_READONLY(ui32, Priority, 1);
            R_READONLY(ERoutingAction, ActionId, ERoutingAction::IDLE);
            R_READONLY(TString, ActionData);
            R_READONLY(TInstant, ChangeDate);
            R_READONLY(TInstant, ExpireDate);

        public:
            bool DeserializeFromJson(const NJson::TJsonValue& data);
        };

        using TSourceRoutings = TVector<TSourceRouting>;

        R_READONLY(TSourceRoutings, SourceRoutings);

    public:
        bool DeserializeFromJson(const NJson::TJsonValue& reply);
    };

    class TFurtherSourceRoutingUpdateRequest : public NNeh::THttpRequest {
    public:
        TFurtherSourceRoutingUpdateRequest(const TString& applicationName, const TString& number, ERoutingAction actionId, ui32 priority = 0, const TString& actionData = "", const TMaybe<TInstant>& expireDate = {});
    };

    class TFurtherRedirectToQueueRequest : public TFurtherSourceRoutingUpdateRequest {
    public:
        TFurtherRedirectToQueueRequest(const TString& applicationName, const TString& number, const TString& queueName, ui32 priority = 0, const TMaybe<TInstant>& expireDate = {});
    };

    class TFurtherRedirectToNumberRequest : public TFurtherSourceRoutingUpdateRequest {
    public:
        TFurtherRedirectToNumberRequest(const TString& applicationName, const TString& number, const TString& targetNumber, ui32 priority = 0, const TMaybe<TInstant>& expireDate = {});
    };

    class TFurtherDropCallRequest : public TFurtherSourceRoutingUpdateRequest {
    public:
        TFurtherDropCallRequest(const TString& applicationName, const TString& number, ui32 priority = 0);
    };

    class TFurtherSourceRoutingRemoveRequest : public NNeh::THttpRequest {
    public:
        TFurtherSourceRoutingRemoveRequest(const TString& applicationName, const TString& number, ui32 priority = 0);
    };

    // GetAgentState, ConnectAgent, DisconnectAgent
    // Refer to docs here: https://wiki.yandex-team.ru/noc/office/tel/api/cc/

    // More docs related to agent handling: https://wiki.yandex-team.ru/noc/office/tel/api/dev/mvp/callcenter/

    class TAgentStateGetRequest : public NNeh::THttpRequest {
    public:
        static const int CC_ID;

        TAgentStateGetRequest(const TString& agentId);
    };

    class TAgentStateGetResponse {
    public:
        class TAgentState {
            // common fields
            R_FIELD(ui32, CallCenterId, 0);
            R_FIELD(TString, Queue);

            // queue specific name
            R_READONLY(ui32, Priority, 0);
            R_READONLY(ui32, Status, 0);
            R_READONLY(TString, CurrentPriority);

            // vendor specific fields
            R_FIELD(TString, MemberName);

            R_READONLY(TString, UniqueId);
            R_READONLY(TString, Interface);
            R_READONLY(TString, Paused);

        public:
            bool DeserializeFromJson(const NJson::TJsonValue& data, TMessagesCollector& errors);
        };

        using TAgentStates = TVector<TAgentState>;

        R_READONLY(TAgentStates, AgentStates);

    public:
        bool DeserializeFromJson(const NJson::TJsonValue& reply, TMessagesCollector& errors);
    };

    class TAgentConnectRequest : public NNeh::THttpRequest {
    public:
        static const ui32 DefaultPriority;

        TAgentConnectRequest(const TString& agentId, const TVector<TString>& queues, ui32 priority = DefaultPriority, bool paused = false);
    };

    class TAgentDisconnectRequest : public NNeh::THttpRequest {
    public:
        TAgentDisconnectRequest(const TString& agentId, const TVector<TString>& queues);
    };

    class TAgentChangeStatusRequest : public NNeh::THttpRequest {
    public:
        static const ui32 DefaultPriority;

        TAgentChangeStatusRequest(const TString& agentId, const TVector<TString>& queues, bool paused, ui32 priority);
    };
};
