#include <drive/backend/support_center/yandex/client.h>

#include <drive/library/cpp/threading/future.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>

#include <library/cpp/testing/unittest/tests_data.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/json/json_reader.h>

#include <util/system/env.h>
#include <util/string/builder.h>

Y_UNIT_TEST_SUITE(CallCenterYandexClientSuite) {
    NCallCenterYandex::TCallCenterYandexClient GetClient() {
        auto config = TStringBuilder()
            << "Host: tel.yandex-team.ru" << Endl
            << "Port: 443" << Endl
            << "IsHttps: true" << Endl
            << "UseDevEnv: true" << Endl
            << "Token: " << GetEnv("Token") << Endl
            << "RequestTimeout: 10s" << Endl;
        INFO_LOG << config << Endl;
        return NCallCenterYandex::TCallCenterYandexClient(NCallCenterYandex::TCallCenterYandexClientConfig::ParseFromString(config));
    }

    Y_UNIT_TEST(PlyabackAppSimple) {
        auto client = GetClient();
        const TString appName = GetEnv("APP", "TEST_CARSH_PLAY");
        TMessagesCollector errors;
        NCallCenterYandex::TAppPlayback app;
        {
            const TString records = R"(
                {
                    "rental_end_problem": {
                        "TYPE": "TOPDOWN",
                        "RECORDS": [ "rental_end_problem" ]
                    },
                    "payments_problem": [ "payments_problem_m", "payments_problem_f" ],
                    "empty_ivr" : []
                }
            )";
            NJson::TJsonValue json;
            UNIT_ASSERT(ReadJsonTree(records, &json));
            UNIT_ASSERT_C(app.DeserializeFromJson(NJson::TMapBuilder("NAME", appName)("CONTENT", json), errors), errors.GetStringReport());
        }
        {
            INFO_LOG << app.GetReport() << Endl;
            INFO_LOG << app.SerializeToJson() << Endl;
            NJson::TJsonValue result;
            UNIT_ASSERT_C(client.UpdateAppPlayback(app, errors, &result), TStringBuilder() << result << Endl <<  errors.GetStringReport());
        }
        {
            NCallCenterYandex::TAppPlayback result;
            UNIT_ASSERT_C(client.GetAppPlaybackInfo(appName, result, errors), errors.GetStringReport());
            INFO_LOG << result.GetReport() << Endl;
        }
    }

    Y_UNIT_TEST(LoadCallRecord) {
        const TString callId = GetEnv("TestCallId");
        UNIT_ASSERT(callId);
        const NCallCenterYandex::TInternalCallCenterConfig config;
        const NCallCenterYandex::TInternalCallCenterClient client(config);
        {
            auto future = client.GetTrackRequest(callId);
            UNIT_ASSERT(future.Wait(config.GetRequestTimeout()));
            UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
            auto request = future.GetValue();
            UNIT_ASSERT_VALUES_EQUAL(request.GetUri(), config.GetCallMediaUri());
            auto params = StringSplitter(request.GetCgiData()).SplitBySet("&").SkipEmpty().ToList<TString>();
            UNIT_ASSERT_VALUES_EQUAL(params.size(), 2);
            UNIT_ASSERT(params.front().size() > callId.size());
            UNIT_ASSERT_VALUES_EQUAL(params.front().substr(0, 6), "stream");
            UNIT_ASSERT(params.back().size() > callId.size());
            UNIT_ASSERT_VALUES_EQUAL(params.back().substr(0, 6), "stream");
        }
        {
            auto future = client.LoadTrack(callId);
            UNIT_ASSERT(future.Wait(config.GetRequestTimeout()));
            UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
            UNIT_ASSERT(future.GetValue());
        }
    }
}
