#include "surge_snapshot.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/database/config.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/offers/actions/abstract.h>
#include <drive/backend/offers/actions/fix_point.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/proto/offer.pb.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags_filter.h>
#include <drive/backend/users_controller/ucontroller.h>

#include <library/cpp/histogram/rt/histogram.h>
#include <library/cpp/json/json_reader.h>

#include <drive/library/cpp/maps_router/router.h>
#include <rtline/api/graph/router/router.h>
#include <rtline/library/unistat/signals.h>
#include <rtline/protos/proto_helper.h>
#include <rtline/util/instant_model.h>
#include <rtline/util/json_processing.h>
#include <rtline/util/algorithm/iterator.h>

#include <util/string/join.h>
#include <util/system/hostname.h>

namespace {
    TNamedSignalSimple SignalSurgeStoreFails("frontend-surge-store-fails");
    TNamedSignalSimple SignalSurgeStoreSuccess("frontend-surge-store-success");
}

NJson::TJsonValue TCarRTFactors::SerializeToJson() const {
    NJson::TJsonValue result;
    if (!!IdleStart) {
        result.InsertValue("idle_start", IdleStart->Seconds());
    }
    if (HasWeather()) {
        result.InsertValue("weather", Weather->SerializeToJson());
    }
    if (HasFinishDuration()) {
        result.InsertValue("finish_duration", *FinishDuration);
    }
    if (HasFinishCoord()) {
        result.InsertValue("finish_coord", FinishCoord->ToString());
    }
    if (HasHiddenDiscount()) {
        result.InsertValue("hidden_discount", *HiddenDiscount);
    }
    return result;
}

NRTLine::TAction TCarRTFactors::SerializeAsAction(TDuration livetime) const {
    NRTLine::TAction action;
    NRTLine::TDocument& doc = action.SetActionType(NRTLine::TAction::atModify).AddDocument();
    doc
        .AddProperty("timestamp", Timestamp.Seconds())
        .SetUrl("SURGE-" + CarId).SetDeadline(Timestamp + livetime);
    doc.AddProperty("generator", HostName());
    if (!!IdleStart) {
        doc.AddProperty("idle_start", IdleStart->Seconds());
    }
    if (HasWeather()) {
        doc.AddProperty("weather", Weather->SerializeToJson());
    }
    if (HasFinishDuration()) {
        doc.AddProperty("finish_duration", *FinishDuration);
    }
    if (HasFinishCoord()) {
        doc.AddProperty("finish_coord", FinishCoord->ToString());
    }
    if (HasHiddenDiscount()) {
        doc.AddProperty("hidden_discount", *HiddenDiscount);
    }
    return action;
}

TRTFactorsConstructor::TRTFactorsConstructor(const TSurgeConstructorConfig& config, const NDrive::IServer* server)
    : TKVSnapshot(server->GetRTLineAPI(config.GetRTLineAPIName()), "surge_constructor")
    , Config(config)
    , Server(server)
    , IndexingClient(RTLineApi->GetIndexingClient())
{
    CHECK_WITH_LOG(IndexingClient);
}

TSet<TString> TRTFactorsConstructor::GetKeys() const {
    auto g = Server->GetDriveAPI()->GetCarsData()->GetCached();
    TSet<TString> result;
    for (auto&& i : g.GetResult()) {
        result.emplace("SURGE-" + i.first);
    }
    return result;
}

TTagsFilter TRTFactorsConstructor::GetVisibilityFilter() const {
    TVector<TDBAction> actions = Server->GetDriveAPI()->GetRolesManager()->GetActions();
    TTagsFilter result;
    for (auto&& i : actions) {
        const TStandartOfferConstructor* builder = i.GetAs<TStandartOfferConstructor>();
        if (!builder) {
            continue;
        }
        result.Merge(builder->GetTagsFilter());
    }
    return result;
}

void TRTFactorsConstructor::AfterRefresh() {
    ui32 expiredSurgeInfos = 0;
    ui32 actualSurgeInfos = 0;
    for (auto it = CarInfosNext.MutableCarInfos().begin(); it != CarInfosNext.MutableCarInfos().end();) {
        if (it->second.GetTimestamp() + Config.GetLivetime() < ModelingNow()) {
            it = CarInfosNext.MutableCarInfos().erase(it);
            ++expiredSurgeInfos;
        } else {
            ++actualSurgeInfos;
            ++it;
        }
    }
    TWriteGuard rg(Mutex);
    CarInfosCurrent = CarInfosNext;
    DEBUG_LOG << "EXPIRED SURGES: " << expiredSurgeInfos << "; ACTUAL SURGES: " << actualSurgeInfos << Endl;
}

void TRTFactorsConstructor::BeforeRefresh() {
    TReadGuard rg(Mutex);
    CarInfosNext.Detach();
}

bool TRTFactorsConstructor::OnDoc(const TReadSearchProtoHelper& helper) {
    TCarRTFactors info;
    if (TCarRTFactors::Parse(helper, info)) {
        CarInfosNext.MutableCarInfos()[info.GetCarId()] = info;
    }
    return true;
}

class TUserWeight {
    R_FIELD(double, WDuration, 0);
    R_FIELD(ui32, WCount, 0);
};

TMap<TString, double> TRTFactorsConstructor::BuildDiscountsInfo(const TSet<TString>& objects) const {
    TMap<TString, double> result;
    const TString userId = Server->GetSettings().GetValueDef<TString>("dynamic_features.discounts.user", "");
    TUserPermissions::TPtr userPermissions = Server->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());
    if (!!userPermissions) {
        auto builders = userPermissions->GetOfferBuilders();
        TVector<TTaggedDevice> taggedObjects;
        if (!Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetCustomObjectsFromCache(taggedObjects, objects)) {
            return result;
        }
        TUserOfferContext uoc(Server, userPermissions, nullptr);
        uoc.SetNeedHistoryFreeTimeFees(false);
        uoc.SetNeedGeoFeatures(false);
        uoc.SetNeedUserFeatures(false);
        uoc.SetNeedUserDestinationSuggest(false);
        uoc.SetAccountName("card");
        uoc.SetAccountDescriptions({});
        uoc.Prefetch();
        for (auto&& obj : taggedObjects) {
            if (obj.IsPerformed() || obj.PriorityUsage()) {
                continue;
            }
            bool wasDiscount = false;
            for (auto&& i : builders) {
                const TStandartOfferConstructor* soc = dynamic_cast<const TStandartOfferConstructor*>(i.Get());
                if (!soc || soc->GetType() != "standart_offer_builder") {
                    continue;
                }
                TVector<IOfferReport::TPtr> offers;
                TOffersBuildingContext offersBuildingContext(Server);
                offersBuildingContext.SetCarWaitingDuration(0);
                offersBuildingContext.SetOverridenWalkingDuration(TDuration::Zero());
                offersBuildingContext.SetUserHistoryContext(uoc);
                offersBuildingContext.SetNeedGeoFeatures(false);
                offersBuildingContext.SetNeedRouteFeatures(false);
                offersBuildingContext.SetNeedTaxiSurge(false);
                offersBuildingContext.SetCarId(obj.GetId());
                NDrive::TInfoEntitySession session;
                auto offerBuildResult = soc->BuildOffers(*userPermissions, userPermissions->GetOfferCorrections(), offers, offersBuildingContext, Server, session);
                if (offerBuildResult == EOfferCorrectorResult::Problems) {
                    ERROR_LOG << "cannot build offers for " << obj.GetId() << ": " << session.GetStringReport() << Endl;
                    continue;
                }
                if (offerBuildResult == EOfferCorrectorResult::Success && offers.size()) {
                    const IOfferWithDiscounts* owd = offers.front()->GetOfferAs<IOfferWithDiscounts>();
                    if (owd) {
                        const double kff = owd->GetSummaryHiddenDiscount().GetDiscount("old_state_riding");
                        if (kff) {
                            result.emplace(obj.GetId(), kff);
                            wasDiscount = true;
                        }
                    }
                    break;
                }
            }
        }
    }
    return result;
}

class TFuturePositionInfo {
private:
    R_READONLY(TString, ObjectId);
    R_READONLY(TGeoCoord, ExpectedCoord);
    R_READONLY(TDuration, Duration);
private:
    NThreading::TFuture<NGraph::TRouter::TOptionalRoute> Route;
public:
    TFuturePositionInfo(const TString& objectId, const TGeoCoord& c)
        : ObjectId(objectId)
        , ExpectedCoord(c)
    {
    }

    void ReadData(const TCarRTFactors& factors) {
        if (factors.HasFinishDuration()) {
            Duration = TDuration::MilliSeconds(factors.GetFinishDurationUnsafe() * 1000);
        }
        if (factors.HasFinishCoord()) {
            ExpectedCoord = factors.GetFinishCoordUnsafe();
        }
    }

    void InitRoute(const NGraph::TRouter& router, const TGeoCoord& from) {
        Route = router.GetRouteAsync(from, ExpectedCoord);
    }

    void InitRoute(const NDrive::IRouter& router, const TGeoCoord& from) {
        Route = router.GetSummary(from, ExpectedCoord, true);
    }

    void InitStub(const TGeoCoord& from) {
        NGraph::TRouter::TRoute route;
        route.HasPath = true;
        route.Length = from.GetLengthTo(ExpectedCoord);
        route.Time = route.Length / 10;
        Route = NThreading::MakeFuture<NGraph::TRouter::TOptionalRoute>(std::move(route));
    }

    bool InitDuration(const TInstant deadlineRoutes) {
        Route.Wait(deadlineRoutes);
        if (!Route.HasValue()) {
            return false;
        }
        const auto& route = Route.GetValue();
        if (!route) {
            return false;
        }
        if (!route->HasPath) {
            return false;
        }
        Duration = TDuration::MilliSeconds(route->Time * 1000);
        return true;
    }
};

TMap<TString, TFuturePositionInfo> TRTFactorsConstructor::BuildFinishDurationInfo(TSet<TString>& objects, const TConstDevicesSnapshot& snapshots) const {
    const ISettings& settings = Server->GetSettings();
    const TRTLineAPI* rtlineRouteAPI = Server->GetRTLineAPI(Config.GetCarRouterAPI());
    auto bSessions = Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", TInstant::Zero());
    TMap<TString, TFuturePositionInfo> tasks;
    if (rtlineRouteAPI && bSessions && Server->GetSettings().GetValueDef("api.car_router.enabled", true)) {
        TCarFactorsSnapshot previous = GetInfoActual();
        const TInstant deadlineRoutes = Now() + Server->GetSettings().GetValueDef<TDuration>("api.car_router.timeout.finish_duration", TDuration::Seconds(10));
        bool enableMapsRouter = settings.GetValue<bool>("api.car_router.maps_router.enabled").GetOrElse(true);
        bool enableStubRouter = settings.GetValue<bool>("stub_router.enabled").GetOrElse(false);
        auto mapsRouter = Server->GetMapsRouter();

        NGraph::TRouter router(rtlineRouteAPI->GetSearchClient());
        TMap<TString, NThreading::TFuture<NGraph::TBaseRouter::TRoute>> routes;
        auto sessions = bSessions->GetSessionsActual();
        for (auto&& i : sessions) {
            const TBillingSession* bs = dynamic_cast<const TBillingSession*>(i.Get());
            if (!bs) {
                continue;
            }
            const TFixPointOffer* fpOffer = dynamic_cast<const TFixPointOffer*>(bs->GetCurrentOffer().Get());
            if (!fpOffer) {
                continue;
            }
            tasks.emplace(fpOffer->GetObjectId(), TFuturePositionInfo(fpOffer->GetObjectId(), fpOffer->GetFinish()));
        }
        TVector<TDBTag> tags;
        {
            auto session = Server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            bool restored = Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({}, {TTagReservationFutures::TypeName}, tags, session);
            if (!restored) {
                ERROR_LOG << "BuildFinishDurationInfo: cannot restore " << session.GetStringReport() << Endl;
            }
        }
        for (auto&& i : tags) {
            if (tasks.contains(i.GetObjectId())) {
                continue;
            }
            const TTagReservationFutures* fTag = i.GetTagAs<TTagReservationFutures>();
            if (!fTag || !fTag->HasExpectedCoord()) {
                continue;
            }
            tasks.emplace(i.GetObjectId(), TFuturePositionInfo(i.GetObjectId(), fTag->GetExpectedCoordUnsafe()));
        }
        for (auto it = tasks.begin(); it != tasks.end();) {
            const TString& objectId = it->first;
            auto itSnapshot = snapshots.find(objectId);
            if (itSnapshot == snapshots.end()) {
                it = tasks.erase(it);
                continue;
            }
            TMaybe<NDrive::TLocation> current = itSnapshot->second.GetLocation();
            if (!current) {
                it = tasks.erase(it);
                continue;
            }

            const auto previousInfo = previous.find(objectId);
            if (previousInfo != previous.end()) {
                it->second.ReadData(previousInfo->second);
            }
            if (enableStubRouter) {
                it->second.InitStub(current->GetCoord());
            } else if (enableMapsRouter && mapsRouter) {
                it->second.InitRoute(*mapsRouter, current->GetCoord());
            } else {
                it->second.InitRoute(router, current->GetCoord());
            }
            ++it;
        }
        for (auto it = tasks.begin(); it != tasks.end();) {
            if (it->second.InitDuration(deadlineRoutes)) {
                objects.emplace(it->first);
                ++it;
            } else {
                it = tasks.erase(it);
            }
        }
    }
    return tasks;
}

TMap<TString, TWeatherInfo> TRTFactorsConstructor::BuildWeatherInfo(const TConstDevicesSnapshot& snapshots) const {
    TMap<TString, TWeatherInfo> result;
    if (Server->GetWeatherAPI() && Server->GetSettings().GetValueDef("api.weather.enabled", true)) {
        TMap<TString, TGeoCoord> deviceCoords;
        for (auto&& i : snapshots) {
            auto location = i.second.GetRawLocation();
            if (!!location) {
                deviceCoords.emplace(i.first, location->GetCoord());
            }
        }
        result = Server->GetWeatherAPI()->GetWeather(deviceCoords);
    }
    return result;
}

bool TRTFactorsConstructor::BuildSurge() const {
    const TInstant now = Now();

    const TTagsFilter filter = GetVisibilityFilter();
    if (filter.IsEmpty()) {
        return true;
    }

    TSet<TString> objects;
    if (!Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetIdsFromCache(filter, objects, TInstant::Zero())) {
        ERROR_LOG << "Cannot fetch data from cache for surge building" << Endl;
        return true;
    }

    auto bSessions = Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", TInstant::Zero());
    TConstDevicesSnapshot snapshots = Server->GetSnapshotsManager().GetSnapshots();

    INFO_LOG << "BuildSurge::BuildDiscountsInfo start" << Endl;
    TMap<TString, double> discounts = BuildDiscountsInfo(objects);
    INFO_LOG << "BuildSurge::BuildFinishDurationInfo start" << Endl;
    TMap<TString, TFuturePositionInfo> secondsForFinish = BuildFinishDurationInfo(objects, snapshots);
    INFO_LOG << "BuildSurge::BuildWeatherInfo start" << Endl;
    TMap<TString, TWeatherInfo> devicesWeather = BuildWeatherInfo(snapshots);

    INFO_LOG << "BuildSurge::Merge start" << Endl;
    TMap<TString, TCarRTFactors> carDemand;
    auto itWeather = devicesWeather.begin();
    auto itFinishInfo = secondsForFinish.begin();
    auto itDiscount = discounts.begin();
    for (auto&& i : objects) {
        auto& demandInfo = carDemand[i];
        demandInfo.SetTimestamp(now);
        demandInfo.SetCarId(i);

        if (Advance(itWeather, devicesWeather.end(), i)) {
            demandInfo.SetWeather(itWeather->second);
        }

        if (Advance(itDiscount, discounts.end(), i)) {
            demandInfo.SetHiddenDiscount(itDiscount->second);
        }

        if (Advance(itFinishInfo, secondsForFinish.end(), i)) {
            demandInfo.SetFinishDuration(itFinishInfo->second.GetDuration().Seconds());
            demandInfo.SetFinishCoord(itFinishInfo->second.GetExpectedCoord());
            continue;
        }

        auto userSessions = bSessions->GetObjectSessions(i);
        auto lastSessionTS = TMaybe<TInstant>();
        for (auto it = userSessions.rbegin(); it != userSessions.rend(); ++it) {
            const TBillingSession* bSession = dynamic_cast<const TBillingSession*>(it->Get());
            if (bSession) {
                TDuration freeDuration;
                TDuration pricedDuration;
                lastSessionTS = bSession->GetLastTS();
                if (bSession->GetFreeAndPricedDurations(freeDuration, pricedDuration) && pricedDuration != TDuration::Zero()) {
                    break;
                }
            }
        }
        demandInfo.SetIdleStart(lastSessionTS);
    }
    const auto& settings = Server->GetSettings();
    bool enableIdleStartFromCompiledSessions = settings.GetValue<bool>("surge.idle_start.enable_compiled_sessions").GetOrElse(true);
    auto depth = enableIdleStartFromCompiledSessions
        ? settings.GetValue<TDuration>("surge.idle_start.depth").GetOrElse(TDuration::Days(7))
        : Server->GetDriveAPI()->GetConfig().GetDeviceTagsHistoryConfig().GetDeep();
    auto horizon = now - depth;
    if (enableIdleStartFromCompiledSessions) {
        INFO_LOG << "BuildSurge::CompiledSessions start" << Endl;
        TVector<TString> ids;
        for (auto&& [id, info] : carDemand) {
            if (info.HasIdleStart()) {
                IdleStartCache.erase(id);
                continue;
            }
            auto cachedIdleStart = IdleStartCache.find(id);
            if (cachedIdleStart) {
                info.SetIdleStart(cachedIdleStart->second);
                continue;
            }
            ids.push_back(id);
        }

        const auto& compiledSessionManager = Server->GetDriveDatabase().GetCompiledSessionManager();
        auto tx = compiledSessionManager.BuildTx<NSQL::ReadOnly | NSQL::Deferred>();
        auto ydbTx = Server->GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("rtfactors_constructor", Server);
        auto optionalCompiledSessions = compiledSessionManager.GetObject<TCompiledRiding>({ ids }, tx, ydbTx, horizon);
        if (!optionalCompiledSessions) {
            tx.Check();
        }
        INFO_LOG << "BuildSurge::CompiledSessions: " << optionalCompiledSessions->size() << " sessions" << Endl;
        std::sort(optionalCompiledSessions->begin(), optionalCompiledSessions->end(), [](const TCompiledRiding& left, const TCompiledRiding& right) {
            return left.GetFinishInstant() < right.GetFinishInstant();
        });
        for (auto&& session : Reversed(*optionalCompiledSessions)) {
            const auto& id = session.GetObjectId();
            if (!session.HasRidingDuration() || !session.GetRidingDurationRef()) {
                continue;
            }
            auto& demandInfo = carDemand[id];
            if (demandInfo.HasIdleStart()) {
                continue;
            }
            demandInfo.SetIdleStart(session.GetFinishInstant());
            IdleStartCache.emplace(id, session.GetFinishInstant());
        }
        INFO_LOG << "BuildSurge::CompiledSessions finish" << Endl;
    }
    for (auto&& [id, info] : carDemand) {
        if (info.HasIdleStart()) {
            continue;
        }
        info.SetIdleStart(horizon);
    }

    INFO_LOG << "BuildSurge::Store start" << Endl;

    SetInfo(carDemand);
    INFO_LOG << "BuildSurge finish" << Endl;

    return true;
}

bool TRTFactorsConstructor::SetInfo(const TMap<TString, TCarRTFactors>& infos) const {
    NRTLine::TSendParams sendParams(true, false, true);
    sendParams.SetInstantReply(true);
    TVector<NRTLine::TAction> actions;
    for (auto&& i : infos) {
        actions.emplace_back(i.second.SerializeAsAction(Config.GetLivetime()));
    }
    TMap<TString, NUtil::THttpReply> results = IndexingClient->SendSync(actions, sendParams, Now() + Config.GetStoreTimeout(), Config.GetStoreMaxInFlight());
    for (auto&& i : results) {
        if (i.second.Code() != 200) {
            SignalSurgeStoreFails.Signal(1);
        } else {
            SignalSurgeStoreSuccess.Signal(1);
        }
    }
    return true;
}

bool TRTFactorsConstructor::GetInfo(const TString& id, TCarRTFactors& info) const {
    TReadGuard rg(Mutex);
    auto it = CarInfosCurrent.find(id);
    if (it == CarInfosCurrent.end() || it->second.GetTimestamp() + Config.GetLivetime() < ModelingNow()) {
        return false;
    }
    info = it->second;
    return true;
}

NRTLine::TSearchReply TKVSnapshot::BuildSearchResult(const TVector<TString>& keys) const {
    NRTLine::TQuery query;
    query.AddExtraParam("meta_search", "first_found");
    query.AddExtraParam("ms", "proto");
    query.SetText(JoinSeq(",", keys));
//    INFO_LOG << query.BuildQuery() << Endl;
    return RTLineApi->GetSearchClient().SendAsyncQuery(query);
}

bool TKVSnapshot::Refresh() {
    TSet<TString> keys = GetKeys();
    TVector<TString> currentKeysPack;
    TVector<NRTLine::TSearchReply> sendResults;
    BeforeRefresh();
    for (auto&& i : keys) {
        currentKeysPack.emplace_back(i);
        if (currentKeysPack.size() == 100) {
            sendResults.emplace_back(BuildSearchResult(currentKeysPack));
            currentKeysPack.clear();
        }
    }
    if (currentKeysPack.size()) {
        sendResults.emplace_back(BuildSearchResult(currentKeysPack));
    }
    ui32 surgesFreshed = 0;
    for (auto&& i : sendResults) {
        if (i.IsSucceeded()) {
            auto& r = i.GetReport();
            for (auto&& groupping : r.GetGrouping()) {
                for (auto&& group : groupping.GetGroup()) {
                    for (auto&& d : group.GetDocument()) {
                        TReadSearchProtoHelper helper(d);
                        ++surgesFreshed;
                        OnDoc(helper);
                    }
                }
            }
        }
    }
    DEBUG_LOG << "KV info refreshed for " << surgesFreshed << " objects" << Endl;
    AfterRefresh();
    return true;
}

NDrive::NProto::TSummarySurgeInfo TSummarySurgeInfo::SerializeToProto() const {
    NDrive::NProto::TSummarySurgeInfo result;
    result.SetDemand(Demand);
    result.SetParkingPrice(ParkingPrice);
    result.SetRidingPrice(RidingPrice);
    result.SetEquilibriumRidingPrice(EquilibriumRidingPrice);
    result.SetEquilibriumParkingPrice(EquilibriumParkingPrice);
    result.SetOriginalRidingPrice(OriginalRidingPrice);
    result.SetOriginalParkingPrice(OriginalParkingPrice);
    if (Weather) {
        *result.MutableWeather() = Weather->SerializeToProto();
    }
    return result;
}

bool TSummarySurgeInfo::DeserializeFromProto(const NDrive::NProto::TSummarySurgeInfo& info) {
    Demand = info.GetDemand();
    ParkingPrice = info.GetParkingPrice();
    RidingPrice = info.GetRidingPrice();
    EquilibriumRidingPrice = info.GetEquilibriumRidingPrice();
    EquilibriumParkingPrice = info.GetEquilibriumParkingPrice();
    OriginalRidingPrice = info.GetOriginalRidingPrice();
    OriginalParkingPrice = info.GetOriginalParkingPrice();
    if (info.HasWeather()) {
        TWeatherInfo wInfo;
        if (!wInfo.DeserializeFromProto(info.GetWeather())) {
            return false;
        }
        SetWeather(std::move(wInfo));
    }
    return true;
}

TSummarySurgeInfo::TSummarySurgeInfo(const ui32 ridingPrice, const ui32 parkingPrice, const TSurgeActivationConfig& sConfig)
    : RidingPrice(ridingPrice)
    , ParkingPrice(parkingPrice)
    , EquilibriumRidingPrice(sConfig.GetEquilibriumRidingPrice())
    , EquilibriumParkingPrice(sConfig.GetEquilibriumParkingPrice())
    , OriginalRidingPrice(ridingPrice)
    , OriginalParkingPrice(parkingPrice) {

}

NJson::TJsonValue TSummarySurgeInfo::SerializeToJson() const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    JWRITE(result, "demand", Demand);
    JWRITE(result, "erp", EquilibriumRidingPrice);
    JWRITE(result, "epp", EquilibriumParkingPrice);
    JWRITE(result, "rp", RidingPrice);
    JWRITE(result, "pp", ParkingPrice);
    JWRITE(result, "orp", OriginalRidingPrice);
    JWRITE(result, "opp", OriginalParkingPrice);
    if (HasWeather()) {
        result.InsertValue("w", Weather->SerializeToJson());
    }
    return result;
}

bool TSummarySurgeInfo::DeserializeFromJson(const NJson::TJsonValue& info) {
    JREAD_DOUBLE_OPT(info, "demand", Demand);
    JREAD_DOUBLE_OPT(info, "erp", EquilibriumRidingPrice);
    JREAD_DOUBLE_OPT(info, "epp", EquilibriumParkingPrice);
    JREAD_DOUBLE_OPT(info, "rp", RidingPrice);
    JREAD_DOUBLE_OPT(info, "pp", ParkingPrice);
    JREAD_DOUBLE_OPT(info, "orp", OriginalRidingPrice);
    JREAD_DOUBLE_OPT(info, "opp", OriginalParkingPrice);
    if (info.Has("w")) {
        TWeatherInfo wInfo;
        if (!wInfo.DeserializeFromJson(info["w"])) {
            return false;
        }
        SetWeather(std::move(wInfo));
    }
    return true;
}

bool TCarRTFactors::Parse(const TReadSearchProtoHelper& helper, TCarRTFactors& result) {
    TString weatherString;
    NJson::TJsonValue weatherJson;
    if (helper.GetProperty("weather", weatherString) && NJson::ReadJsonFastTree(weatherString, &weatherJson)) {
        TWeatherInfo info;
        if (info.DeserializeFromJson(weatherJson)) {
            result.SetWeather(std::move(info));
        }
    }
    {
        ui32 timestamp;
        if (!helper.GetProperty("timestamp", timestamp)) {
            return false;
        }
        result.SetTimestamp(TInstant::Seconds(timestamp));
    }
    {
        double fDuration;
        if (helper.GetProperty("finish_duration", fDuration)) {
            result.SetFinishDuration(fDuration);
        }
    }
    {
        TString fCoord;
        if (helper.GetProperty("finish_coord", fCoord)) {
            TGeoCoord c;
            if (c.DeserializeFromString(fCoord)) {
                result.SetFinishCoord(c);
            }
        }
    }
    {
        double fHiddenDiscount;
        if (helper.GetProperty("hidden_discount", fHiddenDiscount)) {
            result.SetHiddenDiscount(fHiddenDiscount);
        }
    }
    {
        ui64 dIdleSeconds;
        if (helper.GetProperty("idle_start", dIdleSeconds)) {
            result.SetIdleStart(TInstant::Seconds(dIdleSeconds));
        }
    }
    if (helper.GetUrl().StartsWith("SURGE-")) {
        result.CarId = helper.GetUrl().substr(6);
    } else {
        result.CarId = helper.GetUrl();
    }
    return true;
}
