#pragma once

#include "config.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/offers/price/simple.h>

#include <drive/library/cpp/weather/weather_api.h>
#include <drive/library/cpp/threading/concurrent_hash.h>

#include <rtline/library/unistat/signals.h>
#include <rtline/util/auto_actualization.h>
#include <rtline/util/types/accessor.h>

#include <util/datetime/base.h>

class TDBTag;
class TReadSearchProtoHelper;
class TRTLineAPIConfig;
class TSurgeActivationConfig;

namespace NDrive::NProto {
    class TSummarySurgeInfo;
}

namespace NRTLine {
    class TAction;
    class TNehIndexingClient;
    class TSearchReply;
}

class TSummarySurgeInfo {
    R_FIELD(double, Demand, 0);
    R_FIELD(ui32, RidingPrice, 0);
    R_FIELD(ui32, ParkingPrice, 0);
    R_READONLY(ui32, EquilibriumRidingPrice, 0);
    R_READONLY(ui32, EquilibriumParkingPrice, 0);
    R_READONLY(ui32, OriginalRidingPrice, 0);
    R_READONLY(ui32, OriginalParkingPrice, 0);
    R_OPTIONAL(TWeatherInfo, Weather);

public:
    NDrive::NProto::TSummarySurgeInfo SerializeToProto() const;
    bool DeserializeFromProto(const NDrive::NProto::TSummarySurgeInfo& info);

    TSummarySurgeInfo() = default;
    TSummarySurgeInfo(const ui32 ridingPrice, const ui32 parkingPrice, const TSurgeActivationConfig& sConfig);

    NJson::TJsonValue SerializeToJson() const;
    bool DeserializeFromJson(const NJson::TJsonValue& info);
};

class TKVSnapshot: public IAutoActualization {
private:
    NRTLine::TSearchReply BuildSearchResult(const TVector<TString>& keys) const;

protected:
    const TRTLineAPI* RTLineApi = nullptr;

protected:
    virtual TSet<TString> GetKeys() const = 0;
    virtual bool OnDoc(const TReadSearchProtoHelper& helper) = 0;
    virtual void AfterRefresh() = 0;
    virtual void BeforeRefresh() {

    }

public:
    TKVSnapshot(const TRTLineAPI* api, const TString& name)
        : IAutoActualization(name, TDuration::Seconds(20))
        , RTLineApi(api)
    {
        CHECK_WITH_LOG(RTLineApi);
    }

    virtual bool Refresh() override;
};

class TCarRTFactors {
    R_FIELD(TString, CarId);
    R_OPTIONAL(TInstant, IdleStart);
    R_FIELD(TInstant, Timestamp, TInstant::Zero());
    R_OPTIONAL(TWeatherInfo, Weather);
    R_OPTIONAL(double, FinishDuration);
    R_OPTIONAL(TGeoCoord, FinishCoord);
    R_OPTIONAL(double, HiddenDiscount);

public:
    NJson::TJsonValue SerializeToJson() const;
    NRTLine::TAction SerializeAsAction(TDuration livetime) const;

    static bool Parse(const TReadSearchProtoHelper& helper, TCarRTFactors& result);
};

class IOfferBuilderAction;
class TStandartOfferConstructor;

class TTagsFilter;
class TConstDevicesSnapshot;
class TFuturePositionInfo;

class TCarFactorsSnapshot {
private:
    using TCarInfos = TMap<TString, TCarRTFactors>;
    TAtomicSharedPtr<TCarInfos> CarInfos = new TCarInfos;
public:

    TCarInfos::const_iterator begin() const {
        return CarInfos->begin();
    }

    TCarInfos::const_iterator find(const TString& carId) const {
        return CarInfos->find(carId);
    }

    TCarInfos::const_iterator end() const {
        return CarInfos->end();
    }

    void Detach() {
        TAtomicSharedPtr<TCarInfos> newInfo = new TCarInfos(*CarInfos);
        CarInfos = newInfo;
    }

    const TCarInfos& GetCarInfos() const {
        return *CarInfos;
    }

    TCarInfos& MutableCarInfos() {
        return *CarInfos;
    }

    void CopyFrom(const TCarFactorsSnapshot& source) {
        *CarInfos = *source.CarInfos;
    }
};

class TRTFactorsConstructor: public TKVSnapshot {
private:
    mutable TInstant LastRevenueDistributionCalcInstant = TInstant::Zero();
    const TSurgeConstructorConfig Config;
    const NDrive::IServer* Server;
    mutable TRWMutex Mutex;
    mutable TCarFactorsSnapshot CarInfosCurrent;
    mutable TCarFactorsSnapshot CarInfosNext;
    mutable NUtil::TConcurrentHashMap<TString, TMaybe<TInstant>> IdleStartCache;

    const NRTLine::TNehIndexingClient* IndexingClient = nullptr;

private:
    TTagsFilter GetVisibilityFilter() const;
    TMap<TString, double> BuildDiscountsInfo(const TSet<TString>& objects) const;
    TMap<TString, TFuturePositionInfo> BuildFinishDurationInfo(TSet<TString>& objects, const TConstDevicesSnapshot& snapshots) const;
    TMap<TString, TWeatherInfo> BuildWeatherInfo(const TConstDevicesSnapshot& snapshots) const;

protected:
    virtual TSet<TString> GetKeys() const override;
    virtual void AfterRefresh() override;
    virtual void BeforeRefresh() override;
    virtual bool OnDoc(const TReadSearchProtoHelper& helper) override;

public:
    TRTFactorsConstructor(const TSurgeConstructorConfig& config, const NDrive::IServer* server);

    TDuration GetEffectiveDuration() const {
        return Config.GetEffectiveDuration();
    }

    TCarFactorsSnapshot GetInfoActual() const {
        TReadGuard rg(Mutex);
        return CarInfosCurrent;
    }

    TMaybe<TCarRTFactors> GetInfoActual(const TString& carId) const {
        TReadGuard rg(Mutex);
        auto it = CarInfosCurrent.GetCarInfos().find(carId);
        if (it == CarInfosCurrent.GetCarInfos().end()) {
            return {};
        } else {
            return it->second;
        }
    }

    bool BuildSurge() const;
    bool SetInfo(const TMap<TString, TCarRTFactors>& infos) const;
    bool GetInfo(const TString& id, TCarRTFactors& info) const;
};
