#include "history.h"

#include <util/generic/yexception.h>


TOptionalTagHistoryEvents TTagEventsManager::GetEventsByObject(const TString& objectId, NDrive::TEntitySession& session, TEventId sinceId, TInstant sinceTimestamp, const TQueryOptions& queryOptions) const {
    return GetEventsImpl(objectId, /*tagId=*/{}, sinceId, sinceTimestamp, session, queryOptions);
}

TOptionalTagHistoryEvents TTagEventsManager::GetEventsByTag(const TString& tagId, NDrive::TEntitySession& session, TEventId sinceId, TInstant sinceTimestamp, const TQueryOptions& queryOptions) const {
    return GetEventsImpl(/*objectId=*/{}, tagId, sinceId, sinceTimestamp, session, queryOptions);
}

TOptionalTagHistoryEvents TTagEventsManager::GetEventsByRanges(TRange<TEventId> idRange, TRange<TInstant> timestampRange, NDrive::TEntitySession& session, const TQueryOptions& queryOptions) const {
    return GetEventsImpl(/*objectId=*/{}, /*tagId=*/{}, std::move(idRange), std::move(timestampRange), session, queryOptions);
}

TOptionalTagHistoryEvents TTagEventsManager::GetEventsImpl(
    const TString& objectId,
    const TString& tagId,
    TRange<TEventId> idRange,
    TRange<TInstant> timestampRange,
    NDrive::TEntitySession& session,
    const TQueryOptions& queryOptions
) const {
    auto transaction = session.GetTransaction();
    auto query = TStringBuilder() << "SELECT * FROM " << TagsHistoryTableName << " WHERE True";
    if (objectId) {
        query << " AND object_id = " << transaction->Quote(objectId);
    }
    if (tagId) {
        query << " AND tag_id = " << transaction->Quote(tagId);
    }
    if (idRange.From) {
        query << " AND history_event_id >= " << transaction->Quote(*idRange.From);
    }
    if (idRange.To) {
        query << " AND history_event_id < " << transaction->Quote(*idRange.To);
    }
    if (timestampRange.From) {
        query << " AND history_timestamp >= " << transaction->Quote(timestampRange.From->Seconds());
    }
    if (timestampRange.To) {
        query << " AND history_timestamp <= " << transaction->Quote(timestampRange.To->Seconds());
    }
    if (queryOptions.HasActions()) {
        auto actions = TSet<TString>();
        for (auto&& action : queryOptions.GetActionsRef()) {
            actions.insert(ToString(action));
        }
        query << " AND " << queryOptions.PrintCondition("history_action", std::move(actions), *transaction);
    }
    if (queryOptions.GetPerformers()) {
        query << " AND " << queryOptions.PrintCondition("performer", queryOptions.GetPerformers().BuildCondition(), *transaction);
    }
    if (queryOptions.GetTags()) {
        query << " AND " << queryOptions.PrintCondition("tag", queryOptions.GetTags().BuildCondition(), *transaction);
    }
    if (queryOptions.GetTagIds()) {
        query << " AND " << queryOptions.PrintCondition("tag_id", queryOptions.GetTagIds().BuildCondition(), *transaction);
    }
    if (queryOptions.GetObjectIds()) {
        query << " AND " << queryOptions.PrintCondition("object_id", queryOptions.GetObjectIds().BuildCondition(), *transaction);
    }
    if (queryOptions.GetOriginatorIds()) {
        query << " AND " << queryOptions.PrintCondition("history_originator_id", queryOptions.GetOriginatorIds().BuildCondition(), *transaction);
    }
    if (queryOptions.GetUserIds()) {
        query << " AND " << queryOptions.PrintCondition("history_user_id", queryOptions.GetUserIds().BuildCondition(), *transaction);
    }
    for (auto&& [field, condition] : queryOptions.GetGenericConditions()) {
        query << " AND " << queryOptions.PrintCondition(field, condition, *transaction);
    }
    if (const auto& orderBy = queryOptions.GetOrderBy(); !orderBy.empty()) {
        query << " ORDER BY";
        for (size_t i = 0; i < orderBy.size(); ++i) {
            if (i) {
                query << ',';
            }
            query << ' ' << orderBy[i];
            if (queryOptions.GetDescending()) {
                query << " DESC";
            }
        }
    } else {
        query << " ORDER BY history_event_id";
        if (queryOptions.GetDescending()) {
            query << " DESC";
        }
    }
    auto limit = queryOptions.GetLimit();
    if (limit) {
        query << " LIMIT " << limit;
    }

    NStorage::TObjectRecordsSet<TTagHistoryEvent, IHistoryContext> events(&TagHistoryContext);
    auto queryResult = transaction->Exec(query, &events);
    if (!queryResult) {
        session.SetErrorInfo("TagEventsManager::GetEventsImpl", "null QueryResult", EDriveSessionResult::TransactionProblem);
        return {};
    }
    if (!queryResult->IsSucceed()) {
        session.SetErrorInfo("TagEventsManager::GetEventsImpl", "unsuccessful QueryResult", EDriveSessionResult::TransactionProblem);
        return {};
    }

    return events.DetachObjects();
}

bool TAbstractTagsHistoryManager::GetEventsSinceId(TBase::TEventId id, TBase::TEventPtrs& results, const TInstant actuality) const {
    if (Config.GetUseCache()) {
        return TBase::GetEventsSinceId(id, results, actuality);
    } else {
        return TBase::GetEventsSinceIdDirect(id, results);
    }
}
