#pragma once

#include "tags.h"

#include <drive/backend/database/history/manager.h>

class TAbstractTagsHistoryManager: public TAbstractHistoryManager<TConstDBTag> {
private:
    using TBase = TAbstractHistoryManager<TConstDBTag>;

private:
    const ITagsHistoryContext& TagsContext;

protected:
    virtual NEntityTagsManager::EEntityType GetManagerEntityType() const = 0;

public:
    TAbstractTagsHistoryManager(const IHistoryContext& context, const THistoryConfig& config, const TString& tableName)
        : TBase(context, tableName, config)
        , TagsContext(*VerifyDynamicCast<const ITagsHistoryContext*>(&context))
    {
    }

    [[nodiscard]] bool GetEventsSinceId(TBase::TEventId id, TBase::TEventPtrs& results, const TInstant actuality) const override;
};

class TTagEventsManager {
public:
    using TEventId = TTagHistoryEventId;

    struct TQueryOptions: public IBaseSequentialTableImpl::TQueryOptions {
    private:
        using TBase = IBaseSequentialTableImpl::TQueryOptions;

    public:
        using TBase::TBase;

    public:
        R_OPTIONAL(TSet<EObjectHistoryAction>, Actions);
        R_FIELD(NSQL::TStringContainer, Performers);
        R_FIELD(NSQL::TStringContainer, Tags);
        R_FIELD(NSQL::TStringContainer, TagIds);
        R_FIELD(NSQL::TStringContainer, ObjectIds);
        R_FIELD(NSQL::TStringContainer, OriginatorIds);
        R_FIELD(NSQL::TStringContainer, UserIds);
    };

public:
    TTagEventsManager(const IHistoryContext& context, const TString& tableName)
        : TagHistoryContext(context)
        , TagsHistoryTableName(tableName)
    {
    }

    TOptionalTagHistoryEvents GetEventsByObject(const TString& objectId, NDrive::TEntitySession& session, TEventId sinceId = 0, TInstant sinceTimestamp = TInstant::Zero(), const TQueryOptions& queryOptions = {}) const;
    TOptionalTagHistoryEvents GetEventsByTag(const TString& tagId, NDrive::TEntitySession& session, TEventId sinceId = 0, TInstant sinceTimestamp = TInstant::Zero(), const TQueryOptions& queryOptions = {}) const;
    TOptionalTagHistoryEvents GetEventsByRanges(TRange<TEventId> idRange, TRange<TInstant> timestampRange, NDrive::TEntitySession& session, const TQueryOptions& queryOptions) const;

private:
    TOptionalTagHistoryEvents GetEventsImpl(
        const TString& objectId,
        const TString& tagId,
        TRange<TEventId> idRange,
        TRange<TInstant> timestampRange,
        NDrive::TEntitySession& session,
        const TQueryOptions& queryOptions
    ) const;

private:
    const IHistoryContext& TagHistoryContext;
    const TString TagsHistoryTableName;
};

class TCarTagsHistoryManager
    : public TAbstractTagsHistoryManager
    , public TTagEventsManager
{
private:
    using TBase = TAbstractTagsHistoryManager;

protected:
    virtual NEntityTagsManager::EEntityType GetManagerEntityType() const override {
        return NEntityTagsManager::EEntityType::Car;
    }

public:
    static TString TagsHistoryTableName() {
        return "car_tags_history";
    }

public:
    TCarTagsHistoryManager(const IHistoryContext& context, const THistoryConfig& config)
        : TBase(context, config, TagsHistoryTableName())
        , TTagEventsManager(context, TagsHistoryTableName())
    {
    }

private:
    using TBase::GetEventsAll;
};

class TTraceTagsHistoryManager
    : public TDatabaseHistoryManager<TConstDBTag>
    , public TTagEventsManager
{
private:
    using TBase = TDatabaseHistoryManager<TConstDBTag>;

public:
    static TString TagsHistoryTableName() {
        return "trace_tags_history";
    }

public:
    TTraceTagsHistoryManager(const IHistoryContext& context)
        : TBase(context, TagsHistoryTableName())
        , TTagEventsManager(context, TagsHistoryTableName())
    {
    }
};

class TUserTagsHistoryManager
    : public TDatabaseHistoryManager<TConstDBTag>
    , public TTagEventsManager
{
private:
    using TBase = TDatabaseHistoryManager<TConstDBTag>;

public:
    static TString TagsHistoryTableName() {
        return "user_tags_history";
    }

public:
    TUserTagsHistoryManager(const IHistoryContext& context)
        : TBase(context, TagsHistoryTableName())
        , TTagEventsManager(context, TagsHistoryTableName())
    {
    }
};

class TAccountTagsHistoryManager
    : public TDatabaseHistoryManager<TConstDBTag>
    , public TTagEventsManager
{
private:
    using TBase = TDatabaseHistoryManager<TConstDBTag>;

public:
    static TString TagsHistoryTableName() {
        return "account_tags_history";
    }

public:
    TAccountTagsHistoryManager(const IHistoryContext& context)
        : TBase(context, TagsHistoryTableName())
        , TTagEventsManager(context, TagsHistoryTableName())
    {
    }
};
