#pragma once

#include <rtline/util/auto_actualization.h>

#include <library/cpp/threading/hot_swap/hot_swap.h>

#include <util/generic/map.h>
#include <util/generic/noncopyable.h>
#include <util/generic/ptr.h>
#include <util/generic/set.h>
#include <util/system/mutex.h>
#include <util/system/rwlock.h>

template <class TKey, class TPos>
class TRTInvIndex: public IAutoActualization {
private:
    using TInvIndex = TMap<TKey, TSet<TPos>>;

private:
    std::atomic<ui64> CacheActuality = 0;

public:
    class TIndexReader: public TAtomicRefCount<TIndexReader> {
    private:
        TInvIndex InvIndex;

    public:
        using TPtr = TIntrusivePtr<TIndexReader>;

    public:
        TIndexReader() = default;
        TIndexReader(TInvIndex&& index)
            : InvIndex(std::move(index))
        {
        }

        TIndexReader(const TIndexReader& other) = delete;
        TIndexReader& operator=(const TIndexReader& other) = delete;

        const TInvIndex& GetIndex() const {
            return InvIndex;
        }

        const TSet<TPos>& Get(const TKey& key) const {
            auto it = InvIndex.find(key);
            if (it == InvIndex.end()) {
                return Default<TSet<TPos>>();
            } else {
                return it->second;
            }
        }
    };

private:
    TMutex SwitchMutex;
    TMutex RefreshMutex;
    THotSwap<TIndexReader> Snapshot;
    TInvIndex DiffInsert;
    TInvIndex DiffRemove;

public:
    TRTInvIndex(const TString& name, const TDuration period = TDuration::Seconds(1))
        : IAutoActualization(name, period)
    {
        Snapshot.AtomicStore(MakeIntrusive<TIndexReader>());
    }

    typename TIndexReader::TPtr GetInvIndex() const {
        return Snapshot.AtomicLoad();
    }

    class TIndexWriter: public TNonCopyable {
    private:
        TInvIndex* DiffInsert = nullptr;
        TInvIndex* DiffRemove = nullptr;
        THolder<TGuard<TMutex>> Guard;

    public:
        TIndexWriter() = default;
        TIndexWriter(TInvIndex& diffInsert, TInvIndex& diffRemove, TMutex& mutex)
            : DiffInsert(&diffInsert)
            , DiffRemove(&diffRemove)
            , Guard(new TGuard<TMutex>(mutex))
        {
        }
        void AddPosition(const TKey& key, const TPos& val) {
            if (DiffRemove && DiffInsert) {
                auto it = DiffRemove->find(key);
                if (DiffRemove->end() != it) {
                    it->second.erase(val);
                }
                (*DiffInsert)[key].emplace(val);
            }
        }
        void RemovePosition(const TKey& key, const TPos& val) {
            if (DiffRemove && DiffInsert) {
                auto it = DiffInsert->find(key);
                if (DiffInsert->end() != it) {
                    it->second.erase(val);
                }
                (*DiffRemove)[key].emplace(val);
            }
        }
    };

    bool RefreshCache(const TInstant reqActuality) {
        if (CacheActuality.load() < reqActuality.GetValue()) {
            TGuard<TMutex> g(RefreshMutex);
            if (CacheActuality.load() < reqActuality.GetValue()) {
                return Refresh();
            }
        }
        return true;
    }

    TIndexWriter BuildModifier() {
        return TIndexWriter(DiffInsert, DiffRemove, SwitchMutex);
    }

    virtual bool Refresh() override {
        TGuard<TMutex> g(RefreshMutex);
        TInvIndex localDiffInsert;
        TInvIndex localDiffRemove;
        TInstant cacheActualityNew = TInstant::Zero();
        {
            TGuard<TMutex> g(SwitchMutex);
            cacheActualityNew = Now();
            std::swap(DiffInsert, localDiffInsert);
            std::swap(DiffRemove, localDiffRemove);
        }
        if (localDiffInsert.empty() && localDiffRemove.empty()) {
            CacheActuality = cacheActualityNew.GetValue();
            return true;
        }
        typename TIndexReader::TPtr currentSnapshot = Snapshot.AtomicLoad();
        TInvIndex newIndex = currentSnapshot->GetIndex();
        if (localDiffInsert.size() && localDiffRemove.empty() && newIndex.empty()) {
            newIndex = std::move(localDiffInsert);
        } else {
            for (auto&& i : localDiffInsert) {
                auto& newInv = newIndex[i.first];
                for (auto&& k : i.second) {
                    newInv.emplace(k);
                }
            }
            for (auto&& i : localDiffRemove) {
                auto it = newIndex.find(i.first);
                if (it == newIndex.end()) {
                    continue;
                }
                for (auto&& k : i.second) {
                    it->second.erase(k);
                }
            }
        }
        Snapshot.AtomicStore(MakeIntrusive<TIndexReader>(std::move(newIndex)));
        CacheActuality = cacheActualityNew.GetValue();
        return true;
    }
};
