#include "rt_inv_index.h"

#include <library/cpp/testing/unittest/registar.h>

#include <util/datetime/base.h>
#include <util/stream/output.h>
#include <util/string/join.h>
#include <library/cpp/logger/global/global.h>
#include <util/random/random.h>
#include <util/thread/pool.h>


Y_UNIT_TEST_SUITE(RTKeyInvSuite) {
    Y_UNIT_TEST(Simple) {
        TRTInvIndex<TString, ui32> simpleIndex("test");
        {
            auto modifier = simpleIndex.BuildModifier();
            modifier.AddPosition("a", 100);
            modifier.AddPosition("b", 101);
            modifier.AddPosition("c", 102);
            modifier.AddPosition("a", 99);
        }
        simpleIndex.Refresh();
        auto index1 = simpleIndex.GetInvIndex();
        {
            CHECK_WITH_LOG(JoinSeq(",", index1->Get("a")) == "99,100") << JoinSeq(",", index1->Get("a"));
            CHECK_WITH_LOG(JoinSeq(",", index1->Get("c")) == "102") << JoinSeq(",", index1->Get("c"));
            CHECK_WITH_LOG(JoinSeq(",", index1->Get("b")) == "101") << JoinSeq(",", index1->Get("b"));
            CHECK_WITH_LOG(index1->Get("baa").empty());
            {
                auto modifier = simpleIndex.BuildModifier();
                modifier.RemovePosition("a", 98);
                modifier.RemovePosition("a", 100);
                modifier.AddPosition("b", 100);
            }
        }
        simpleIndex.Refresh();
        auto index2 = simpleIndex.GetInvIndex();
        CHECK_WITH_LOG(JoinSeq(",", index2->Get("a")) == "99") << JoinSeq(",", index2->Get("a"));
        CHECK_WITH_LOG(JoinSeq(",", index2->Get("c")) == "102") << JoinSeq(",", index2->Get("c"));
        CHECK_WITH_LOG(JoinSeq(",", index2->Get("b")) == "100,101") << JoinSeq(",", index2->Get("b"));
    }

    class TIndexModifier: public IObjectInQueue {
    private:
        TRTInvIndex<TString, ui32>* Index;
        const TString Key;
        const ui32 StartPos;
        const ui32 FinishPos;
        const TDuration SleepDuration;
    public:
        TIndexModifier(const TString& key, const ui32 startPos, const ui32 finishPos, TRTInvIndex<TString, ui32>& index, const TDuration sleepDuration)
            : Index(&index)
            , Key(key)
            , StartPos(startPos)
            , FinishPos(finishPos)
            , SleepDuration(sleepDuration) {
        }

        void Process(void* /*threadSpecificResource*/) override {
            for (ui32 i = StartPos; i <= FinishPos; ++i) {
                {
                    auto modifier = Index->BuildModifier();
                    modifier.AddPosition(Key, i);
                    modifier.RemovePosition(Key, i - 1);
                }
                Index->Refresh();
                Sleep(RandomNumber<double>() * SleepDuration);
            }
        }
    };

    class TIndexSearcher: public IObjectInQueue {
    private:
        TRTInvIndex<TString, ui32>* Index;
        const TString Key;
        const ui32 StartPos;
        const ui32 FinishPos;
        const TDuration SleepDuration;
    public:
        TIndexSearcher(const TString& key, const ui32 startPos, const ui32 finishPos, TRTInvIndex<TString, ui32>& index, const TDuration sleepDuration)
            : Index(&index)
            , Key(key)
            , StartPos(startPos)
            , FinishPos(finishPos)
            , SleepDuration(sleepDuration) {
        }

        void Process(void* /*threadSpecificResource*/) override {
            for (ui32 i = StartPos; i <= FinishPos; ++i) {
                auto index = Index->GetInvIndex();
                const auto& info = index->Get(Key);
                CHECK_WITH_LOG(info.size() == 1);
                Sleep(RandomNumber<double>() * SleepDuration);
            }
        }
    };

    Y_UNIT_TEST(SimpleMT) {
        TRTInvIndex<TString, ui32> simpleIndex("test");
        {
            THolder<IThreadPool> pool = CreateThreadPool(32);
            for (ui32 i = 0; i < 16; ++i) {
                pool->SafeAddAndOwn(MakeHolder<TIndexModifier>(::ToString(i), 1, 1000, simpleIndex, TDuration::MilliSeconds(10)));
            }
            while (true) {
                auto index = simpleIndex.GetInvIndex()->GetIndex();
                if (index.size() == 16) {
                    break;
                }
                Sleep(TDuration::MilliSeconds(5));
            }
            for (ui32 i = 0; i < 16; ++i) {
                pool->SafeAddAndOwn(MakeHolder<TIndexSearcher>(::ToString(i), 1, 1000, simpleIndex, TDuration::MilliSeconds(10)));
            }
        }
        auto reader = simpleIndex.GetInvIndex();
        for (ui32 i = 0; i < 16; ++i) {
            auto invData = reader->Get(::ToString(i));
            CHECK_WITH_LOG(invData.size() == 1);
            CHECK_WITH_LOG(*invData.begin() == 1000);
        }
    }
}
