#include "tag.h"

#include "tags.h"

#include <drive/backend/database/drive/proto/drive.pb.h>
#include <drive/backend/roles/permissions.h>

#include <rtline/util/blob_with_header.h>
#include <rtline/util/algorithm/type_traits.h>
#include <rtline/util/types/string_pool.h>

static Y_THREAD(TStringPool) TagNamesPool;

NDrive::ITag::TAggregateEvolutionPolicy::TAggregateEvolutionPolicy() {
}

NDrive::ITag::TAggregateEvolutionPolicy::TAggregateEvolutionPolicy(const TString& errorMessage, const TString& uiMessage, TMaybe<i32> errorCode)
    : ErrorCode(errorCode)
    , ErrorMessage(errorMessage)
    , UIErrorMessage(uiMessage)
{
}

NDrive::ITag::TAggregateEvolutionPolicy::~TAggregateEvolutionPolicy() {
}

NDrive::ITag::TAggregateEvolutionPolicy& NDrive::ITag::TAggregateEvolutionPolicy::operator+=(const TAggregateEvolutionPolicy& other) {
    if (!other) {
        Error = other.OptionalError().OrElse(Error);
        ErrorCode = other.OptionalErrorCode().OrElse(ErrorCode);
        ErrorMessage = other.GetErrorMessage();
        Result = other.OptionalResult().OrElse(Result);
        UIErrorMessage = other.GetUIErrorMessage();
        UIErrorTitle = other.GetUIErrorTitle();
        return *this;
    }
    if (other.NewTag) {
        NewTag = other.NewTag;
    }
    if (other.Command) {
        Command = other.Command;
    }
    Policies.insert(Policies.end(), other.Policies.begin(), other.Policies.end());
    return *this;
}

NTagActions::TTagActions NDrive::ITag::GetVisibilityFeatures(const TUserPermissions& permissions) const {
    return permissions.GetActionsByTagIdx(DescriptionIndex);
}

TString NDrive::ITag::GetServiceReport(const ITagsMeta& /*tagsManager*/) const {
    return GetComment();
}

bool NDrive::ITag::CopyOnEvolve(const ITag& source, const TTagEvolutionAction* evolution, const NDrive::IServer& server) {
    Y_UNUSED(source, evolution, server);
    return false;
}

NDrive::TScheme NDrive::ITag::GetScheme(const NDrive::IServer* /*server*/) const {
    return NDrive::TScheme();
}

bool NDrive::ITag::DeserializeSpecialData(const TStringBuf& tagDataStr, const TStringBuf& snapshotDataStr) {
    TBlobWithHeader<NDrive::NProto::TTagHeader> bwh;
    if (!bwh.Load(TBlob::FromString(Base64Decode(tagDataStr)))) {
        return false;
    }
    if (!!snapshotDataStr) {
        if (bwh.GetHeader().HasIsSnapshotProto() && bwh.GetHeader().GetIsSnapshotProto()) {
            const TBlob data = TBlob::FromString(Base64Decode(snapshotDataStr));
            Snapshot = IObjectSnapshot::ConstructFromBlob(data);
        } else {
            NJson::TJsonValue jsonSnapshot;
            if (NJson::ReadJsonFastTree(snapshotDataStr, &jsonSnapshot)) {
                Snapshot = IObjectSnapshot::ConstructFromJson(jsonSnapshot);
            }
        }
        if (!Snapshot) {
            WARNING_LOG << "Incorrect snapshot for " << Name << Endl;
        }
    }

    return DoDeserializeSpecialData(bwh.GetHeader(), bwh.GetData());
}

TBlob NDrive::ITag::SerializeSpecialData() const {
    NDrive::NProto::TTagHeader h;
    h.SetIsSnapshotProto(true);
    TBlob tagData = DoSerializeSpecialData(h);
    TBlobWithHeader<NDrive::NProto::TTagHeader> bwh(h, std::move(tagData));
    return bwh.Save();
}

TTagDescription::TPtr NDrive::ITag::GetMetaDescription(const TString& type) const {
    TTagDescription::TPtr description = TTagDescription::TFactory::Construct(type, "default");
    description->SetType(type);
    description->SetName(type);
    description->SetComment("base object");
    return description;
}

bool NDrive::ITag::DeserializeWithDecoder(const TTagDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const ITagsHistoryContext* hContext) {
    Y_ASSERT(hContext);
    if (!hContext) {
        ERROR_LOG << "Incorrect context for tag restore" << Endl;
        return false;
    }
    TString name;
    READ_DECODER_VALUE_TEMP(decoder, values, name, Name);
    Name = TlsRef(TagNamesPool).Get(name);

    READ_DECODER_VALUE(decoder, values, Performer);

    if (!!decoder.GetStringValue(decoder.GetTagPriority(), values)) {
        i32 tagPriority = 0;
        READ_DECODER_VALUE_TEMP(decoder, values, tagPriority, TagPriority);
        TagPriority = tagPriority;
    }
    auto td = hContext->GetTagDescription(Name);
    if (!td) {
        return false;
    }
    SetDescriptionIndex(td->GetIndex());
    TString snapshotStr;
    READ_DECODER_VALUE_TEMP(decoder, values, snapshotStr, Snapshot);
    const TStringBuf& dataStr = decoder.GetStringValue(decoder.GetData(), values);

    return DeserializeSpecialData(dataStr, snapshotStr);
}

TString NDrive::ITag::GetStringData() const {
    TBlob data = SerializeSpecialData();
    return Base64Encode(TStringBuf(data.AsCharPtr(), data.Size()));
}

TString NDrive::ITag::GetStringSnapshot() const {
    return GetStringSnapshot(Snapshot);
}

TString NDrive::ITag::GetStringSnapshot(IObjectSnapshot::TPtr snapshot) {
    if (!!snapshot) {
        const TBlob snapshotBlob = snapshot->SerializeToBlob();
        return Base64Encode(TStringBuf(snapshotBlob.AsCharPtr(), snapshotBlob.Size()));
    } else {
        return "";
    }
}

NStorage::TTableRecord NDrive::ITag::SerializeToTableRecord(const bool forReport) const {
    NStorage::TTableRecord row;
    row.Set("tag", Name);
    if (!forReport || !!Performer) {
        row.Set("performer", Performer);
    }
    if (TagPriority) {
        row.Set("priority", *TagPriority);
    }
    row.Set("data", GetStringData());
    row.Set("snapshot", GetStringSnapshot());
    return row;
}

NJson::TJsonValue NDrive::ITag::SerializeToJson() const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    result.InsertValue("tag", Name);
    if (!!Performer) {
        result.InsertValue("performer", Performer);
    }
    if (TagPriority) {
        result.InsertValue("priority", *TagPriority);
    }
    SerializeSpecialDataToJson(result);
    return result;
}

NDrive::ITag::TAggregateEvolutionPolicy NDrive::ITag::BuildCustomEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, const EEvolutionMode eMode, NDrive::TEntitySession& session) const {
    Y_UNUSED(dbTag, server, permissions, requestData, eMode, session);
    if (!newTag) {
        return TAggregateEvolutionPolicy::CreateError("Incorrect tag for evolution");
    }
    return {};
}

template struct TExpectedSizeOf<NDrive::ITag, 72>;

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::ITag& object) {
    return object.SerializeToJson();
}
