#pragma once

#include "abstract.h"
#include "tag_description.h"

#include <drive/backend/database/transaction/tx.h>
#include <drive/backend/device_snapshot/abstract.h>

#include <drive/telematics/protocol/actions.h>

#include <library/cpp/object_factory/object_factory.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/types/accessor.h>

#include <util/datetime/base.h>
#include <util/generic/ptr.h>

class IEntityTagsManager;
class ITagsHistoryContext;
class ITagsMeta;
class TBlob;
class TConstDBTag;
class TDBTag;
class TTagEvolutionAction;
class TUserPermissions;
using TUserPermissionsPtr = TIntrusiveConstPtr<TUserPermissions>;
using TUserPermissionsConstPtr = TUserPermissionsPtr;

namespace NDrive {
    class IServer;

    namespace NVega {
        struct TCommand;
    }
}

namespace NDrive::NProto {
    class TTagHeader;
}

enum class EEvolutionMode {
    Default = 0 /* "default" */,
    Force = 1 /* "force" */,
    IgnoreTelematic = 2 /* "ignore_telematic" */
};

namespace NDrive {
    class TEntitySession;

    class ITag {
    public:
        using TFactory = NObjectFactory::TParametrizedObjectFactory<ITag, TString>;
        using TConstPtr = TAtomicSharedPtr<const ITag>;
        using TPtr = TAtomicSharedPtr<ITag>;

    public:
        R_FIELD(TString, Performer);
        R_FIELD(TString, Name);
        R_FIELD(ui32, DescriptionIndex, Max<ui32>());
        R_OPTIONAL(i32, TagPriority);
        R_FIELD(TString, Comment);
        R_FIELD(TInstant, SLAInstant);

    private:
        IObjectSnapshot::TPtr Snapshot;

    public:
        virtual NTagActions::TTagActions GetVisibilityFeatures(const TUserPermissions& permissions) const;

        class TEvolutionContext;

        class ICustomEvolutionPolicy {
        public:
            using TPtr = TAtomicSharedPtr<ICustomEvolutionPolicy>;

        public:
            virtual bool ExecuteAfterEvolveCommit(const TString& objectId, const NDrive::ITag* tag, const TUserPermissions& permissions, const NDrive::IServer* server) const = 0;
            virtual bool ExecuteBeforeEvolution(const TConstDBTag& fromTag, NDrive::ITag::TPtr toTag, const NDrive::IServer* server, const TUserPermissions& permissions, const TEvolutionContext* eContext, NDrive::TEntitySession& session) const = 0;
            virtual bool ExecuteAfterEvolution(const TConstDBTag& fromTag, NDrive::ITag::TPtr toTag, const NDrive::IServer* server, const TUserPermissions& permissions, const TEvolutionContext* eContext, NDrive::TEntitySession& session) const = 0;

            virtual ~ICustomEvolutionPolicy() = default;
        };

        class TEvolutionContext {
            R_FIELD(EEvolutionMode, Mode, EEvolutionMode::Default);
            R_FIELD(bool, Switching, false);
            R_FIELD(bool, Replacing, false);
            R_FIELD(TVector<ICustomEvolutionPolicy::TPtr>, Policies);
        };

        class TAggregateEvolutionPolicy {
        private:
            R_OPTIONAL(TError, Error);
            R_OPTIONAL(i32, ErrorCode);
            R_OPTIONAL(EDriveSessionResult, Result);
            R_FIELD(TString, ErrorMessage);
            R_FIELD(TString, UIErrorTitle);
            R_FIELD(TString, UIErrorMessage);

        public:
            TAtomicSharedPtr<NDrive::NVega::TCommand> Command;
            ITag::TPtr NewTag;
            TVector<ICustomEvolutionPolicy::TPtr> Policies;

        public:
            TAggregateEvolutionPolicy();
            TAggregateEvolutionPolicy(const TString& errorMessage, const TString& uiMessage = {}, TMaybe<i32> errorCode = {});
            ~TAggregateEvolutionPolicy();

            explicit operator bool() const {
                return !Error && ErrorMessage.empty();
            }

            TAggregateEvolutionPolicy& operator+=(const TAggregateEvolutionPolicy& other);

        public:
            static TAggregateEvolutionPolicy CreateError(const TString& errorMessage, const TString& uiMessage = {}, TMaybe<i32> errorCode = {}) {
                return TAggregateEvolutionPolicy(errorMessage, uiMessage, errorCode);
            }
        };

        virtual TAggregateEvolutionPolicy BuildCustomEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, const EEvolutionMode eMode, NDrive::TEntitySession& session) const;

        i32 GetTagPriority(const i32 valueDef) const {
            return TagPriority ? *TagPriority : valueDef;
        }

        TInstant GetSLAInstantUnsafe() const {
            return SLAInstant;
        }
        bool HasSLAInstant() const {
            return SLAInstant != TInstant::Zero();
        }

        bool HasPerformer() const {
            return !!Performer;
        }

        template <class T>
        const T* GetObjectSnapshotAs() const {
            return dynamic_cast<const T*>(Snapshot.Get());
        }

        IObjectSnapshot::TPtr GetObjectSnapshot() const {
            return Snapshot;
        }

        ITag& SetObjectSnapshot(IObjectSnapshot::TPtr value) {
            Snapshot = value;
            return *this;
        }

    protected:
        virtual TBlob DoSerializeSpecialData(NDrive::NProto::TTagHeader& h) const = 0;
        virtual bool DoDeserializeSpecialData(const NDrive::NProto::TTagHeader& h, const TBlob& data) = 0;
        TBlob SerializeSpecialData() const;

    public:
        ITag() = default;
        ITag(const TString& name)
            : Name(name)
        {
        }
        virtual ~ITag() = default;

        bool IsBusyFor(const TString& userId) const {
            return !!GetPerformer() && (GetPerformer() != userId);
        }

        virtual TMaybe<NEntityTagsManager::EMultiplePerformersPolicy> GetMultiPerformingAbility() const {
            return {};
        }

        virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const = 0;

        virtual TString GetPerformGroup() const {
            return "default";
        }

        [[nodiscard]] virtual bool CopyOnEvolve(const ITag& source, const TTagEvolutionAction* evolution, const NDrive::IServer& server);

        virtual NDrive::TScheme GetScheme(const NDrive::IServer* /*server*/) const;

        bool DeserializeSpecialData(const TStringBuf& tagDataStr, const TStringBuf& snapshotDataStr);
        TString GetStringData() const;
        TString GetStringSnapshot() const;
        static TString GetStringSnapshot(IObjectSnapshot::TPtr snapshot);
        virtual TTagDescription::TPtr GetMetaDescription(const TString& type) const;

        virtual TString GetHRDescription() const {
            return GetName() + ": " + GetComment();
        }

        virtual EUniquePolicy GetUniquePolicy() const = 0;

        virtual TString GetServiceReport(const ITagsMeta& tagsManager) const;

        [[nodiscard]] virtual bool OnAfterAdd(const TDBTag& /*self*/, const TString& /*userId*/, const NDrive::IServer* /*server*/, TEntitySession& /*session*/) const {
            return true;
        }

        [[nodiscard]] virtual bool OnBeforeAdd(const TString& /*objectId*/, const TString& /*userId*/, const NDrive::IServer* /*server*/, TEntitySession& /*session*/) {
            return true;
        }

        [[nodiscard]] virtual bool OnAfterRemove(const TDBTag& /*self*/, const TString& /*userId*/, const NDrive::IServer* /*server*/, TEntitySession& /*session*/) const {
            return true;
        }

        [[nodiscard]] virtual bool OnBeforeRemove(const TDBTag& /*self*/, const TString& /*userId*/, const NDrive::IServer* /*server*/, TEntitySession& /*session*/) {
            return true;
        }

        [[nodiscard]] virtual bool OnAfterDropPerform(const TDBTag& /*self*/, const TUserPermissions& /*permissions*/, const NDrive::IServer* /*server*/, TEntitySession& /*session*/) const {
            return true;
        }

        [[nodiscard]] virtual bool OnBeforeDropPerform(const TDBTag& /*self*/, const NDrive::IServer* /*server*/, NDrive::TEntitySession& /*session*/) {
            return true;
        }

        [[nodiscard]] virtual bool OnAfterPerform(TDBTag& /*self*/, const TUserPermissions& /*permissions*/, const NDrive::IServer* /*server*/, TEntitySession& /*session*/) const {
            return true;
        }

        [[nodiscard]] virtual bool OnBeforePerform(TDBTag& /*self*/, const TUserPermissions& /*permissions*/, const NDrive::IServer* /*server*/, TEntitySession& /*session*/) {
            return true;
        }

        [[nodiscard]] virtual bool OnAfterEvolve(const TDBTag& /*fromTag*/, ITag::TPtr /*toTag*/, const TUserPermissions& /*permissions*/, const NDrive::IServer* /*server*/, NDrive::TEntitySession& /*session*/, const TEvolutionContext* /*eContext*/) const {
            return true;
        }

        [[nodiscard]] virtual bool ProvideDataOnEvolve(const TDBTag& /*fromTag*/, const TUserPermissions& /*permissions*/, const NDrive::IServer* /*server*/, NDrive::TEntitySession& /*session*/) {
            return true;
        }

        [[nodiscard]] virtual bool OnBeforeEvolve(const TDBTag& /*fromTag*/, ITag::TPtr /*toTag*/, const TUserPermissions& /*permissions*/, const NDrive::IServer* /*server*/, NDrive::TEntitySession& /*session*/, const TEvolutionContext* /*eContext*/) const {
            return true;
        }

        [[nodiscard]] virtual bool OnAfterUpdate(const TDBTag& /*self*/, ITag::TPtr /*toTag*/, const TString& /*userId*/, const NDrive::IServer* /*server*/, TEntitySession& /*session*/) const {
            return true;
        }

        [[nodiscard]] virtual bool OnBeforeUpdate(const TDBTag& /*self*/, ITag::TPtr /*toTag*/, const TString& /*userId*/, const NDrive::IServer* /*server*/,TEntitySession& /*session*/) const {
            return true;
        }

        [[nodiscard]] virtual bool OnSLAExpired(const TDBTag& /*dbTag*/, const TUserPermissions& /*permissions*/, const IEntityTagsManager* /*manager*/, const NDrive::IServer* /*server*/, NDrive::TEntitySession& /*session*/) const {
            return true;
        }

        class TTagDecoder: public TBaseDecoder {
            R_FIELD(i32, Name, -1);
            R_FIELD(i32, Performer, -1);
            R_FIELD(i32, TagPriority, -1);
            R_FIELD(i32, Snapshot, -1);
            R_FIELD(i32, Data, -1);

        public:
            TTagDecoder() = default;
            TTagDecoder(const TMap<TString, ui32>& decoderBase) {
                Name = GetFieldDecodeIndex("tag", decoderBase);
                Performer = GetFieldDecodeIndex("performer", decoderBase);
                TagPriority = GetFieldDecodeIndex("priority", decoderBase);
                Snapshot = GetFieldDecodeIndex("snapshot", decoderBase);
                Data = GetFieldDecodeIndex("data", decoderBase);
            }
        };
        using TDecoder = TTagDecoder;

        bool DeserializeWithDecoder(const TTagDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const ITagsHistoryContext* hContext);
        NStorage::TTableRecord SerializeToTableRecord(const bool forReport = false) const;

        NJson::TJsonValue SerializeToJson() const;
        virtual void SerializeSpecialDataToJson(NJson::TJsonValue& /*json*/) const = 0;
    };
}
