#include "tag_description.h"
#include "tag.h"

#include <drive/backend/abstract/frontend.h>

#include <drive/telematics/protocol/actions.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/writer/json_value.h>

#include <rtline/util/json_processing.h>

#include <util/string/split.h>

TTagDescription::TFactory::TRegistrator<TTagDescription> TTagDescription::Registrator("default");

bool TTagDescription::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Name);
    READ_DECODER_VALUE(decoder, values, Type);
    if (!Name || !Type) {
        return false;
    }

    READ_DECODER_VALUE(decoder, values, Comment);
    {
        TString displayName;
        READ_DECODER_VALUE_TEMP(decoder, values, displayName, DisplayName);
        if (displayName) {
            DisplayName = displayName;
        }
    }
    READ_DECODER_VALUE_DEF(decoder, values, Index, Index);
    READ_DECODER_VALUE_DEF(decoder, values, DefaultPriority, 0);

    NJson::TJsonValue jsonMeta;
    READ_DECODER_VALUE_JSON(decoder, values, jsonMeta, Meta);
    if (!DeserializeMetaFromJson(jsonMeta)) {
        return false;
    }

    if (decoder.GetRevision() >= 0) {
        ui32 revision = 0;
        if (TryFromString(values.at(decoder.GetRevision()), revision)) {
            Revision = revision;
        }
    }
    Initialize();
    return true;
}

TTagDescription::TTagDescription(const TString& name, const TString& type)
    : Name(name)
    , Type(type)
{
    Initialize();
}

NDrive::TScheme TTagDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result;

    result.Add<TFSString>("color", "Цвет для отображения в сервисном приложении");
    result.Add<TFSBoolean>("deprecated", "Устаревший").SetDefault(false);
    result.Add<TFSBoolean>("enable_sessions", "Строить сервисные сессии по данному тегу").SetDefault(false);
    result.Add<TFSBoolean>("enable_report", "Передавать данные тега в отчёт").SetDefault(false);
    result.Add<TFSDuration>("sla_duration", "SLA продолжительность обработки тега").SetRequired(false);
    result.Add<TFSString>("sla_evolution_tag", "В случае невыполнения условия SLA, эволюция в тег").SetRequired(false);
    result.Add<TFSNumeric>("sla_evolution_priority", "В случае невыполнения условия SLA, приоритет тега").SetRequired(false);
    result.Add<TFSVariants>("cleanup_policy", "Политика очистки истории тега").InitVariants<ECleanupPolicy>();
    result.Add<TFSVariants>("unique_policy", "Политика уникальности тега для объектов").InitVariants<EUniquePolicy>();
    result.Add<TFSBoolean>("transferred_to_double", "Переносить на дубликат").SetDefault(false);

    NDrive::TScheme& iFaceAction = result.Add<TFSArray>("actions", "Допустимые действия при взятии тега").SetElement<NDrive::TScheme>();
    {
        TSet<TString> tagActions = {
            "attach-device",
            "detach-device",
            "attached-devices",
            "ignore-telematics",
            "view-telematics",
            "order",
            "fuel-card-number",
            "install",
            "add-tag-photo-after",
            "add-tag-photo-before",
            "maintenance_list",
            "add_maintenance"
        };
        if (server) {
            auto tagActionsString = server->GetSettings().GetValue<TString>("administration.tags.actions").GetOrElse({});
            StringSplitter(tagActionsString).SplitBySet(" ,").SkipEmpty().AddTo(&tagActions);
        }
        {
            const auto list = NDrive::ListActions();
            tagActions.insert(list.begin(), list.end());
        }

        iFaceAction.Add<TFSVariants>("name", "Наименование действия").SetMultiSelect(false).SetVariants(std::move(tagActions));
    }
    result.Add<TFSString>("tag_flow", "Какому flow принадлежит тег").SetRequired(false);
    result.Add<TFSNumeric>("tag_flow_priority", "Приоритет в пределах конкретного flow").SetRequired(false);
    if (server) {
        TAttributedEntity::FillAttributesScheme(*server, "tags_description", result);
    }
    return result;
}

NJson::TJsonValue TTagDescription::SerializeMetaToJson() const {
    NJson::TJsonValue jsonMeta = DoSerializeMetaToJson();
    NJson::TJsonValue jsonActions(NJson::JSON_ARRAY);
    for (auto&& i : AvailableCarActions) {
        NJson::TJsonValue jsonAction;
        jsonAction["name"] = i;
        jsonActions.AppendValue(jsonAction);
    }
    jsonMeta["actions"] = jsonActions;
    jsonMeta["cleanup_policy"] = NJson::ToJson(CleanupPolicy);
    if (CleanupPolicy == ECleanupPolicy::Ignore) {
        jsonMeta["unlimited_history"] = true;
    }

    JWRITE_DEF(jsonMeta, "color", Color, "");
    JWRITE(jsonMeta, "enable_sessions", EnableSessions);
    JWRITE(jsonMeta, "enable_report", EnableReport);
    JWRITE(jsonMeta, "deprecated", Deprecated);
    JWRITE(jsonMeta, "tag_flow", TagFlow);
    JWRITE(jsonMeta, "tag_flow_priority", TagFlowPriority);
    JWRITE_ENUM(jsonMeta, "unique_policy", UniquePolicy);
    TJsonProcessor::WriteDurationString(jsonMeta, "sla_duration", SLADuration);
    TJsonProcessor::Write(jsonMeta, "sla_evolution_tag", SLAEvolution);
    TJsonProcessor::Write(jsonMeta, "sla_evolution_priority", SLAEvolutionPriority);
    JWRITE(jsonMeta, "transferred_to_double", TransferredToDouble);

    TAttributedEntity::SerializeAttributes(jsonMeta);
    return jsonMeta;
}

NJson::TJsonValue TTagDescription::BuildJsonReport(ELocalization locale) const {
    auto localization = NDrive::HasServer() ? NDrive::GetServer().GetLocalization() : nullptr;
    NJson::TJsonValue report = DoBuildJsonReport(locale);
    report.InsertValue("name", Name);
    if (Color) {
        report.InsertValue("color", Color);
    }
    if (!!Comment) {
        report.InsertValue("comment", Comment);
    }
    if (DefaultPriority != 0) {
        report.InsertValue("default_priority", DefaultPriority);
    }
    report.InsertValue("display_name", localization->GetLocalString(locale, GetDisplayName()));
    if (Index != Max<ui32>()) {
        report.InsertValue("index", Index);
    }
    TJsonProcessor::WriteDurationString(report, "sla_duration", SLADuration);
    TJsonProcessor::Write(report, "sla_evolution_tag", SLAEvolution);
    TJsonProcessor::Write(report, "sla_evolution_priority", SLAEvolutionPriority);
    TJsonProcessor::Write(report, "revision", Revision);

    report.InsertValue("tag_flow", TagFlow);
    report.InsertValue("tag_flow_priority", TagFlowPriority);
    report.InsertValue("tag_type", Type);

    return report;
}

NStorage::TTableRecord TTagDescription::SerializeToTableRecord() const {
    NStorage::TTableRecord row;
    row.Set("default_priority", DefaultPriority);
    row.Set("name", Name);
    row.Set("type", Type);
    row.Set("comment", Comment);
    if (!!DisplayName) {
        row.Set("display_name", DisplayName);
    }
    if (Index != Max<ui32>()) {
        row.Set("description_index", Index);
    }
    row.Set("meta", SerializeMetaToJson().GetStringRobust());
    if (Revision) {
        row.Set("revision", Revision);
    }

    return row;
}

NJson::TJsonValue TTagDescription::SerializeToJson() const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "default_priority", DefaultPriority);
    NJson::InsertField(result, "name", Name);
    NJson::InsertField(result, "type", Type);
    NJson::InsertField(result, "comment", Comment);
    NJson::InsertField(result, "meta", SerializeMetaToJson());
    NJson::InsertNonNull(result, "display_name", DisplayName);
    NJson::InsertNonNull(result, "revision", Revision);
    if (Index != Max<ui32>()) {
        NJson::InsertNonNull(result, "description_index", Index);
    }
    return result;
}

bool TTagDescription::DeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    if (!NJson::ParseField(jsonMeta["cleanup_policy"], CleanupPolicy)) {
        return false;
    }
    if (!CleanupPolicy) {
        bool unlimitedHistory = false;
        if (!NJson::ParseField(jsonMeta["unlimited_history"], unlimitedHistory)) {
            return false;
        }
        if (unlimitedHistory) {
            CleanupPolicy = ECleanupPolicy::Ignore;
        }
    }
    JREAD_STRING_OPT(jsonMeta, "color", Color);
    JREAD_BOOL_OPT(jsonMeta, "enable_sessions", EnableSessions);
    JREAD_BOOL_OPT(jsonMeta, "enable_report", EnableReport);
    JREAD_BOOL_OPT(jsonMeta, "deprecated", Deprecated);
    JREAD_DURATION_OPT(jsonMeta, "sla_duration", SLADuration);
    JREAD_STRING_OPT(jsonMeta, "sla_evolution_tag", SLAEvolution);
    JREAD_INT_OPT(jsonMeta, "sla_evolution_priority", SLAEvolutionPriority);
    JREAD_FROM_STRING_OPT(jsonMeta, "unique_policy", UniquePolicy);
    JREAD_INT_OPT(jsonMeta, "tag_flow_priority", TagFlowPriority);
    JREAD_STRING_OPT(jsonMeta, "tag_flow", TagFlow);
    JREAD_BOOL_OPT(jsonMeta, "transferred_to_double", TransferredToDouble);

    if (jsonMeta.Has("actions")) {
        if (!jsonMeta["actions"].IsArray()) {
            return false;
        }
        for (auto&& i : jsonMeta["actions"].GetArraySafe()) {
            if (!i.Has("name")) {
                return false;
            }
            AvailableCarActions.insert(i["name"].GetString());
        }
    }
    return TAttributedEntity::DeserializeAttributes(jsonMeta) && DoDeserializeMetaFromJson(jsonMeta);
}

bool TTagDescription::DeserializeFromTableRecord(const NStorage::TTableRecord& row) {
    Name = row.Get("name");
    Type = row.Get("type");
    if (!Name || !Type) {
        return false;
    }
    Comment = row.Get("comment");
    row.TryGet("description_index", Index);
    if (const auto& displayName = row.Get("display_name")) {
        DisplayName = displayName;
    }

    if (!row.TryGet("default_priority", DefaultPriority)) {
        DefaultPriority = 0;
    }

    NJson::TJsonValue jsonMeta;
    if (!NJson::ReadJsonFastTree(row.Get("meta"), &jsonMeta)) {
        return false;
    }

    ui32 revision = 0;
    if (row.TryGet("revision", revision)) {
        Revision = revision;
    }

    if (!DeserializeMetaFromJson(jsonMeta)) {
        return false;
    }

    Initialize();
    return true;
}

void TTagDescription::Initialize() {
    auto instance = Hold(NDrive::ITag::TFactory::Construct(Type));
    if (instance) {
        EntityTypes = instance->GetObjectType();
    } else {
        EntityTypes.Clear();
    }
}

TTagDescription::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
    Type = GetFieldDecodeIndex("type", decoderBase);
    Name = GetFieldDecodeIndex("name", decoderBase);
    Comment = GetFieldDecodeIndex("comment", decoderBase);
    DisplayName = GetFieldDecodeIndex("display_name", decoderBase);
    Index = GetFieldDecodeIndex("description_index", decoderBase);
    DefaultPriority = GetFieldDecodeIndex("default_priority", decoderBase);
    Meta = GetFieldDecodeIndex("meta", decoderBase);
    Revision = GetFieldDecodeIndex("revision", decoderBase);

}

template <>
NJson::TJsonValue NJson::ToJson(const ECleanupPolicy& value) {
    return ToString(value);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, ECleanupPolicy& result) {
    return NJson::TryFromJson(value, NJson::Stringify(result));
}
