#pragma once

#include "abstract.h"

#include <drive/backend/common/attributed.h>
#include <drive/backend/database/history/common.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/types/accessor.h>

#include <util/datetime/base.h>

namespace NDrive {
    class IServer;
}

enum class ECleanupPolicy {
    Aggressive  /* "aggressive" */,
    Ignore      /* "ignore" */,
};

enum class EUniquePolicy {
    Undefined = 0 /* "undefined" */,
    NoUnique = 1 /* "no_unique" */,
    SkipIfExists = 2 /* "skip_if_exists" */,
    Rewrite = 3 /* "rewrite" */,
};

class TTagDescription : public TAttributedEntity<TAttributedEntityDefaultFieldNames> {
public:
    using TFactory = NObjectFactory::TObjectFactory<TTagDescription, TString>;
    using TConstPtr = TAtomicSharedPtr<const TTagDescription>;
    using TPtr = TAtomicSharedPtr<TTagDescription>;
    using TId = TString;

public:
    R_FIELD(TString, Name);
    R_FIELD(ui32, Index, Max<ui32>());
    R_FIELD(TString, Type);
    R_FIELD(TString, Comment);
    R_FIELD(int, DefaultPriority, 0);
    R_FIELD(TString, Color);
    R_FIELD(bool, EnableSessions, false);
    R_FIELD(bool, EnableReport, false);
    R_FIELD(TSet<TString>, AvailableCarActions);
    R_FIELD(bool, Deprecated, false);
    R_OPTIONAL(TSet<NEntityTagsManager::EEntityType>, EntityTypes);
    R_OPTIONAL(TString, DisplayName);
    R_OPTIONAL(TDuration, SLADuration);
    R_OPTIONAL(TString, SLAEvolution);
    R_OPTIONAL(i32, SLAEvolutionPriority);
    R_OPTIONAL(ECleanupPolicy, CleanupPolicy);
    R_FIELD(EUniquePolicy, UniquePolicy, EUniquePolicy::Undefined);
    R_FIELD(TString, TagFlow);
    R_FIELD(i32, TagFlowPriority, 0);
    R_FIELD(bool, TransferredToDouble, false);
    R_OPTIONAL(ui32, Revision);

private:
    static TFactory::TRegistrator<TTagDescription> Registrator;

protected:
    virtual NJson::TJsonValue DoSerializeMetaToJson() const {
        return NJson::JSON_MAP;
    }

    virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& /*jsonSpecials*/) {
        return true;
    }

    virtual NJson::TJsonValue DoBuildJsonReport(ELocalization /*locale*/) const {
        return DoSerializeMetaToJson();
    }

    bool DeserializeFromTableRecord(const NStorage::TTableRecord& row);
    NJson::TJsonValue SerializeMetaToJson() const;
    bool DeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta);

public:
    class TDecoder: public TBaseDecoder {
        R_FIELD(i32, Type, -1);
        R_FIELD(i32, Name, -1);
        R_FIELD(i32, Comment, -1);
        R_FIELD(i32, DisplayName, -1);
        R_FIELD(i32, Index, -1);
        R_FIELD(i32, DefaultPriority, -1);
        R_FIELD(i32, Meta, -1);
        R_FIELD(i32, Revision, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase);
    };

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);

public:
    virtual ~TTagDescription() = default;

    TTagDescription::TPtr Clone() const {
        return ConstructFromTableRecord(SerializeToTableRecord());
    }

    TTagDescription(const TString& name, const TString& type);
    TTagDescription() = default;

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const;

    static TTagDescription::TPtr ConstructFromTableRecord(const NStorage::TTableRecord& row) {
        TTagDescription::TPtr result = TFactory::Construct(row.Get("type"), "default");
        if (!result->DeserializeFromTableRecord(row)) {
            return nullptr;
        }
        return result;
    }

    const TString& GetDisplayName() const {
        return DisplayName.GetOrElse(Name);
    }

    NJson::TJsonValue BuildJsonReport(ELocalization locale) const;

    template <class T>
    const T* GetAs() const {
        return dynamic_cast<const T*>(this);
    }

    NStorage::TTableRecord SerializeToTableRecord() const;
    NJson::TJsonValue SerializeToJson() const;

    bool operator< (const TTagDescription& item) const {
        return Name < item.Name;
    }

    bool HasIndex() const{
        return GetIndex() != Max<ui32>();
    }

private:
    void Initialize();
};

inline bool operator<(const TTagDescription::TConstPtr& left, const TTagDescription::TConstPtr& right) {
    if (!left) {
        return true;
    }
    if (!right) {
        return false;
    }
    return *left < *right;
}
