#include "tag_modification.h"

bool ITagsModificationContextImplBase::AddTag(const TString& objectId, ITag::TPtr tag, bool checkCorrect) {
    auto modification = GetModification(objectId, checkCorrect);
    if (!modification) {
        return false;
    }
    modification->MutableTagsForAdd().emplace_back(tag);
    return true;
}

bool ITagsModificationContextImplBase::RemoveTag(const TString& objectId, const TDBTag& tag, bool checkCorrect) {
    auto modification = GetModification(objectId, checkCorrect);
    if (!modification) {
        return false;
    }
    modification->MutableTagsForRemove().emplace_back(tag);
    return true;
}

bool ITagsModificationContextImplBase::UpdateTag(const TString& objectId, const TDBTag& tag, bool checkCorrect) {
    return UpdateTag(objectId, tag, nullptr, checkCorrect);
}

bool ITagsModificationContextImplBase::UpdateTag(const TString& objectId, const TDBTag& tag, ITag::TPtr tagTo, bool checkCorrect) {
    auto modification = GetModification(objectId, checkCorrect);
    if (!modification) {
        return false;
    }
    modification->MutableTagsForUpdate().emplace_back(tag, tagTo);
    return true;
}

bool ITagsModificationContextImplBase::EvolveTag(const TString& objectId, const TDBTag& tagFrom, ITag::TPtr tagTo, bool checkCorrect) {
    auto modification = GetModification(objectId, checkCorrect);
    if (!modification) {
        return false;
    }
    modification->MutableTagsForEvolve().emplace_back(tagFrom, tagTo);
    return true;
}

bool ITagsModificationContextImplBase::ProposeTag(const TString& objectId, const TDBTag& tag, bool checkCorrect) {
    auto modification = GetModification(objectId, checkCorrect);
    if (!modification) {
        return false;
    }
    modification->MutableTagsForPropose().emplace_back(tag);
    return true;
}

TAbstractTagsModification* ITagsModificationContextImplBase::GetModification(const TString& objectId, bool /*checkObject = true*/) {
    auto it = MutableTagModifications().find(objectId);
    if (it == MutableTagModifications().end()) {
        TAbstractTagsModification modNew(objectId, "");
        return &MutableTagModifications().emplace(objectId, std::move(modNew)).first->second;
    }
    else {
        return &it->second;
    }
}

void ITagsModificationContextImplBase::NotifyModificationImpl(const TAbstractTagsModification::EModificationAction action, const TString& tagName, const TVector<TString>& objectReports, const bool dryRunMode, const bool emptyReport, NDrive::INotifier::TPtr notifier) const {
    TString actionHeader;
    switch (action) {
    case TAbstractTagsModification::EModificationAction::Add:
        actionHeader = "установки";
        break;
    case TAbstractTagsModification::EModificationAction::Remove:
        actionHeader = "снятия";
        break;
    case TAbstractTagsModification::EModificationAction::Update:
        actionHeader = "изменения";
        break;
    case TAbstractTagsModification::EModificationAction::Evolve:
        actionHeader = "эволюции";
        break;
    default:
        break;
    }
    TString header = "для " + actionHeader + " тега '" + tagName + "' машины не обнаружены (триггер " + GetProcessName() + ")";
    if (objectReports.size()) {
        TString actionStr;
        switch (action) {
        case TAbstractTagsModification::EModificationAction::Add:
            actionStr = "добавлено";
            break;
        case TAbstractTagsModification::EModificationAction::Remove:
            actionStr = "снято";
            break;
        case TAbstractTagsModification::EModificationAction::Update:
            actionStr = "изменено";
            break;
        case TAbstractTagsModification::EModificationAction::Evolve:
            actionStr = "эволюционировано";
            break;
        default:
            break;
        }
        header = "Задание '" + tagName + "' " + actionStr + " для " + ToString(objectReports.size()) + " машин (процесс " + GetProcessName() + ")";
    }
    if (dryRunMode) {
        header += " (только нотификация)";
    }
    if (objectReports.size()) {
        NDrive::INotifier::MultiLinesNotify(notifier, header, objectReports);
    } else if (emptyReport) {
        NDrive::INotifier::Notify(notifier, header);
    }
}
