#pragma once

#include "tags.h"
#include "tags_manager.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/database/drive_api.h>

#include <rtline/util/types/accessor.h>

#include <util/generic/map.h>
#include <util/generic/serialized_enum.h>

class TAbstractTagsModification {
public:
    enum class EModificationAction {
        Add /* "add" */,
        Remove /* "remove" */,
        Update /* "update_data" */,
        DirectUpdate /* "direct_update_data" */,
        Evolve /* "evolve" */,
        Propose /* "propose" */,
    };

private:
    using TEvolveTags = std::pair<TDBTag, ITag::TPtr>;
    R_READONLY(TString, ObjectId);
    R_READONLY(TString, HRReport);
    R_FIELD(TVector<ITag::TPtr>, TagsForAdd);
    R_FIELD(TVector<TDBTag>, TagsForRemove);
    R_FIELD(TVector<TEvolveTags>, TagsForUpdate);
    R_FIELD(TVector<TEvolveTags>, TagsForEvolve);
    R_FIELD(TVector<TDBTag>, TagsForPropose);

public:
    TAbstractTagsModification(const TString& objectId, const TString& report)
        : ObjectId(objectId)
        , HRReport(report)
    {
    }

    virtual ~TAbstractTagsModification() = default;

    bool IsEmpty() const {
        return TagsForRemove.empty() && TagsForUpdate.empty() && TagsForEvolve.empty() && TagsForAdd.empty() && TagsForPropose.empty();
    }

    template<class TContainer, class TNameAction, class TInfoAction>
    void AddNotificationsDataByTags(TMap<TString, TVector<TString>>& result, const TContainer& container, TNameAction getTagName, TInfoAction createInfo) const {
        for (const auto& element : container) {
            result[getTagName(element)].emplace_back(createInfo(this));
        }
    }

    template<class TInfoAction>
    void AddNotificationsDataByTags(TMap<TString, TVector<TString>>& result, const EModificationAction action, TInfoAction createInfo) const {
        switch (action) {
        case EModificationAction::Add:
            AddNotificationsDataByTags(result, TagsForAdd, [](ITag::TPtr tag) {
                return tag->GetName();
            }, createInfo);
            break;
        case EModificationAction::Remove:
            AddNotificationsDataByTags(result, TagsForRemove, [](const TDBTag& tag) {
                return tag->GetName();
            }, createInfo);
            break;
        case EModificationAction::Update:
            AddNotificationsDataByTags(result, TagsForUpdate, [](const std::pair<TDBTag, ITag::TPtr>& tag) {
                return tag.first->GetName();
            }, createInfo);
            break;
        case EModificationAction::Evolve:
            AddNotificationsDataByTags(result, TagsForEvolve, [](const std::pair<TDBTag, ITag::TPtr>& tag) {
                return tag.first->GetName();
            }, createInfo);
            break;
        case EModificationAction::Propose:
            AddNotificationsDataByTags(result, TagsForPropose, [](const TDBTag& tag) {
                return tag->GetName();
            }, createInfo);
            break;
        default:
            break;
        }
    }
};

class ITagsModificationContextImplBase {
private:
    using TModificationsByObject = TMap<TString, TAbstractTagsModification>;
    R_FIELD(TModificationsByObject, TagModifications);
    R_FIELD(TString, Filter);
    R_FIELD(EUniquePolicy, AddTagPolicy, EUniquePolicy::SkipIfExists);
    R_FIELD(ui32, Delta, 100);
    R_FIELD(ui32, Attempts, 7);
    R_FIELD(TDuration, SleepTime, TDuration::Seconds(5));

protected:
    const NDrive::IServer& Server;

public:
    ITagsModificationContextImplBase(const NDrive::IServer& server)
        : Server(server)
    {
    }

    virtual ~ITagsModificationContextImplBase() = default;

    void Clear() {
        TagModifications.clear();
    }

    bool AddTag(const TString& objectId, ITag::TPtr tag, bool checkCorrect = false);
    bool RemoveTag(const TString& objectId, const TDBTag& tag, bool checkCorrect = false);
    bool UpdateTag(const TString& objectId, const TDBTag& tag, bool checkCorrect = false);
    bool UpdateTag(const TString& objectId, const TDBTag& tag, ITag::TPtr tagTo, bool checkCorrect = false);
    bool EvolveTag(const TString& objectId, const TDBTag& tagFrom, ITag::TPtr tagTo, bool checkCorrect = false);
    bool ProposeTag(const TString& objectId, const TDBTag& tag, bool checkCorrect = false);

    template <class T, class TActor>
    void NotifyModifications(const TMap<TString, T>& modifications, const bool dryRunMode, const bool emptyReport, NDrive::INotifier::TPtr notifier, TActor createReport) const {
        for (const auto& action : GetEnumAllValues<TAbstractTagsModification::EModificationAction>()) {
            TMap<TString, TVector<TString>> objectReportsByTag;
            for (auto&& i : modifications) {
                i.second.AddNotificationsDataByTags(objectReportsByTag, action, createReport);
            }
            for (const auto& tagData : objectReportsByTag) {
                NotifyModificationImpl(action, tagData.first, tagData.second, dryRunMode, emptyReport, notifier);
            }
        }
    }

    void NotifyModifications(const bool dryRunMode, const bool emptyReport, NDrive::INotifier::TPtr notifier) const {
        NotifyModifications(TagModifications, dryRunMode, emptyReport, notifier, [](const TAbstractTagsModification* tagsModification) {
            return tagsModification ? tagsModification->GetHRReport() : "";
        });
    }

    template <class TIter, class TFunc>
    bool ApplyModification(const TIter begin, const TIter end, const IEntityTagsManager& entityTagsManager, NDrive::TEntitySession& session, TFunc getModification) const {
        TUserPermissions::TPtr robotPermissions = Server.GetDriveAPI()->GetUserPermissions(GetRobotUserId(), TUserPermissionsFeatures());

        TVector<TDBTag> tagsForRemove;
        TVector<TDBTag> tagsForUpdate;
        for (auto iter = begin; iter != end; ++iter) {
            const auto& object = getModification(iter);
            if (!entityTagsManager.AddTags(object.GetTagsForAdd(), GetRobotUserId(), object.GetObjectId(), &Server, session, AddTagPolicy, Filter)) {
                ERROR_LOG << "cannot add tags: " << session.GetReport() << Endl;
                return false;
            }

            if (!!robotPermissions) {
                for (const auto& evolveTag : object.GetTagsForEvolve()) {
                    if (!entityTagsManager.EvolveTag(evolveTag.first, evolveTag.second, *robotPermissions, &Server, session)) {
                        ERROR_LOG << "cannot evolve tag: " << session.GetReport() << Endl;
                        return false;
                    }
                }
            }

            for (const auto& proposeTag : object.GetTagsForPropose()) {
                if (!entityTagsManager.ProposeTag(proposeTag, GetRobotUserId(), session)) {
                    ERROR_LOG << "cannot propose tag: " << session.GetReport() << Endl;
                    return false;
                }
            }

            tagsForRemove.insert(tagsForRemove.end(), object.GetTagsForRemove().begin(), object.GetTagsForRemove().end());

            for (const auto& updateTag : object.GetTagsForUpdate()) {
                if (updateTag.second) {
                    if (!entityTagsManager.UpdateTagData(updateTag.first, updateTag.second, GetRobotUserId(), &Server, session)) {
                        ERROR_LOG << "cannot propose tag: " << session.GetReport() << Endl;
                        return false;
                    }
                } else {
                    tagsForUpdate.push_back(updateTag.first);
                }
            }
        }
        if (tagsForRemove && !entityTagsManager.RemoveTags(tagsForRemove, GetRobotUserId(), &Server, session, true)) {
            ERROR_LOG << "cannot remove tags: " << session.GetReport() << Endl;
            return false;
        }
        if (tagsForUpdate && !entityTagsManager.UpdateTagsData(tagsForUpdate, GetRobotUserId(), session)) {
            ERROR_LOG << "cannot update tags: " << session.GetReport() << Endl;
            return false;
        }
        return true;
    }

    template <class TIter>
    bool ApplyModification(const TIter begin, const TIter end, const IEntityTagsManager& entityTagsManager, NDrive::TEntitySession& session) const {
        return ApplyModification(begin, end, entityTagsManager, session, [](const TIter a) { return a->second; });
    }

    template <class T, class TActor>
    bool ApplyModification(const TMap<TString, T>& objects, TActor finishActor, const IEntityTagsManager& entityTagsManager, const bool dryRunMode = false, const TString& comment = "") const {
        for (auto itBeg = objects.begin(); itBeg != objects.end(); ) {
            auto itEnd = itBeg;
            for (ui64 d = 0; d < Delta && itEnd != objects.end(); ++d, ++itEnd) {
                continue;
            }

            bool fail = true;
            for (ui32 att = 0; att < Attempts; ++att) {
                fail = false;
                if (!dryRunMode) {
                    auto tx = entityTagsManager.BuildTx<NSQL::Writable>();
                    tx.SetComment(comment);
                    if (ApplyModification(itBeg, itEnd, entityTagsManager, tx) && tx.Commit()) {
                        break;
                    }
                    ERROR_LOG << "transaction problem: " << tx.GetReport() << Endl;
                    fail = true;
                    Sleep(SleepTime);
                }
            }
            if (fail) {
                return false;
            }
            itBeg = itEnd;
        }
        finishActor();
        return true;
    }

    template <class TActor>
    bool ApplyModification(TActor finishActor, const IEntityTagsManager& entityTagsManager, const bool dryRunMode = false, const TString& comment = "") const {
        return ApplyModification(TagModifications, finishActor, entityTagsManager, dryRunMode, comment);
    }

    bool ApplyModification(const IEntityTagsManager& entityTagsManager, const bool dryRunMode = false, const TString& comment = "") const {
        return ApplyModification([]() {}, entityTagsManager, dryRunMode, comment);
    }

    virtual const TString& GetRobotUserId() const = 0;

private:
    void NotifyModificationImpl(const TAbstractTagsModification::EModificationAction action, const TString& tagName, const TVector<TString>& objectReports, const bool dryRunMode, const bool emptyReport, NDrive::INotifier::TPtr notifier) const;
    virtual const TString& GetProcessName() const = 0;
    virtual TAbstractTagsModification* GetModification(const TString& objectId, bool checkObject = true);
};

template<class TObjectData>
class TTagsModificationContextImpl : public ITagsModificationContextImplBase {
public:
    TTagsModificationContextImpl(const NDrive::IServer& server)
        : ITagsModificationContextImplBase(server)
    {
    }

private:
    virtual TMaybe<TObjectData> GetFetchedObjectData(const TString& /*objectId*/) const {
        return {};
    }

    virtual TAbstractTagsModification* GetModification(const TString& objectId, bool checkObject = true) override {
        auto it = MutableTagModifications().find(objectId);
        if (it == MutableTagModifications().end()) {
            auto objectInfo = GetFetchedObjectData(objectId);
            if (checkObject && !objectInfo) {
                return nullptr;
            }
            TAbstractTagsModification modNew(objectId, objectInfo ? objectInfo->GetHRReport() : "");
            return &MutableTagModifications().emplace(objectId, std::move(modNew)).first->second;
        } else {
            return &it->second;
        }
    }
};

template<class TObjectData>
class TEntityTagsModificationContextImpl : public TTagsModificationContextImpl<TObjectData> {
    using TBase = TTagsModificationContextImpl<TObjectData>;
public:
    TEntityTagsModificationContextImpl(const TString& processName, const TString& userId, const NDrive::IServer& server)
        : TTagsModificationContextImpl<TObjectData>(server)
        , ProcessName(processName)
        , RobotUserId(userId)
    {
    }

    template <class TIter, class TFunc>
    bool ApplyModification(const TIter begin, const TIter end, NDrive::TEntitySession& session, TFunc getModification) const {
        return TBase::ApplyModification(begin, end, GetTagsManager(), session, getModification());
    }

    template <class TIter>
    bool ApplyModification(const TIter begin, const TIter end, NDrive::TEntitySession& session) const {
        return TBase::ApplyModification(begin, end, GetTagsManager(), session);
    }

    template <class T, class TActor>
    bool ApplyModification(const TMap<TString, T>& objects, TActor finishActor, const bool dryRunMode = false, const TString& comment = "") const {
        return TBase::ApplyModification(objects, finishActor, GetTagsManager(), dryRunMode, comment);
    }

    template <class TActor>
    bool ApplyModification(TActor finishActor, const bool dryRunMode = false, const TString& comment = "") const {
        return TBase::ApplyModification(finishActor, GetTagsManager(), dryRunMode, comment);
    }

    bool ApplyModification(const bool dryRunMode = false, const TString& comment = "") const {
        return TBase::ApplyModification(GetTagsManager(), dryRunMode, comment);
    }

    virtual const TString& GetRobotUserId() const override {
        return RobotUserId;
    }

private:
    virtual const IEntityTagsManager& GetTagsManager() const = 0;

    virtual const TString& GetProcessName() const override {
        return ProcessName;
    };

private:
    const TString ProcessName;
    const TString RobotUserId;
};

class TCarTagsModificationContext : public TEntityTagsModificationContextImpl<TDriveCarInfo> {
    using TBase = TEntityTagsModificationContextImpl<TDriveCarInfo>;
public:
    using TBase::TBase;

    virtual TMaybe<TDriveCarInfo> GetFetchedObjectData(const TString& objectId) const override {
        return Server.GetDriveAPI()->GetCarsData()->GetObject(objectId);
    }

    virtual const IEntityTagsManager& GetTagsManager() const override {
        return Server.GetDriveAPI()->GetTagsManager().GetDeviceTags();
    }
};

class TUserTagsModificationContext : public TEntityTagsModificationContextImpl<TDriveUserData> {
    using TBase = TEntityTagsModificationContextImpl<TDriveUserData>;
public:
    using TBase::TBase;

    virtual TMaybe<TDriveUserData> GetFetchedObjectData(const TString& objectId) const override {
        return Server.GetDriveAPI()->GetUsersData()->GetCachedObject(objectId);
    }

    virtual const IEntityTagsManager& GetTagsManager() const override {
        return Server.GetDriveAPI()->GetTagsManager().GetUserTags();
    }
};
