#include "tags.h"

#include <drive/backend/database/drive/proto/drive.pb.h>
#include <drive/backend/database/transaction/assert.h>

#include <library/cpp/json/json_reader.h>

#include <rtline/util/blob_with_header.h>
#include <rtline/util/algorithm/type_traits.h>

NDrive::TEntitySession* ITagsHistoryContext::GetCurrentTx() const {
    return nullptr;
}

TTagDescription::TConstPtr ITagsHistoryContext::GetTagDescription(const TString& name) const {
    const auto& tagsMeta = GetTagsManager();
    auto result = tagsMeta.GetDescriptionByName(name);
    if (!result) {
        auto tx = GetCurrentTx();
        if (tx) {
            auto optionalTagDescription = tagsMeta.GetDescription(name, *tx);
            R_ENSURE(optionalTagDescription, {}, "cannot GetDescription " << name, *tx);
            result = *optionalTagDescription;
        }
    }
    return result;
}

TDBTagDecoder::TDBTagDecoder(const TMap<TString, ui32>& decoderOriginal)
    : NDrive::ITag::TTagDecoder(decoderOriginal)
{
    ObjectId = GetFieldDecodeIndexOrThrow("object_id", decoderOriginal);
    TagId = GetFieldDecodeIndexOrThrow("tag_id", decoderOriginal);
    Filter = GetFieldDecodeIndex("filter", decoderOriginal);
}

NJson::TJsonValue TConstDBTag::BuildJsonReport() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    if (!!TagData) {
        result = TagData->SerializeToJson();
    }
    result.InsertValue("object_id", ObjectId);
    if (!!TagId) {
        result.InsertValue("tag_id", TagId);
    }
    if (Filter) {
        result.InsertValue("filter", Filter);
    }
    return result;
}

void TConstDBTag::DoBuildReportItem(NJson::TJsonValue& item) const {
    item.InsertValue("object_id", ObjectId);
    item.InsertValue("tag_id", TagId);
    if (Filter) {
        item.InsertValue("filter", Filter);
    }
    if (!!TagData) {
        item.InsertValue("tag_details", TagData->SerializeToJson());
        item.InsertValue("tag_name", TagData->GetName());
    }
}

TDBTag TConstDBTag::Clone(const ITagsHistoryContext& tagsHistoryContext) const {
    if (!TagData) {
        ALERT_LOG << GetTagId() << "@" << GetObjectId() << ": Clone failed: no TagData" << Endl;
        Y_ASSERT(TagData);
        return {};
    }
    TDBTag result;
    if (TBaseDecoder::DeserializeFromTableRecord(result, SerializeToTableRecord(), &tagsHistoryContext)) {
        return result;
    } else {
        ALERT_LOG << GetTagId() << "@" << GetObjectId() << ": Clone failed: serialize-deserialized failed for " << TagData->GetName() << Endl;
        Y_ASSERT(false);
        return {};
    }
}

NJson::TJsonValue TConstDBTag::SerializeToJson() const {
    NJson::TJsonValue result = TagData ? TagData->SerializeToJson() : NJson::JSON_NULL;
    result.InsertValue("tag_id", GetTagId());
    result.InsertValue("object_id", GetObjectId());
    const auto& filter = GetFilter();
    if (filter) {
        result.InsertValue("filter", filter);
    }
    return result;
}

bool TConstDBTag::DeserializeWithDecoder(const TDBTagDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext) {
    CHECK_WITH_LOG(hContext);
    const ITagsHistoryContext* context = VerifyDynamicCast<const ITagsHistoryContext*>(hContext);
    READ_DECODER_VALUE(decoder, values, TagId);
    READ_DECODER_VALUE(decoder, values, ObjectId);
    if (decoder.GetFilter() > -1) {
        READ_DECODER_VALUE(decoder, values, Filter);
    }

    TString name;
    READ_DECODER_VALUE_TEMP(decoder, values, name, Name);
    if (!name) {
        ERROR_LOG << "cannot deserialize " << TagId << ": name is missing " << Endl;
        return false;
    }
    {
        auto description = context->GetTagDescription(name);
        if (!description) {
            ERROR_LOG << "cannot deserialize " << TagId << ": cannot get description " << name << Endl;
            return false;
        }
        TagData = ITag::TFactory::Construct(description->GetType());
        if (!TagData) {
            ERROR_LOG << "cannot deserialize " << TagId << ": cannot construct tag type " << description->GetType() << Endl;
            return false;
        }
    }
    return TagData->DeserializeWithDecoder(decoder, values, context);
}

NStorage::TTableRecord TConstDBTag::SerializeToTableRecord() const {
    NStorage::TTableRecord row = TagData->SerializeToTableRecord();
    Y_ENSURE_BT(!!ObjectId);
    row.Set("object_id", ObjectId);
    if (!!TagId) {
        row.Set("tag_id", TagId);
    }
    if (Filter) {
        row.Set("filter", Filter);
    }
    return row;
}

bool TTaggedObject::IsPerformed() const {
    for (auto&& i : Tags) {
        if (i->GetPerformer()) {
            return true;
        }
    }
    return false;
}

bool TTaggedObject::PriorityUsage() const {
    for (auto&& i : Tags) {
        if (i->GetTagPriority(0)) {
            return true;
        }
    }
    return false;
}

TMap<TString, TSet<TString>> TTaggedObject::GetPerformGroups() const {
    TMap<TString, TSet<TString>> result;
    for (auto&& tag : GetTags()) {
        auto performGroup = (tag && tag->GetPerformer()) ? tag->GetPerformGroup() : TString();
        if (performGroup) {
            result[tag->GetPerformGroup()].insert(tag->GetPerformer());
        }
    }
    return result;
}

TSet<TString> TTaggedObject::GetPerformers() const {
    TSet<TString> result;
    for (auto&& tag : GetTags()) {
        if (tag && tag->GetPerformer()) {
            result.insert(tag->GetPerformer());
        }
    }
    return result;
}

TMaybe<TDBTag> TTaggedObject::GetTag(TStringBuf name) const {
    for (auto&& i : Tags) {
        if (i->GetName() == name) {
            return i;
        }
    }
    return {};
}

bool TTaggedObject::HasTag(TStringBuf name) const {
    for (auto&& i : Tags) {
        if (i->GetName() == name) {
            return true;
        }
    }
    return false;
}

bool TTaggedObject::FilterTags(const TSet<TString>& tagNamesSet, TVector<TDBTag>* resultTags) const {
    for (auto&& i : Tags) {
        if (tagNamesSet.contains(i->GetName())) {
            if (resultTags) {
                resultTags->emplace_back(i);
            } else {
                return true;
            }
        }
    }
    if (resultTags) {
        return resultTags->size();
    } else {
        return false;
    }
}

bool TTaggedObject::RemoveTag(const TConstDBTag& tag, TMap<TString, TSet<TString>>* performerMaps, TTagsInvIndex::TIndexWriter* invIndexTagName, TMap<TString, TDBTag>* tagsInfo) {
    ui32 delta = 0;
    for (ui32 i = 0; i + delta < Tags.size();) {
        Tags[i] = Tags[i + delta];
        if (Tags[i].GetTagId() == tag.GetTagId()) {
            if (invIndexTagName) {
                bool found = false;
                for (auto&& tagNameChecker : Tags) {
                    if (tagNameChecker.GetTagId() != tag.GetTagId() && Tags[i]->GetName() == tagNameChecker->GetName()) {
                        found = true;
                        break;
                    }
                }
                if (!found) {
                    invIndexTagName->RemovePosition(Tags[i]->GetName(), Id);
                }
            }
            if (!!tag->GetPerformer() && performerMaps) {
                TSet<TString>& tags = (*performerMaps)[tag->GetPerformer()];
                const size_t removed = tags.erase(tag.GetTagId());
                Y_ASSERT(removed);
                if (tags.empty()) {
                    performerMaps->erase(tag->GetPerformer());
                }
            }
            ++delta;
        } else {
            ++i;
        }
    }
    if (tagsInfo) {
        tagsInfo->erase(tag.GetTagId());
    }
    Tags.resize(Tags.size() - delta);
    return delta;
}

bool TTaggedObject::RefreshTag(const TConstDBTag& tag, const ITagsHistoryContext& context, TMap<TString, TSet<TString>>* performerMaps, TTagsInvIndex::TIndexWriter* invIndexTagName, TMap<TString, TDBTag>* tagsInfo) {
    TString oldTagName;
    for (auto&& i : Tags) {
        if (i.GetTagId() == tag.GetTagId()) {
            if (invIndexTagName && i->GetName() != tag->GetName()) {
                bool found = false;
                for (auto&& tagNameChecker : Tags) {
                    if (tagNameChecker.GetTagId() != tag.GetTagId() && i->GetName() == tagNameChecker->GetName()) {
                        found = true;
                        break;
                    }
                }
                if (!found) {
                    invIndexTagName->RemovePosition(i->GetName(), Id);
                }
                invIndexTagName->AddPosition(tag->GetName(), Id);
            }
            if (i->GetPerformer() != tag->GetPerformer() && performerMaps) {
                if (!!i->GetPerformer()) {
                    TSet<TString>& tags = (*performerMaps)[i->GetPerformer()];
                    size_t removed = tags.erase(tag.GetTagId());
                    Y_ASSERT(removed);
                    if (tags.empty()) {
                        performerMaps->erase(i->GetPerformer());
                    }
                }
                if (!!tag->GetPerformer()) {
                    const bool added = (*performerMaps)[tag->GetPerformer()].emplace(tag.GetTagId()).second;
                    Y_ASSERT(added);
                }
            }
            i = tag.Clone(context);
            if (!i) {
                return false;
            }
            if (tagsInfo) {
                (*tagsInfo)[tag.GetTagId()] = i;
            }
            return true;
        }
    }
    if (!!tag->GetPerformer() && performerMaps) {
        const bool added = (*performerMaps)[tag->GetPerformer()].emplace(tag.GetTagId()).second;
        Y_ASSERT(added);
    }
    auto tagCopy = tag.Clone(context);
    if (!tagCopy) {
        return false;;
    }
    if (invIndexTagName) {
        invIndexTagName->AddPosition(tag->GetName(), Id);
    }
    Tags.emplace_back(tag.Clone(context));
    if (tagsInfo) {
        (*tagsInfo)[tag.GetTagId()] = Tags.back();
    }
    return false;
}

template <>
NJson::TJsonValue NJson::ToJson(const TConstDBTag& object) {
    return object.SerializeToJson();
}

template <>
NJson::TJsonValue NJson::ToJson(const TDBTag& object) {
    return object.SerializeToJson();
}

template <>
NJson::TJsonValue NJson::ToJson(const TTaggedObject& object) {
    NJson::TJsonValue result;
    result["id"] = object.GetId();
    result["tags"] = NJson::ToJson(object.GetTags());
    result["timestamp"] = NJson::ToJson(object.GetTimestamp());
    return result;
}

template <class T>
void AddEventInfo(const TObjectEvent<T>& objectEvent, NJson::TJsonValue& result) {
    NJson::InsertField(result, "history_action", ::ToString(objectEvent.GetHistoryAction()));
    NJson::InsertField(result, "history_timestamp", objectEvent.GetHistoryTimestamp().Seconds());
    NJson::InsertField(result, "history_user_id", objectEvent.GetHistoryUserId());
    NJson::InsertNonNull(result, "history_originator_id", objectEvent.GetHistoryOriginatorId());
    NJson::InsertNonNull(result, "history_comment", objectEvent.GetHistoryComment());
    if (objectEvent.GetHistoryEventId() != objectEvent.IncorrectEventId) {
        NJson::InsertField(result, "history_event_id", objectEvent.GetHistoryEventId());
    }
}

template <>
NJson::TJsonValue NJson::ToJson(const TObjectEvent<TConstDBTag>& objectEvent) {
    NJson::TJsonValue result = NJson::ToJson<TConstDBTag>(objectEvent);
    AddEventInfo(objectEvent, result);
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const TObjectEvent<TDBTag>& objectEvent) {
    NJson::TJsonValue result = NJson::ToJson<TDBTag>(objectEvent);
    AddEventInfo(objectEvent, result);
    return result;
}

template struct TExpectedSizeOf<TConstDBTag, 40>;
template struct TExpectedSizeOf<TDBTag, 40>;
template struct TExpectedSizeOf<TTagHistoryEvent, 80>;
