#pragma once

#include "rt_inv_index.h"
#include "tag.h"
#include "tag_description.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/database/history/event.h>

#include <ydb/public/sdk/cpp/client/ydb_value/value.h>

#include <library/cpp/object_factory/object_factory.h>

#include <rtline/library/storage/structured.h>

class ITagsHistoryContext: public IHistoryContext {
public:
    virtual NDrive::TEntitySession* GetCurrentTx() const;
    virtual TTagDescription::TConstPtr GetTagDescription(const TString& name) const;
    virtual const ITagsMeta& GetTagsManager() const = 0;
};

class TTagsInvIndex
   : public TRTInvIndex<TString, TString>
   , public IMessageProcessor
{
private:
    using TBase = TRTInvIndex<TString, TString>;

public:
    TTagsInvIndex()
        : TBase("car_tags_inv_index")
    {
        RegisterGlobalMessageProcessor(this);
    }

    ~TTagsInvIndex() {
        UnregisterGlobalMessageProcessor(this);
    }

    virtual TString Name() const override {
        return "car_tags_inv_index";
    }
    virtual bool Process(IMessage* /*message*/) override {
        return false;
    }
};

using ITag = NDrive::ITag;

class TDBTagDecoder: public NDrive::ITag::TTagDecoder {
    R_FIELD(i32, ObjectId, -1);
    R_FIELD(i32, TagId, -1);
    R_FIELD(i32, Filter, -1);

public:
    TDBTagDecoder() = default;
    TDBTagDecoder(const TMap<TString, ui32>& decoderOriginal);
};

class TDBTag;

class TConstDBTag {
protected:
    TString ObjectId;
    TString TagId;
    TString Filter;
    ITag::TPtr TagData;

public:
    using TDecoder = TDBTagDecoder;

public:
    TConstDBTag() = default;

    const TString& GetObjectId() const {
        return ObjectId;
    }

    const TString& GetTagId() const {
        return TagId;
    }

    const TString& GetFilter() const {
        return Filter;
    }

    NJson::TJsonValue BuildJsonReport() const;
    void DoBuildReportItem(NJson::TJsonValue& item) const;
    TDBTag Clone(const ITagsHistoryContext& tagsHistoryContext) const;

    static const TString& GetInstanceId(const TConstDBTag& e) {
        return e.GetTagId();
    }

    static const TString& GetObjectId(const TConstDBTag& e) {
        return e.GetObjectId();
    }

    static const TString& GetTypeId(const TConstDBTag& e) {
        return e->GetName();
    }

    const ITag& operator*() const {
        return *Yensured(TagData);
    }

    const ITag* operator->() const {
        return Yensured(TagData).Get();
    }

    explicit operator bool() const {
        return TagData.operator bool();
    }

    NJson::TJsonValue SerializeToJson() const;

    bool DeserializeWithDecoder(const TDBTagDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext);
    NStorage::TTableRecord SerializeToTableRecord() const;

    bool HasData() const {
        return !!TagData;
    }

    template <class T>
    const T* GetTagAs() const {
        return dynamic_cast<const T*>(TagData.Get());
    }

    ITag::TConstPtr GetData() const {
        return TagData;
    }

    template <class T>
    bool Is() const {
        return dynamic_cast<const T*>(TagData.Get());
    }
};

class TDBTag: public TConstDBTag {
public:
    TDBTag() = default;
    TDBTag(TConstDBTag&& other)
        : TConstDBTag(std::move(other))
    {
    }

    TDBTag& SetData(ITag::TPtr value) {
        TagData = value;
        return *this;
    }

    TDBTag& SetObjectId(const TString& value) {
        ObjectId = value;
        return *this;
    }

    TDBTag& SetTagId(const TString& value) {
        TagId = value;
        return *this;
    }

    TDBTag& SetFilter(const TString& value) {
        Filter = value;
        return *this;
    }

    using TConstDBTag::operator->;
    ITag* operator->() {
        return Yensured(TagData).Get();
    }

    using TConstDBTag::GetData;
    ITag::TPtr GetData() {
        return TagData;
    }

    template <class T>
    T* MutableTagAs() {
        return dynamic_cast<T*>(TagData.Get());
    }

    using EPrimitiveType = NYdb::EPrimitiveType;
    static TMap<TString, EPrimitiveType> GetYDBSchema() {
        TMap<TString, EPrimitiveType> schema = {
            {"tag_id",                 EPrimitiveType::String},
            {"snapshot",               EPrimitiveType::String},
            {"object_id",              EPrimitiveType::String},
            {"tag",                    EPrimitiveType::String},
            {"data",                   EPrimitiveType::String},
            {"performer",              EPrimitiveType::String},
            {"priority",               EPrimitiveType::Int32},
            {"filter",                  EPrimitiveType::String},
        };
        return schema;
    }
};
using TDBTags = TVector<TDBTag>;
using TOptionalDBTags = TMaybe<TDBTags>;

class TTaggedObject {
private:
    TString Id;
    TInstant Timestamp;
    TVector<TDBTag> Tags;

public:
    TTaggedObject() = default;
    TTaggedObject(const TString& id, const TVector<TDBTag>& tags, TInstant timestamp)
        : Id(id)
        , Timestamp(timestamp)
        , Tags(tags)
    {
    }

    explicit operator bool() const {
        return !Id.empty();
    }

    template <class T = ITag>
    bool IsFreeForPerform(const TString& userId) const {
        for (auto&& i : Tags) {
            if (!i.GetTagAs<T>()) {
                continue;
            }
            if (!!i->GetPerformer() && i->GetPerformer() != userId) {
                return false;
            }
        }
        return true;
    }

    const TString& GetId() const {
        return Id;
    }

    const TVector<TDBTag>& GetTags() const {
        return Tags;
    }

    TVector<TDBTag>& MutableTags() {
        return Tags;
    }

    TVector<TDBTag>&& DetachTags() {
        return std::move(Tags);
    }

    TInstant GetTimestamp() const {
        return Timestamp;
    }

    template <class T>
    TDBTag GetFirstTagByClass() const {
        for (auto&& i : Tags) {
            if (i.GetTagAs<T>()) {
                return i;
            }
        }
        return Default<TDBTag>();
    }

    template <class T>
    TVector<TDBTag> GetTagsByClass() const {
        TVector<TDBTag> result;
        for (auto&& i : Tags) {
            if (i.GetTagAs<T>()) {
                result.emplace_back(i);
            }
        }
        return result;
    }

    bool IsPerformed() const;
    bool PriorityUsage() const;

    TMap<TString, TSet<TString>> GetPerformGroups() const;
    TSet<TString> GetPerformers() const;
    TMaybe<TDBTag> GetTag(TStringBuf name) const;
    bool HasTag(TStringBuf name) const;
    bool FilterTags(const TSet<TString>& tagNamesSet, TVector<TDBTag>* resultTags) const;
    bool RemoveTag(const TConstDBTag& tag, TMap<TString, TSet<TString>>* performerMaps, TTagsInvIndex::TIndexWriter* invIndexTagName, TMap<TString, TDBTag>* tagsInfo);
    bool RefreshTag(const TConstDBTag& tag, const ITagsHistoryContext& context, TMap<TString, TSet<TString>>* performerMaps, TTagsInvIndex::TIndexWriter* invIndexTagName, TMap<TString, TDBTag>* tagsInfo);
    bool RemoveTag(const TConstDBTag& tag, TMap<TString, TSet<TString>>& performerMaps, TTagsInvIndex::TIndexWriter& invIndexTagName, TMap<TString, TDBTag>& tagsInfo) {
        return RemoveTag(tag, &performerMaps, &invIndexTagName, &tagsInfo);
    }
    bool RefreshTag(const TConstDBTag& tag, const ITagsHistoryContext& context, TMap<TString, TSet<TString>>& performerMaps, TTagsInvIndex::TIndexWriter& invIndexTagName, TMap<TString, TDBTag>& tagsInfo) {
        return RefreshTag(tag, context, &performerMaps, &invIndexTagName, &tagsInfo);
    }
    bool RemoveTag(const TConstDBTag& tag) {
        return RemoveTag(tag, nullptr, nullptr, nullptr);
    }
    bool RefreshTag(const TConstDBTag& tag, const ITagsHistoryContext& context) {
        return RefreshTag(tag, context, nullptr, nullptr, nullptr);
    }
};

using TTaggedDevice = TTaggedObject;
using TTaggedTrace = TTaggedObject;
using TTaggedUser = TTaggedObject;

class TDBTagMeta;
class TDeviceTagsManager;
class TTraceTagsManager;
class TUserTagsManager;
class TAccountTagsManager;

namespace NDrive {
    class TEntitySession;
}

template <class T>
class IDBEntitiesWithPropositionsManager;
template <class T>
class TPropositionsManager;

class ITagsMeta {
public:
    ITagsMeta() = default;
    virtual ~ITagsMeta() = default;

public:
    using TOptionalTagDescription = TMaybe<TTagDescription::TConstPtr>;
    using TTagDescriptions = TVector<TTagDescription::TConstPtr>;
    using TTagDescriptionsByName = TMap<TString, TTagDescription::TConstPtr>;
    using TId = TString;

public:
    virtual TTagDescription::TConstPtr GetDescriptionByName(const TString& name, const TInstant reqActuality = TInstant::Zero()) const = 0;
    virtual TTagDescription::TConstPtr GetDescriptionByIndex(const ui32 index) const = 0;
    virtual TString GetTagTypeByName(const TString& name, const TInstant reqActuality = TInstant::Zero()) const = 0;
    virtual TTagDescriptions GetTagsByType(const TString& tagType, const TInstant reqActuality = TInstant::Zero()) const = 0;

    virtual ITag::TPtr CreateTag(const TString& name, const TString& comment = {}, TInstant reqActuality = TInstant::Zero()) const = 0;
    virtual bool RegisterTag(TTagDescription::TConstPtr description, const TString& userId, NDrive::TEntitySession& session, bool force = false) const = 0;
    virtual bool UnregisterTag(const TString& tagName, const TString& userId, NDrive::TEntitySession& session) const = 0;
    virtual TOptionalTagDescription GetDescription(const TString& name, NDrive::TEntitySession& tx) const = 0;
    virtual TTagDescriptionsByName GetRegisteredTags(const TInstant reqActuality = TInstant::Zero()) const = 0;
    virtual TTagDescriptionsByName GetRegisteredTags(NEntityTagsManager::EEntityType type, const TSet<TString>& tagType = {}, const TInstant actuality = TInstant::Zero()) const = 0;
    virtual TSet<TString> GetRegisteredTagNames(const TSet<TString>& tagTypes, const TInstant reqActuality = TInstant::Zero()) const = 0;
    virtual EUniquePolicy GetTagUniquePolicy(ITag::TConstPtr tag, TTagDescription::TConstPtr description = nullptr, const EUniquePolicy overridePolicy = EUniquePolicy::Undefined) const = 0;

    virtual const IDBEntitiesWithPropositionsManager<TDBTagMeta>& GetEntityManager() const = 0;
    virtual const TPropositionsManager<TDBTagMeta>& GetPropositionManager() const = 0;

    template <class T>
    TAtomicSharedPtr<T> CreateTagAs(const TString& name, const TString& comment = {}, TInstant actuality = TInstant::Zero()) const {
        return std::dynamic_pointer_cast<T>(CreateTag(name, comment, actuality));
    }
};

class IDriveTagsManager: public IStartStopProcess {
public:
    virtual ~IDriveTagsManager() = default;

    virtual const ITagsMeta& GetTagsMeta() const = 0;

    virtual const ITagsHistoryContext& GetContext() const = 0;
    virtual const IEntityTagsManager& GetEntityTagManager(NEntityTagsManager::EEntityType entityType) const = 0;

    virtual const TDeviceTagsManager& GetDeviceTags() const = 0;
    virtual const TTraceTagsManager& GetTraceTags() const = 0;
    virtual const TUserTagsManager& GetUserTags() const = 0;
    virtual const TAccountTagsManager& GetAccountTags() const = 0;
};

using TCarTagHistoryEvent = TObjectEvent<TConstDBTag>;
using TTraceTagHistoryEvent = TObjectEvent<TConstDBTag>;
using TUserTagHistoryEvent = TObjectEvent<TConstDBTag>;
using TAccountTagHistoryEvent = TObjectEvent<TConstDBTag>;

using TTagHistoryEvent = TObjectEvent<TConstDBTag>;
using TTagHistoryEventId = TTagHistoryEvent::TEventId;
using TTagHistoryEventPtr = TObjectEventPtr<TConstDBTag>;
using TTagHistoryEvents = TObjectEvents<TConstDBTag>;
using TOptionalTagHistoryEvent = TOptionalObjectEvent<TConstDBTag>;
using TOptionalTagHistoryEvents = TOptionalObjectEvents<TConstDBTag>;
using TOptionalTagHistoryEventId = TMaybe<TTagHistoryEventId>;
