#include "tags_cache.h"

#include "tags_manager.h"

#include <drive/backend/logging/evlog.h>

namespace {
    TTagCache::TOptionalTaggedObject GetCachedObjectImpl(
        const TString& objectId,
        const IEntityTagsManager& tagManager,
        TDuration defaultLifetime,
        NUtil::TConcurrentCache<TString, TTaggedObject>& objectCache,
        NDrive::TEntitySession& tx,
        bool& expired,
        bool& cached
    ) {
        auto eventLogger = NDrive::GetThreadEventLogger();
        auto lifetimeKey = TStringBuilder() << "tags_manager." << tagManager.GetEntityType() << ".cache_lifetime";
        auto lifetime = NDrive::HasServer()
            ? NDrive::GetServer().GetSettings().GetValue<TDuration>(lifetimeKey)
            : Nothing();
        auto now = Now();
        auto threshold = now - lifetime.GetOrElse(defaultLifetime);
        auto optionalObject = objectCache.find(objectId);
        if (optionalObject && optionalObject->GetTimestamp() > threshold) {
            if (eventLogger) {
                eventLogger->AddEvent(NJson::TMapBuilder
                    ("event", "TagCacheHit")
                    ("object_id", objectId)
                    ("timestamp", NJson::ToJson(optionalObject->GetTimestamp()))
                );
            }
            cached = true;
            return std::move(*optionalObject);
        }
        if (optionalObject) {
            expired = true;
        }

        if (eventLogger) {
            eventLogger->AddEvent(NJson::TMapBuilder
                ("event", "TagCacheMiss")
                ("object_id", objectId)
            );
        }

        auto restoredObject = tagManager.RestoreObject(objectId, tx);
        if (!restoredObject) {
            return {};
        }

        Y_ASSERT(objectId == restoredObject->GetId());
        objectCache.update(objectId, *restoredObject);
        cached = false;
        return std::move(*restoredObject);
    }
}

TTagCache::TTagCache(const IEntityTagsManager& tagManager, const TString& tableName, TDuration defaultLifetime)
    : IAutoActualization(tableName + "-cache", TDuration::Seconds(1))
    , TagManager(tagManager)
    , DefaultLifetime(defaultLifetime)
    , CacheHit({ tableName + "-cache-hit" }, false)
    , CacheMiss({ tableName + "-cache-miss" }, false)
    , CacheExpired({ tableName + "-cache-expired" }, false)
    , CacheInvalidated({ tableName + "-cache-invalidated" }, false)
    , ObjectCache(64 * 1024)
{
}

TTagCache::TExpectedTaggedObject TTagCache::GetCachedObject(const TString& objectId, TInstant statementDeadline) const {
    auto now = Now();
    auto lockTimeout = TDuration::Zero();
    auto statementTimeout = statementDeadline - now;
    auto tx = TagManager.BuildTx<NSQL::ReadOnly | NSQL::Deferred>(lockTimeout, statementTimeout);
    return WrapUnexpected<TCodedException>([&] {
        auto optionalResult = GetCachedObject(objectId, tx);
        if (!optionalResult) {
            tx.Check();
        }
        return std::move(*optionalResult);
    });
}

TTagCache::TOptionalTaggedObject TTagCache::GetCachedObject(const TString& objectId, NDrive::TEntitySession& tx) const {
    bool cached = false;
    bool expired = false;
    auto result = GetCachedObjectImpl(
        objectId,
        TagManager,
        DefaultLifetime,
        ObjectCache,
        tx,
        expired,
        cached
    );
    if (result) {
        if (cached) {
            CacheHit.Signal(1);
        } else {
            CacheMiss.Signal(1);
        }
        if (expired) {
            CacheExpired.Signal(1);
        }
    }
    return result;
}

bool TTagCache::GetStartFailIsProblem() const {
    return false;
}

bool TTagCache::Refresh() {
    auto session = TagManager.BuildSession(true);
    if (!LastEventId) {
        LastEventId = TagManager.GetMaxEventIdOrThrow(session);
    }

    auto since = LastEventId ? *LastEventId + 1 : 0;
    auto optionalEvents = TagManager.GetEventsSince(since, session, 1000);
    if (!optionalEvents) {
        ERROR_LOG << GetName() << ": cannot GetEventsSince " << since << ": " << session.GetStringReport() << Endl;
        return false;
    }
    for (auto&& ev : *optionalEvents) {
        LastEventId = std::max(LastEventId.GetOrElse(0), ev.GetHistoryEventId());
        bool erased = ObjectCache.erase(ev.GetObjectId());
        if (erased) {
            CacheInvalidated.Signal(1);
            INFO_LOG << GetName() << ": invalidate " << ev.GetObjectId() << Endl;
        }
    }
    return true;
}

bool TTagCache::Update(const TTaggedObject& object) const {
    if (!object) {
        return false;
    }
    ObjectCache.update(object.GetId(), object);
    return true;
}
