#include "tags_filter.h"

#include <rtline/library/json/cast.h>
#include <rtline/util/json_processing.h>

#include <util/string/join.h>
#include <util/string/split.h>

TTagsFilter::TTagsFilter(const TSet<TString>& tags, bool withPriority, bool mustHave) {
    TStringBuilder sb;
    for (auto&& i : tags) {
        if (!sb.Empty()) {
            sb << ",";
        }
        if (withPriority) {
            sb << "?";
        }
        if (!mustHave) {
            sb << "-";
        }
        sb << i;
    }
    Y_ENSURE_BT(DeserializeFromString(sb));
}

TTagsFilter TTagsFilter::BuildFromString(const TString& filterDescription) {
    TTagsFilter result;
    result.DeserializeFromString(filterDescription);
    return result;
}

NJson::TJsonValue TTagsFilter::GetDebugInfo() const {
    NJson::TJsonValue result;
    for (auto&& condition : MatchConditions) {
        result["conditions"].AppendValue(condition.SerializeToJson());
    }
    if (OriginalString) {
        result["original_string"] = *OriginalString;
    }
    return result;
}

TSet<TString> TTagsFilter::GetMonoTagNames() const {
    TSet<TString> result;
    for (auto&& i : MatchConditions) {
        auto tn = i.GetMonoTagNames();
        result.insert(tn.begin(), tn.end());
    }
    return result;
}

TSet<TString> TTagsFilter::GetRelatedTagNames() const {
    TSet<TString> result;
    auto line = ToLinearString();
    StringSplitter(line).SplitBySet(" *,#?!()&#+-'").SkipEmpty().Collect(&result);
    return result;
}

bool TTagsFilter::IsInclusive() const {
    for (auto&& condition : MatchConditions) {
        if (condition.IsInclusive()) {
            return true;
        }
    }
    return false;
}

bool TTagsFilter::IsMatching(const TTaggedObject& object, TString* resultFilter) const {
    return IsMatching(object.GetTags(), resultFilter);
}

TString TTagsFilter::ToLinearString() const {
    TStringStream ss;
    for (auto&& i : MatchConditions) {
        if (!ss.Empty()) {
            ss << ",";
        }
        ss << i.ToString();
    }
    return ss.Str();
}

TString TTagsFilter::ToString() const {
    if (OriginalString) {
        return *OriginalString;
    } else {
        return ToLinearString();
    }
}

NJson::TJsonValue TTagsFilter::SerializeToJson() const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    result["conditions"] = ToString();
    return result;
}

bool TTagsFilter::DeserializeFromJson(const NJson::TJsonValue& jsonValue) {
    TString conditionsStr;
    if (!jsonValue.Has("conditions")) {
        const NJson::TJsonValue::TArray* arr;
        if (!jsonValue["details"]["conditions"].GetArrayPointer(&arr)) {
            return false;
        }
        MatchConditions.clear();
        for (auto&& i : *arr) {
            TMatchCondition condition;
            if (!condition.DeserializeFromJson(i)) {
                return false;
            }
            MatchConditions.emplace_back(std::move(condition));
        }
        conditionsStr = ToString();
    } else {
        conditionsStr = jsonValue["conditions"].GetString();
    }
    return DeserializeFromString(conditionsStr);
}

bool TTagsFilter::DeserializeFromString(const TString& conditionsStr) {
    OriginalString = MakeAtomicShared<TString>(conditionsStr);
    TVector<TMatchCondition> conditions;
    if (!TMatchCondition::BuildConditions(*OriginalString, conditions)) {
        return false;
    }
    while (conditions.size()) {
        TVector<TMatchCondition> nextConditions;
        for (auto&& i : conditions) {
            if (!i.GetDeep()) {
                MatchConditions.emplace_back(std::move(i));
            } else {
                TVector<TMatchCondition> flatConditions;
                if (!i.Linearize(flatConditions)) {
                    return false;
                }
                nextConditions.insert(nextConditions.end(), flatConditions.begin(), flatConditions.end());
            }
        }
        conditions = std::move(nextConditions);
    };

    for (auto&& i : MatchConditions) {
        i.SortTokens();
    }
    return true;
}

bool TTagsFilter::TMatchCondition::ParseFromString(TStringBuf rawMatchCondition) {
    TVector<TStringBuf> filters;
    SplitWithBraces(rawMatchCondition, filters, '*', &Deep);
    for (auto&& i : filters) {
        if (!!i) {
            MatchTokens.emplace_back(TMatchToken(i));
        }
    }
    return true;
}

void TTagsFilter::TMatchCondition::SortTokens() {
    std::sort(MatchTokens.begin(), MatchTokens.end());
    MatchTokens.erase(
        std::unique(MatchTokens.begin(), MatchTokens.end()),
        MatchTokens.end()
    );
}

bool TTagsFilter::TMatchCondition::SplitWithBraces(TStringBuf buf, TVector<TStringBuf>& result, const char splitChar, ui32* deep) {
    i32 br = 0;
    if (deep) {
        *deep = 0;
    }
    ui32 startPos = 0;
    for (ui32 pos = 0; pos < buf.size(); ++pos) {
        const char c = buf[pos];
        if (c == ' ') {
            continue;
        }
        if (c == '(') {
            ++br;
            if (deep) {
                *deep = Max<ui32>(*deep, br);
            }
        }
        if (c == ')') {
            --br;
        }
        if (br < 0) {
            return false;
        }
        if (br == 0 && c == splitChar) {
            startPos = pos + 1;
            result.emplace_back();
            continue;
        }
        if (result.empty()) {
            startPos = pos;
            result.emplace_back();
        }
        result.back() = buf.SubStr(startPos, pos - startPos + 1);
    }
    return true;
}

NJson::TJsonValue TTagsFilter::TMatchCondition::SerializeToJson() const {
    NJson::TJsonValue result;
    NJson::TJsonValue& tokensJson = result.InsertValue("tokens", NJson::JSON_ARRAY);
    for (auto&& i : MatchTokens) {
        tokensJson.AppendValue(i.SerializeToJson());
    }
    return result;
}

bool TTagsFilter::TMatchCondition::DeserializeFromJson(const NJson::TJsonValue& info) {
    const NJson::TJsonValue::TArray* arr;
    if (!info["tokens"].GetArrayPointer(&arr)) {
        return false;
    }
    for (auto&& i : *arr) {
        TMatchToken token;
        if (!token.DeserializeFromJson(i)) {
            return false;
        }
        if (!token.TagName) {
            continue;
        }
        MatchTokens.emplace_back(token);
    }
    Sort(MatchTokens.begin(), MatchTokens.end());
    return true;
}

TSet<TString> TTagsFilter::TMatchCondition::GetMonoTagNames() const {
    TSet<TString> result;
    if (MatchTokens.size() == 1) {
        result.emplace(MatchTokens.front().TagName);
    }
    return result;
}

TString TTagsFilter::TMatchCondition::ToString() const {
    TStringStream ss;
    for (auto&& i : MatchTokens) {
        if (!ss.Empty()) {
            ss << "*";
        }
        ss << i.ToString();
    }
    return ss.Str();
}

bool TTagsFilter::TMatchCondition::IsInclusive() const {
    for (auto&& token : MatchTokens) {
        if (token.IsInObjectTags) {
            return true;
        }
    }
    return false;
}

bool TTagsFilter::TMatchCondition::BuildConditions(TStringBuf conditionsStr, TVector<TMatchCondition>& result) {
    TVector<TStringBuf> filters;
    SplitWithBraces(conditionsStr, filters, ',', nullptr);
    for (auto&& i : filters) {
        TMatchCondition condition;
        if (!condition.ParseFromString(i)) {
            return false;
        }
        result.emplace_back(std::move(condition));
    }
    return true;
}

bool TTagsFilter::TMatchCondition::Linearize(TVector<TMatchCondition>& result) const {
    TVector<TMatchToken> tokens;
    TVector<TVector<TMatchCondition>> conditionsList;
    for (auto&& i : MatchTokens) {
        if (i.TagName.StartsWith("(")) {
            if (!i.TagName.EndsWith(")")) {
                return false;
            }

            TVector<TMatchCondition> condLocal;
            if (!TMatchCondition::BuildConditions(i.TagName.SubStr(1).Chop(1), condLocal)) {
                return false;
            }
            conditionsList.emplace_back(std::move(condLocal));
        } else {
            tokens.emplace_back(i);
        }
    }
    if (conditionsList.empty()) {
        return false;
    }
    TMatchCondition first;
    first.MatchTokens = tokens;
    result.emplace_back(std::move(first));
    for (auto&& cl : conditionsList) {
        TVector<TMatchCondition> nextResult;
        for (auto&& r : result) {
            for (auto&& c : cl) {
                TMatchCondition rNew = r;
                rNew.MatchTokens.insert(rNew.MatchTokens.end(), c.MatchTokens.begin(), c.MatchTokens.end());
                rNew.Deep = Max(rNew.Deep, c.Deep);
                nextResult.emplace_back(std::move(rNew));
            }
        }
        std::swap(nextResult, result);
    }
    return true;
}

bool TTagsFilter::TMatchToken::DeserializeFromJson(const NJson::TJsonValue& info) {
    JREAD_STRING(info, "tag_name", TagName);
    JREAD_BOOL_OPT(info, "has_in_object", IsInObjectTags);
    JREAD_FROM_STRING_OPT(info, "hidden", Hidden);
    JREAD_FROM_STRING_OPT(info, "performed", Performed);
    return true;
}

NJson::TJsonValue TTagsFilter::TMatchToken::SerializeToJson() const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    JWRITE(result, "tag_name", TagName);
    JWRITE_DEF(result, "has_in_object", IsInObjectTags, true);
    JWRITE_ENUM_DEF(result, "hidden", Hidden, EHidden::Any);
    JWRITE_ENUM_DEF(result, "performed", Performed, EPerformed::Any);
    return result;
}

TString TTagsFilter::TMatchToken::ToString() const {
    TStringStream ss;
    if (!IsInObjectTags) {
        ss << "-";
    }
    if (Performed == EPerformed::Perform) {
        ss << "'";
    } else if (Performed == EPerformed::NoPerform) {
        ss << "#";
    }
    if (Hidden == EHidden::Yes) {
        ss << "&";
    } else if (Hidden == EHidden::No) {
        ss << "?";
    }
    ss << TagName;
    return ss.Str();
}

TTagsFilter::TMatchToken::TMatchToken(TStringBuf matchTokenStr) {
    TStringBuf token = matchTokenStr;
    while (token.EndsWith(" ")) {
        token.Chop(1);
    }
    while (token.StartsWith(" ")) {
        token.Skip(1);
    }
    while (token.size()) {
        if (token.StartsWith("&")) {
            TagName = "";
            Hidden = EHidden::Yes;
        } else if (token.StartsWith("?")) {
            TagName = "";
            Hidden = EHidden::No;
        } else if (token.StartsWith("+")) {
            TagName = "";
            IsInObjectTags = true;
        } else if (token.StartsWith("-")) {
            TagName = "";
            IsInObjectTags = false;
        } else if (token.StartsWith("#")) {
            TagName = "";
            Performed = EPerformed::NoPerform;
        } else if (token.StartsWith("'")) {
            TagName = "";
            Performed = EPerformed::Perform;
        } else {
            if (!TagName) {
                TagName = token;
            }
            if (!token.StartsWith(" ")) {
                break;
            }
        }
        token.Skip(1);
    }
}

template <>
NJson::TJsonValue NJson::ToJson(const TTagsFilter& object) {
    return object.ToString();
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTagsFilter& result) {
    TString str;
    return TryFromJson(value, str) && result.DeserializeFromString(str);
}
