#pragma once

#include "tags.h"

#include <util/generic/string.h>
#include <util/string/vector.h>

template <class TTagType>
class TTagDataSelector {
public:
    static i32 GetPriority(const TTagType& /*tag*/) {
        return 0;
    }

    static const TString& GetPerformer(const TTagType& /*tag*/) {
        return Default<TString>();
    }

    static const TString& GetName(const TTagType& tag) {
        return tag;
    }
};

template<>
class TTagDataSelector<TDBTag> {
public:
    static i32 GetPriority(const TDBTag& tag) {
        return tag->GetTagPriority(0);
    }

    static const TString& GetPerformer(const TDBTag& tag) {
        return tag->GetPerformer();
    }

    static const TString& GetName(const TDBTag& tag) {
        return tag->GetName();
    }
};

class TTagsFilter {
public:
    enum class EPerformed {
        Any = 0,
        NoPerform = 1,
        Perform = 2,
    };

    enum class EHidden {
        Any = 0,
        No = 1,
        Yes = 2,
    };

    template <class TTagType>
    using TPreparedTags = std::tuple<
        TVector<const TTagType*>,
        i32
    >;

private:
    struct TMatchToken {
    public:
        TStringBuf TagName;
        bool IsInObjectTags = true;
        EHidden Hidden = EHidden::Any;
        EPerformed Performed = EPerformed::Any;

    public:
        TMatchToken() = default;
        explicit TMatchToken(TStringBuf matchTokenStr);

        bool DeserializeFromJson(const NJson::TJsonValue& info);
        NJson::TJsonValue SerializeToJson() const;

        TString ToString() const;

        template <class TTagType = TDBTag>
        bool MatchDeep(const TTagType& item, const i64 maxPriority) const {
            if (Performed == EPerformed::NoPerform) {
                if (TTagDataSelector<TTagType>::GetPerformer(item)) {
                    return false;
                }
            } else if (Performed == EPerformed::Perform) {
                if (!TTagDataSelector<TTagType>::GetPerformer(item)) {
                    return false;
                }
            }
            if (Hidden == EHidden::Yes) {
                if (TTagDataSelector<TTagType>::GetPriority(item) >= maxPriority) {
                    return false;
                }
            } else if (Hidden == EHidden::No) {
                if (TTagDataSelector<TTagType>::GetPriority(item) < maxPriority) {
                    return false;
                }
            }
            return true;
        }

        template <class TTagType = TDBTag>
        int Match(const TTagType& item) const {
            if (TagName < TTagDataSelector<TTagType>::GetName(item)) {
                return -1;
            }
            if (TagName == TTagDataSelector<TTagType>::GetName(item)) {
                return 0;
            } else {
                return 1;
            }
        }

        auto Tuple() const {
            return std::tie(TagName, Performed, IsInObjectTags, Hidden);
        }
        bool operator<(const TMatchToken& item) const {
            return Tuple() < item.Tuple();
        }
        bool operator==(const TMatchToken& item) const {
            return Tuple() == item.Tuple();
        }
    };

    class TMatchCondition {
    public:
        R_READONLY(ui32, Deep, 0);

    private:
        TVector<TMatchToken> MatchTokens;

    public:
        TMatchCondition() = default;

        const TVector<TMatchToken>& GetMatchTokens() const {
            return MatchTokens;
        }

        TSet<TString> GetMonoTagNames() const;
        static bool SplitWithBraces(TStringBuf buf, TVector<TStringBuf>& result, const char splitChar, ui32* deep);

        void SortTokens();

        NJson::TJsonValue SerializeToJson() const;
        bool DeserializeFromJson(const NJson::TJsonValue& info);

        bool ParseFromString(TStringBuf rawMatchCondition);
        TString ToString() const;

        bool IsInclusive() const;

        template <class TTagType = TDBTag>
        bool IsMatching(TConstArrayRef<const TTagType*> tags, const i64 maxPriority) const {
            auto itConditionTags = MatchTokens.begin();
            auto itObjectTags = tags.begin();

            while (itConditionTags != MatchTokens.end() && itObjectTags != tags.end()) {
                int matchResult = itConditionTags->Match(**itObjectTags);
                if (matchResult == 1) {
                    ++itObjectTags;
                } else if (matchResult == 0) {
                    bool found = false;
                    while (matchResult == 0) {
                        if (itConditionTags->MatchDeep(**itObjectTags, maxPriority)) {
                            found = true;
                        }
                        if (++itObjectTags == tags.end()) {
                            break;
                        }
                        matchResult = itConditionTags->Match(**itObjectTags);
                    }
                    if (!itConditionTags->IsInObjectTags) {
                        if (found) {
                            return false;
                        }
                    } else {
                        if (!found) {
                            return false;
                        }
                    }
                    ++itConditionTags;
                } else if (matchResult == -1) {
                    if (itConditionTags->IsInObjectTags) {
                        return false;
                    }
                    ++itConditionTags;
                } else {
                    ythrow yexception() << "incorrect case for matching: " << matchResult << Endl;
                }
            }
            while (itConditionTags != MatchTokens.end()) {
                if (itConditionTags->IsInObjectTags) {
                    return false;
                }
                ++itConditionTags;
            }
            return true;
        }

        static bool BuildConditions(TStringBuf s, TVector<TMatchCondition>& result);

        bool Linearize(TVector<TMatchCondition>& result) const;
    };

public:
    template <class TTags, class TTagType = typename TTags::value_type>
    static auto PrepareMatching(const TTags& tags) {
        i32 priorityMax = -Max<i32>();
        TVector<const TTagType*> objectTags;
        objectTags.reserve(tags.size());
        for (auto&& tag : tags) {
            objectTags.emplace_back(&tag);
            priorityMax = Max(priorityMax, TTagDataSelector<TTagType>::GetPriority(tag));
        }
        const auto predSort = [](const TTagType* l, const TTagType* r)->bool {
            if (TTagDataSelector<TTagType>::GetName(*l) == TTagDataSelector<TTagType>::GetName(*r)) {
                if (!!TTagDataSelector<TTagType>::GetPerformer(*r) != !!TTagDataSelector<TTagType>::GetPerformer(*l)) {
                    return !!TTagDataSelector<TTagType>::GetPerformer(*r);
                } else {
                    return TTagDataSelector<TTagType>::GetPriority(*l) < TTagDataSelector<TTagType>::GetPriority(*r);
                }
            } else {
                return TTagDataSelector<TTagType>::GetName(*l) < TTagDataSelector<TTagType>::GetName(*r);
            }
        };
        std::sort(objectTags.begin(), objectTags.end(), predSort);
        return std::make_tuple(std::move(objectTags), priorityMax);
    }

public:
    TTagsFilter() = default;
    TTagsFilter(const TSet<TString>& tags, bool withPriority = false, bool mustHave = true);

    explicit operator bool() const {
        return !IsEmpty();
    }

    void Merge(const TTagsFilter& filter) {
        for (auto&& i : filter.MatchConditions) {
            MatchConditions.emplace_back(i);
        }
    }

    static TTagsFilter BuildFromString(const TString& filterDescription);
    bool DeserializeFromString(const TString& conditionsStr);
    TString ToString() const;
    TString ToLinearString() const;

    NJson::TJsonValue GetDebugInfo() const;
    TSet<TString> GetMonoTagNames() const;
    TSet<TString> GetRelatedTagNames() const;

    bool IsInclusive() const;
    bool IsMatching(const TTaggedObject& object, TString* resultFilter = nullptr) const;

    template <class TTags, class TTagType = typename TTags::value_type>
    bool IsMatching(const TTags& tags, TString* resultFilter = nullptr) const {
        auto prepared = PrepareMatching(tags);
        return IsMatching(prepared, resultFilter);
    }

    template <class TTagType>
    bool IsMatching(const TPreparedTags<TTagType>& prepared, TString* resultFilter = nullptr) const {
        const auto& objectTags = std::get<0>(prepared);
        const auto priorityMax = std::get<1>(prepared);
        for (auto&& matchCondition : MatchConditions) {
            if (matchCondition.template IsMatching<TTagType>(objectTags, priorityMax)) {
                if (resultFilter) {
                    *resultFilter = matchCondition.ToString();
                }
                return true;
            }
        }
        return false;
    }

    NJson::TJsonValue SerializeToJson() const;
    bool DeserializeFromJson(const NJson::TJsonValue& jsonValue);

    bool IsEmpty() const {
        return MatchConditions.empty();
    }

    const TVector<TMatchCondition>& GetMatchConditions() const {
        return MatchConditions;
    }

private:
    TVector<TMatchCondition> MatchConditions;
    TAtomicSharedPtr<TString> OriginalString;
};

using TTagsFilters = TVector<TTagsFilter>;
